/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.llap.cli.service;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.conf.HiveConfUtil;
import org.apache.hadoop.hive.llap.LlapUtil;
import org.apache.hadoop.hive.llap.cli.LlapSliderUtils;
import org.apache.hadoop.hive.llap.cli.service.LlapConfigJsonCreator;
import org.apache.hadoop.hive.llap.cli.service.LlapServiceCommandLine;
import org.apache.hadoop.hive.llap.cli.service.LlapTarComponentGatherer;
import org.apache.hadoop.hive.llap.configuration.LlapDaemonConfiguration;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.service.client.ServiceClient;
import org.apache.hadoop.yarn.service.utils.CoreFileSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LlapServiceDriver {
    private static final Logger LOG = LoggerFactory.getLogger((String)LlapServiceDriver.class.getName());
    private static final String LLAP_RELATIVE_PACKAGE_DIR = "/package/LLAP/";
    private static final String OUTPUT_DIR_PREFIX = "llap-yarn-";
    private final HiveConf conf;
    private final LlapServiceCommandLine cl;

    public LlapServiceDriver(LlapServiceCommandLine cl) throws Exception {
        this.cl = cl;
        SessionState ss = SessionState.get();
        this.conf = ss != null ? ss.getConf() : new HiveConf(SessionState.class);
        HiveConfUtil.copyFromProperties((Properties)cl.getConfig(), (HiveConf)this.conf);
        if (this.conf == null) {
            throw new Exception("Cannot load any configuration to run command");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int run() throws Exception {
        Properties propsDirectOptions = new Properties();
        Path tmpDir = new Path(this.cl.getDirectory());
        long t0 = System.nanoTime();
        FileSystem fs = FileSystem.get((Configuration)this.conf);
        FileSystem rawFs = FileSystem.getLocal((Configuration)this.conf).getRawFileSystem();
        int threadCount = Math.max(1, Runtime.getRuntime().availableProcessors() / 2);
        ExecutorService executor = Executors.newFixedThreadPool(threadCount, new ThreadFactoryBuilder().setNameFormat("llap-pkg-%d").build());
        int rc = 0;
        try {
            this.setupConf(propsDirectOptions);
            URL logger = this.conf.getResource("llap-daemon-log4j2.properties");
            if (logger == null) {
                throw new Exception("Unable to find required config file: llap-daemon-log4j2.properties");
            }
            Path home = new Path(System.getenv("HIVE_HOME"));
            Path scriptParent = new Path(new Path(home, "scripts"), "llap");
            Path scripts = new Path(scriptParent, "bin");
            if (!rawFs.exists(home)) {
                throw new Exception("Unable to find HIVE_HOME:" + home);
            }
            if (!rawFs.exists(scripts)) {
                LOG.warn("Unable to find llap scripts:" + scripts);
            }
            String javaHome = this.getJavaHome();
            LlapTarComponentGatherer tarComponentGatherer = new LlapTarComponentGatherer(this.cl, this.conf, propsDirectOptions, fs, rawFs, executor, tmpDir);
            tarComponentGatherer.createDirs();
            tarComponentGatherer.submitTarComponentGatherTasks();
            LlapConfigJsonCreator lcjCreator = new LlapConfigJsonCreator(this.conf, rawFs, tmpDir, this.cl.getCache(), this.cl.getXmx(), javaHome);
            lcjCreator.createLlapConfigJson();
            LOG.debug("Config Json generation took " + (System.nanoTime() - t0) + " ns");
            tarComponentGatherer.waitForFinish();
            rc = this.cl.isStarting() ? this.startLlap(tmpDir, scriptParent) : 0;
        }
        finally {
            executor.shutdown();
            rawFs.close();
            fs.close();
        }
        if (rc == 0) {
            LOG.debug("Exiting successfully");
        } else {
            LOG.info("Exiting with rc = " + rc);
        }
        return rc;
    }

    private void setupConf(Properties propsDirectOptions) throws Exception {
        long containerSize;
        long xmx;
        long cache;
        for (String f : LlapDaemonConfiguration.DAEMON_CONFIGS) {
            this.conf.addResource(f);
            if (this.conf.getResource(f) != null) continue;
            throw new Exception("Unable to find required config file: " + f);
        }
        for (String f : LlapDaemonConfiguration.SSL_DAEMON_CONFIGS) {
            this.conf.addResource(f);
        }
        this.conf.reloadConfiguration();
        LlapServiceDriver.populateConfWithLlapProperties((Configuration)this.conf, this.cl.getConfig());
        if (this.cl.getName() != null) {
            this.conf.set(HiveConf.ConfVars.LLAP_DAEMON_SERVICE_HOSTS.varname, "@" + this.cl.getName());
            propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_DAEMON_SERVICE_HOSTS.varname, "@" + this.cl.getName());
        }
        if (this.cl.getLogger() != null) {
            HiveConf.setVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_DAEMON_LOGGER, (String)this.cl.getLogger());
            propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_DAEMON_LOGGER.varname, this.cl.getLogger());
        }
        boolean isDirect = HiveConf.getBoolVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_DIRECT);
        String cacheStr = LlapUtil.humanReadableByteCount((long)this.cl.getCache());
        String sizeStr = LlapUtil.humanReadableByteCount((long)this.cl.getSize());
        String xmxStr = LlapUtil.humanReadableByteCount((long)this.cl.getXmx());
        if (this.cl.getSize() != -1L) {
            if (this.cl.getCache() != -1L) {
                if (!HiveConf.getBoolVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAPPED)) {
                    Preconditions.checkArgument((this.cl.getCache() < this.cl.getSize() ? 1 : 0) != 0, (Object)("Cache size (" + cacheStr + ") has to be smaller than the container sizing (" + sizeStr + ")"));
                } else if (this.cl.getCache() < this.cl.getSize()) {
                    LOG.warn("Note that this might need YARN physical memory monitoring to be turned off (yarn.nodemanager.pmem-check-enabled=false)");
                }
            }
            if (this.cl.getXmx() != -1L) {
                Preconditions.checkArgument((this.cl.getXmx() < this.cl.getSize() ? 1 : 0) != 0, (Object)("Working memory (Xmx=" + xmxStr + ") has to be smaller than the container sizing (" + sizeStr + ")"));
            }
            if (isDirect && !HiveConf.getBoolVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAPPED)) {
                Preconditions.checkArgument((this.cl.getXmx() + this.cl.getCache() <= this.cl.getSize() ? 1 : 0) != 0, (Object)("Working memory (Xmx=" + xmxStr + ") + cache size (" + cacheStr + ") has to be smaller than the container sizing (" + sizeStr + ")"));
            }
        }
        if (this.cl.getExecutors() != -1) {
            this.conf.setLong(HiveConf.ConfVars.LLAP_DAEMON_NUM_EXECUTORS.varname, (long)this.cl.getExecutors());
            propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_DAEMON_NUM_EXECUTORS.varname, String.valueOf(this.cl.getExecutors()));
        }
        if (this.cl.getIoThreads() != -1) {
            this.conf.setLong(HiveConf.ConfVars.LLAP_IO_THREADPOOL_SIZE.varname, (long)this.cl.getIoThreads());
            propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_IO_THREADPOOL_SIZE.varname, String.valueOf(this.cl.getIoThreads()));
        }
        if ((cache = this.cl.getCache()) != -1L) {
            this.conf.set(HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE.varname, Long.toString(cache));
            propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE.varname, Long.toString(cache));
        }
        if ((xmx = this.cl.getXmx()) != -1L) {
            long xmxMb = xmx / 0x100000L;
            this.conf.setLong(HiveConf.ConfVars.LLAP_DAEMON_MEMORY_PER_INSTANCE_MB.varname, xmxMb);
            propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_DAEMON_MEMORY_PER_INSTANCE_MB.varname, String.valueOf(xmxMb));
        }
        if ((containerSize = this.cl.getSize()) == -1L) {
            long heapSize = xmx;
            if (!isDirect) {
                heapSize += cache;
            }
            containerSize = Math.min((long)((double)heapSize * 1.2), heapSize + 0x40000000L);
            if (isDirect) {
                containerSize += cache;
            }
        }
        long containerSizeMB = containerSize / 0x100000L;
        long minAllocMB = this.conf.getInt("yarn.scheduler.minimum-allocation-mb", -1);
        String containerSizeStr = LlapUtil.humanReadableByteCount((long)containerSize);
        Preconditions.checkArgument((containerSizeMB >= minAllocMB ? 1 : 0) != 0, (Object)("Container size (" + containerSizeStr + ") should be greater than minimum allocation(" + LlapUtil.humanReadableByteCount((long)(minAllocMB * 1024L * 1024L)) + ")"));
        this.conf.setLong(HiveConf.ConfVars.LLAP_DAEMON_YARN_CONTAINER_MB.varname, containerSizeMB);
        propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_DAEMON_YARN_CONTAINER_MB.varname, String.valueOf(containerSizeMB));
        LOG.info("Memory settings: container memory: {} executor memory: {} cache memory: {}", new Object[]{containerSizeStr, xmxStr, cacheStr});
        if (!StringUtils.isEmpty((CharSequence)this.cl.getLlapQueueName())) {
            this.conf.set(HiveConf.ConfVars.LLAP_DAEMON_QUEUE_NAME.varname, this.cl.getLlapQueueName());
            propsDirectOptions.setProperty(HiveConf.ConfVars.LLAP_DAEMON_QUEUE_NAME.varname, this.cl.getLlapQueueName());
        }
    }

    private String getJavaHome() {
        String javaHome = this.cl.getJavaPath();
        if (StringUtils.isEmpty((CharSequence)javaHome)) {
            javaHome = System.getenv("JAVA_HOME");
            String jreHome = System.getProperty("java.home");
            if (javaHome == null) {
                javaHome = jreHome;
            } else if (!javaHome.equals(jreHome)) {
                LOG.warn("Java versions might not match : JAVA_HOME=[{}],process jre=[{}]", (Object)javaHome, (Object)jreHome);
            }
        }
        if (StringUtils.isEmpty((CharSequence)javaHome)) {
            throw new RuntimeException("Could not determine JAVA_HOME from command line parameters, environment or system properties");
        }
        LOG.info("Using [{}] for JAVA_HOME", (Object)javaHome);
        return javaHome;
    }

    private static void populateConfWithLlapProperties(Configuration conf, Properties properties) {
        for (Map.Entry<Object, Object> props : properties.entrySet()) {
            String key = (String)props.getKey();
            if (HiveConf.getLlapDaemonConfVars().contains(key)) {
                conf.set(key, (String)props.getValue());
                continue;
            }
            if (key.startsWith("llap.") || key.startsWith("hive.llap.")) {
                LOG.warn("Adding key [{}] even though it is not in the set of known llap-server keys", (Object)key);
                conf.set(key, (String)props.getValue());
                continue;
            }
            LOG.warn("Ignoring unknown llap server parameter: [{}]", (Object)key);
        }
    }

    private int startLlap(Path tmpDir, Path scriptParent) throws IOException, InterruptedException {
        String version = System.getenv("HIVE_VERSION");
        if (StringUtils.isEmpty((CharSequence)version)) {
            version = DateTimeFormatter.BASIC_ISO_DATE.format(LocalDateTime.now());
        }
        Object outputDir = this.cl.getOutput();
        Path packageDir = null;
        if (outputDir == null) {
            outputDir = OUTPUT_DIR_PREFIX + version;
            packageDir = new Path(Paths.get(".", new String[0]).toAbsolutePath().toString(), OUTPUT_DIR_PREFIX + version);
        } else {
            packageDir = new Path((String)outputDir);
        }
        int rc = this.runPackagePy(tmpDir, scriptParent, version, (String)outputDir);
        if (rc == 0) {
            String tarballName = this.cl.getName() + "-" + version + ".tar.gz";
            this.startCluster((Configuration)this.conf, this.cl.getName(), tarballName, packageDir, this.conf.getVar(HiveConf.ConfVars.LLAP_DAEMON_QUEUE_NAME));
        }
        return rc;
    }

    private int runPackagePy(Path tmpDir, Path scriptParent, String version, String outputDir) throws IOException, InterruptedException {
        Path scriptPath = new Path(new Path(scriptParent, "yarn"), "package.py");
        ArrayList<String> scriptArgs = new ArrayList<String>(this.cl.getArgs().length + 7);
        scriptArgs.addAll(Arrays.asList("python", scriptPath.toString(), "--input", tmpDir.toString(), "--output", outputDir, "--javaChild"));
        scriptArgs.addAll(Arrays.asList(this.cl.getArgs()));
        LOG.debug("Calling package.py via: " + scriptArgs);
        ProcessBuilder builder = new ProcessBuilder(scriptArgs);
        builder.redirectError(ProcessBuilder.Redirect.INHERIT);
        builder.redirectOutput(ProcessBuilder.Redirect.INHERIT);
        builder.environment().put("HIVE_VERSION", version);
        return builder.start().waitFor();
    }

    private void startCluster(Configuration conf, String name, String packageName, Path packageDir, String queue) {
        ServiceClient sc;
        LOG.info("Starting cluster with " + name + ", " + packageName + ", " + queue + ", " + packageDir);
        try {
            sc = LlapSliderUtils.createServiceClient(conf);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        try {
            try {
                LOG.info("Executing the stop command");
                sc.actionStop(name, true);
            }
            catch (Exception ex) {
                LOG.info(ex.getLocalizedMessage());
            }
            try {
                LOG.info("Executing the destroy command");
                sc.actionDestroy(name);
            }
            catch (Exception ex) {
                LOG.info(ex.getLocalizedMessage());
            }
            LOG.info("Uploading the app tarball");
            CoreFileSystem fs = new CoreFileSystem(conf);
            String llapPackageDir = HiveConf.getVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_HDFS_PACKAGE_DIR) + LLAP_RELATIVE_PACKAGE_DIR;
            fs.createWithPermissions(new Path(llapPackageDir), FsPermission.getDirDefault());
            fs.copyLocalFileToHdfs(new File(packageDir.toString(), packageName), new Path(llapPackageDir), new FsPermission("755"));
            LOG.info("Executing the launch command");
            File yarnfile = new File(new Path(packageDir, "Yarnfile").toString());
            Long lifetime = null;
            sc.actionLaunch(yarnfile.getAbsolutePath(), name, lifetime, queue);
            LOG.debug("Started the cluster via service API");
        }
        catch (IOException | YarnException e) {
            throw new RuntimeException(e);
        }
        finally {
            try {
                sc.close();
            }
            catch (IOException e) {
                LOG.info("Failed to close service client", (Throwable)e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws Exception {
        LlapServiceCommandLine cl = new LlapServiceCommandLine(args);
        int ret = 0;
        try {
            ret = new LlapServiceDriver(cl).run();
        }
        catch (Throwable t) {
            System.err.println("Failed: " + t.getMessage());
            t.printStackTrace();
            ret = 3;
        }
        finally {
            LOG.info("LLAP service driver finished");
        }
        LOG.debug("Completed processing - exiting with " + ret);
        System.exit(ret);
    }
}

