/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.io;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.stats.CacheStats;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import org.apache.iceberg.exceptions.NotFoundException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.io.ByteBufferInputStream;
import org.apache.iceberg.io.IOUtil;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.io.SeekableInputStream;
import org.apache.iceberg.relocated.com.google.common.base.MoreObjects;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContentCache {
    private static final Logger LOG = LoggerFactory.getLogger(ContentCache.class);
    private static final int BUFFER_CHUNK_SIZE = 0x400000;
    private final long expireAfterAccessMs;
    private final long maxTotalBytes;
    private final long maxContentLength;
    private final Cache<String, FileContent> cache;

    public ContentCache(long expireAfterAccessMs, long maxTotalBytes, long maxContentLength) {
        ValidationException.check(expireAfterAccessMs >= 0L, "expireAfterAccessMs is less than 0", new Object[0]);
        ValidationException.check(maxTotalBytes > 0L, "maxTotalBytes is equal or less than 0", new Object[0]);
        ValidationException.check(maxContentLength > 0L, "maxContentLength is equal or less than 0", new Object[0]);
        this.expireAfterAccessMs = expireAfterAccessMs;
        this.maxTotalBytes = maxTotalBytes;
        this.maxContentLength = maxContentLength;
        Caffeine<Object, Object> builder = Caffeine.newBuilder();
        if (expireAfterAccessMs > 0L) {
            builder = builder.expireAfterAccess(Duration.ofMillis(expireAfterAccessMs));
        }
        this.cache = builder.maximumWeight(maxTotalBytes).weigher((key, value) -> (int)Math.min(value.length, Integer.MAX_VALUE)).softValues().removalListener((location, fileContent, cause) -> LOG.debug("Evicted {} from ContentCache ({})", location, (Object)cause)).recordStats().build();
    }

    public long expireAfterAccess() {
        return this.expireAfterAccessMs;
    }

    public long maxContentLength() {
        return this.maxContentLength;
    }

    public long maxTotalBytes() {
        return this.maxTotalBytes;
    }

    public CacheStats stats() {
        return this.cache.stats();
    }

    public InputFile tryCache(InputFile input) {
        if (input.getLength() <= this.maxContentLength) {
            return new CachingInputFile(this, input);
        }
        return input;
    }

    public void invalidate(String key) {
        this.cache.invalidate(key);
    }

    @Deprecated
    public void invalidateAll() {
        this.cache.invalidateAll();
    }

    public void cleanUp() {
        this.cache.cleanUp();
    }

    public long estimatedCacheSize() {
        return this.cache.estimatedSize();
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).add("expireAfterAccessMs", this.expireAfterAccessMs).add("maxContentLength", this.maxContentLength).add("maxTotalBytes", this.maxTotalBytes).add("cacheStats", this.cache.stats()).toString();
    }

    private static FileContent download(InputFile input) {
        FileContent fileContent;
        block10: {
            SeekableInputStream stream = input.newStream();
            try {
                long fileLength;
                int bytesRead;
                ArrayList<ByteBuffer> buffers = Lists.newArrayList();
                for (long totalBytesToRead = fileLength = input.getLength(); totalBytesToRead > 0L; totalBytesToRead -= (long)bytesRead) {
                    int bytesToRead = (int)Math.min(0x400000L, totalBytesToRead);
                    byte[] buf = new byte[bytesToRead];
                    bytesRead = IOUtil.readRemaining(stream, buf, 0, bytesToRead);
                    if (bytesRead < bytesToRead) {
                        throw new IOException(String.format(Locale.ROOT, "Failed to read %d bytes: %d bytes in stream", fileLength, fileLength - totalBytesToRead));
                    }
                    buffers.add(ByteBuffer.wrap(buf));
                }
                fileContent = new FileContent(fileLength, buffers);
                if (stream == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new UncheckedIOException(ex);
                }
            }
            stream.close();
        }
        return fileContent;
    }

    private static class CachingInputFile
    implements InputFile {
        private final ContentCache contentCache;
        private final InputFile input;

        private CachingInputFile(ContentCache cache, InputFile input) {
            this.contentCache = cache;
            this.input = input;
        }

        @Override
        public long getLength() {
            FileContent buf = this.contentCache.cache.getIfPresent(this.input.location());
            if (buf != null) {
                return buf.length;
            }
            return this.input.getLength();
        }

        @Override
        public SeekableInputStream newStream() {
            try {
                return this.cachedStream();
            }
            catch (FileNotFoundException e) {
                throw new NotFoundException(e, "Failed to open file: %s", this.input.location());
            }
            catch (IOException e) {
                return this.input.newStream();
            }
        }

        @Override
        public String location() {
            return this.input.location();
        }

        @Override
        public boolean exists() {
            FileContent buf = this.contentCache.cache.getIfPresent(this.input.location());
            return buf != null || this.input.exists();
        }

        private SeekableInputStream cachedStream() throws IOException {
            try {
                FileContent content = this.contentCache.cache.get(this.input.location(), k -> ContentCache.download(this.input));
                return ByteBufferInputStream.wrap(content.buffers);
            }
            catch (UncheckedIOException ex) {
                throw ex.getCause();
            }
        }
    }

    private static class FileContent {
        private final long length;
        private final List<ByteBuffer> buffers;

        private FileContent(long length, List<ByteBuffer> buffers) {
            this.length = length;
            this.buffers = buffers;
        }
    }
}

