/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hive.ql.exec.vector.expressions.gen;

import org.apache.hadoop.hive.ql.exec.vector.VectorExpressionDescriptor;

import org.apache.hadoop.hive.serde2.io.HiveDecimalWritable;
import org.apache.hadoop.hive.serde2.typeinfo.DecimalTypeInfo;

/**
 * Generated from template Decimal64ColumnCompareDecimal64Scalar.txt, which covers decimal64
 * comparison  expressions between a column and a scalar, however output is not produced in a
 * separate column. The selected vector of the input {@link org.apache.hadoop.hive.ql.exec.vector.VectorizedRowBatch VectorizedRowBatch} is updated for
 * in-place filtering.
 */
public class Decimal64ColNotEqualDecimal64Scalar extends LongColNotEqualLongScalar {

  private static final long serialVersionUID = 1L;

  public Decimal64ColNotEqualDecimal64Scalar(int colNum, long value, int outputColumnNum) {
    super(colNum, value, outputColumnNum);
  }

  public Decimal64ColNotEqualDecimal64Scalar() {
    super();
  }

  @Override
  public String vectorExpressionParameters() {
    DecimalTypeInfo decimalTypeInfo = (DecimalTypeInfo) inputTypeInfos[0];
    HiveDecimalWritable writable = new HiveDecimalWritable();
    writable.deserialize64(value, decimalTypeInfo.scale());
    return getColumnParamString(0, inputColumnNum[0]) + ", decimal64Val " + value +
        ", decimalVal " + writable.toString();
  }

  @Override
  public VectorExpressionDescriptor.Descriptor getDescriptor() {
    return (new VectorExpressionDescriptor.Builder())
        .setMode(
            VectorExpressionDescriptor.Mode.PROJECTION)
        .setNumArguments(2)
        .setArgumentTypes(
            VectorExpressionDescriptor.ArgumentType.DECIMAL_64,
            VectorExpressionDescriptor.ArgumentType.DECIMAL_64)
        .setInputExpressionTypes(
            VectorExpressionDescriptor.InputExpressionType.COLUMN,
            VectorExpressionDescriptor.InputExpressionType.SCALAR).build();
  }

  @Override
  public boolean shouldConvertDecimal64ToDecimal() {
    return false;
  }
}
