/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hive.ql.exec.vector.expressions.gen;

import java.util.Arrays;

import org.apache.hadoop.hive.ql.exec.vector.LongColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.Decimal64ColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.VectorExpressionDescriptor;
import org.apache.hadoop.hive.ql.exec.vector.VectorizedRowBatch;
import org.apache.hadoop.hive.ql.exec.vector.expressions.NullUtil;
import org.apache.hadoop.hive.ql.exec.vector.expressions.Decimal64Util;
import org.apache.hadoop.hive.ql.exec.vector.expressions.VectorExpression;
import org.apache.hadoop.hive.serde2.typeinfo.DecimalTypeInfo;
import org.apache.hadoop.hive.serde2.io.HiveDecimalWritable;
import org.apache.hadoop.hive.ql.metadata.HiveException;

/**
 * Generated from template Decimal64ColumnArithmeticDecimal64Column.txt, which covers
 * decimal64 arithmetic expressions between columns.
 */
public class Decimal64ColDivideDecimal64Column extends VectorExpression {

  private static final long serialVersionUID = 1L;

  private static final long[] powerOfTenTable = {
    1L,                   // 0
    10L,
    100L,
    1_000L,
    10_000L,
    100_000L,
    1_000_000L,
    10_000_000L,
    100_000_000L,           // 8
    1_000_000_000L,
    10_000_000_000L,
    100_000_000_000L,
    1_000_000_000_000L,
    10_000_000_000_000L,
    100_000_000_000_000L,
    1_000_000_000_000_000L,
    10_000_000_000_000_000L,   // 16
    100_000_000_000_000_000L,
    1_000_000_000_000_000_000L, // 18
  };

  public Decimal64ColDivideDecimal64Column(int colNum1, int colNum2, int outputColumnNum) {
    super(colNum1, colNum2, outputColumnNum);
  }

  public Decimal64ColDivideDecimal64Column() {
    super();
  }

  /*
   * Division that rounds up if the subsequent digits are greater than half.
   * Sign of the result will depend on the both x and y.
   * If x and y have different sign, then the result is multiplied by -1.
   * If x and y have same sign, then the result is always positive.
   */
  private long roundingDivision(long divident, long divisor) {
    final long absDivident = Math.abs(divident);
    final long absDivisor = Math.abs(divisor);
    final boolean differentSigns = (divident > 0) ^ (divisor > 0);
    final long remainder = absDivident % absDivisor;
    long result = absDivident  / absDivisor;
    if (remainder*2 >= absDivisor) {
      result += 1; // round half away from 0
    }
    if (differentSigns) {
      return -1*result;
    } else {
      return result;
    }
  }

  @Override
  public void evaluate(VectorizedRowBatch batch) throws HiveException {

    // return immediately if batch is empty
    final int n = batch.size;
    if (n == 0) {
      return;
    }

    if (childExpressions != null) {
      super.evaluateChildren(batch);
    }

    Decimal64ColumnVector inputColVector1 = (Decimal64ColumnVector) batch.cols[inputColumnNum[0]];
    Decimal64ColumnVector inputColVector2 = (Decimal64ColumnVector) batch.cols[inputColumnNum[1]];
    Decimal64ColumnVector outputColVector = (Decimal64ColumnVector) batch.cols[outputColumnNum];
    int[] sel = batch.selected;

    long[] vector1 = inputColVector1.vector;
    long[] vector2 = inputColVector2.vector;
    long[] outputVector = outputColVector.vector;
    boolean[] outputIsNull = outputColVector.isNull;
    int outputScale = ((DecimalTypeInfo) outputTypeInfo).scale();
    long scaleFactor = powerOfTenTable[outputScale];

    final long outputDecimal64AbsMax =
        HiveDecimalWritable.getDecimal64AbsMax(outputColVector.precision);

    /*
     * Propagate null values for a two-input operator and set isRepeating and noNulls appropriately.
     */
    NullUtil.propagateNullsColCol(
        inputColVector1, inputColVector2, outputColVector, sel, n, batch.selectedInUse);

    /*
     * Disregard nulls for processing. In other words,
     * the arithmetic operation is performed even if one or
     * more inputs are null. This is to improve speed by avoiding
     * conditional checks in the inner loop.
     */
    if (inputColVector1.isRepeating && inputColVector2.isRepeating) {
      if(vector2[0] == 0) {   
        outputColVector.noNulls = false;
        outputColVector.isRepeating = true;
        Arrays.fill(outputIsNull, true);
        NullUtil.setNullOutputEntriesColScalar(outputColVector, batch.selectedInUse, sel, n);
        return;
      }
      final long result = roundingDivision(vector1[0] * scaleFactor, vector2[0]);
      outputVector[0] = result;
      if (Math.abs(result) > outputDecimal64AbsMax) {
        outputColVector.noNulls = false;
        outputIsNull[0] = true;
      }
    } else if (inputColVector1.isRepeating) {
      final long repeatedValue1 = vector1[0];
      if (batch.selectedInUse) {
        for(int j = 0; j != n; j++) {
          int i = sel[j];
          if(vector2[i] == 0) {
            outputColVector.noNulls = false;
            outputIsNull[i] = true;
            outputVector[i] = LongColumnVector.NULL_VALUE;
          } else {
            final long result = roundingDivision(repeatedValue1 * scaleFactor, vector2[i]);
            outputVector[i] = result;
            if (Math.abs(result) > outputDecimal64AbsMax) {
              outputColVector.noNulls = false;
              outputIsNull[i] = true;
            }
          }
        }
      } else {
        for(int i = 0; i != n; i++) {
          if(vector2[i] == 0) {
            outputColVector.noNulls = false;
            outputIsNull[i] = true;
            outputVector[i] = LongColumnVector.NULL_VALUE;
          } else {
            final long result = roundingDivision(repeatedValue1 * scaleFactor, vector2[i]);
            outputVector[i] = result;
            if (Math.abs(result) > outputDecimal64AbsMax) {
              outputColVector.noNulls = false;
              outputIsNull[i] = true;
            }
          }
        }
      }
    } else if (inputColVector2.isRepeating) {
      final long repeatedValue2 = vector2[0];
      if(repeatedValue2 == 0) {
        outputColVector.noNulls = false;
        outputColVector.isRepeating = true;
        Arrays.fill(outputIsNull, true);
        NullUtil.setNullOutputEntriesColScalar(outputColVector, batch.selectedInUse, sel, n);
        return;
      }
      if (batch.selectedInUse) {
        for(int j = 0; j != n; j++) {
          int i = sel[j];
          final long result = roundingDivision(vector1[i] * scaleFactor, repeatedValue2);
          outputVector[i] = result;
          if (Math.abs(result) > outputDecimal64AbsMax) {
            outputColVector.noNulls = false;
            outputIsNull[i] = true;
          }
        }
      } else {
        for(int i = 0; i != n; i++) {
          final long result = roundingDivision(vector1[i] * scaleFactor, repeatedValue2);
          outputVector[i] = result;
          if (Math.abs(result) > outputDecimal64AbsMax) {
            outputColVector.noNulls = false;
            outputIsNull[i] = true;
          }
        }
      }
    } else {
      if (batch.selectedInUse) {
        for(int j = 0; j != n; j++) {
          int i = sel[j];
          if(vector2[i] == 0) {
            outputColVector.noNulls = false;
            outputIsNull[i] = true;
            outputVector[i] = LongColumnVector.NULL_VALUE;
          } else {
            final long result = roundingDivision(vector1[i] * scaleFactor, vector2[i]);
            outputVector[i] = result;
            if (Math.abs(result) > outputDecimal64AbsMax) {
              outputColVector.noNulls = false;
              outputIsNull[i] = true;
            }
          }
        }
      } else {
        for(int i = 0; i != n; i++) {
          if(vector2[i] == 0) {
            outputColVector.noNulls = false;
            outputIsNull[i] = true;
            outputVector[i] = LongColumnVector.NULL_VALUE;
          } else {
            final long result = roundingDivision(vector1[i] * scaleFactor, vector2[i]);
            outputVector[i] = result;
            if (Math.abs(result) > outputDecimal64AbsMax) {
              outputColVector.noNulls = false;
              outputIsNull[i] = true;
            }
          }
        }
      }
    }

    // Currently, we defer division, etc to regular HiveDecimal so we don't do any null
    // default value setting here.
  }

  @Override
  public String vectorExpressionParameters() {
    return getColumnParamString(0, inputColumnNum[0]) + ", " + getColumnParamString(1, inputColumnNum[1]);
  }

  @Override
  public VectorExpressionDescriptor.Descriptor getDescriptor() {
    return (new VectorExpressionDescriptor.Builder())
        .setMode(
            VectorExpressionDescriptor.Mode.PROJECTION)
        .setNumArguments(2)
        .setArgumentTypes(
            VectorExpressionDescriptor.ArgumentType.DECIMAL_64,
            VectorExpressionDescriptor.ArgumentType.DECIMAL_64)
        .setInputExpressionTypes(
            VectorExpressionDescriptor.InputExpressionType.COLUMN,
            VectorExpressionDescriptor.InputExpressionType.COLUMN).build();
  }

  @Override
  public boolean shouldConvertDecimal64ToDecimal() {
    return false;
  }
}
