/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hbase.thirdparty.com.google.protobuf.util;

import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.google.errorprone.annotations.CompileTimeConstant;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;
import javax.annotation.Nullable;
import org.apache.hbase.thirdparty.com.google.common.math.IntMath;
import org.apache.hbase.thirdparty.com.google.common.math.LongMath;
import org.apache.hbase.thirdparty.com.google.protobuf.Duration;
import org.apache.hbase.thirdparty.com.google.protobuf.Timestamp;
import org.apache.hbase.thirdparty.com.google.protobuf.util.Durations;

public final class Timestamps {
    static final long TIMESTAMP_SECONDS_MIN = -62135596800L;
    static final long TIMESTAMP_SECONDS_MAX = 253402300799L;
    static final int NANOS_PER_SECOND = 1000000000;
    static final int NANOS_PER_MILLISECOND = 1000000;
    static final int NANOS_PER_MICROSECOND = 1000;
    static final int MILLIS_PER_SECOND = 1000;
    static final int MICROS_PER_SECOND = 1000000;
    public static final Timestamp MIN_VALUE = Timestamp.newBuilder().setSeconds(-62135596800L).setNanos(0).build();
    public static final Timestamp MAX_VALUE = Timestamp.newBuilder().setSeconds(253402300799L).setNanos(999999999).build();
    public static final Timestamp EPOCH = Timestamp.newBuilder().setSeconds(0L).setNanos(0).build();
    private static final ThreadLocal<SimpleDateFormat> timestampFormat = new ThreadLocal<SimpleDateFormat>(){

        @Override
        protected SimpleDateFormat initialValue() {
            return Timestamps.createTimestampFormat();
        }
    };
    @Nullable
    private static final Method INSTANT_NOW = Timestamps.instantMethod("now");
    @Nullable
    private static final Method INSTANT_GET_EPOCH_SECOND = Timestamps.instantMethod("getEpochSecond");
    @Nullable
    private static final Method INSTANT_GET_NANO = Timestamps.instantMethod("getNano");

    private static SimpleDateFormat createTimestampFormat() {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.ENGLISH);
        GregorianCalendar calendar = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
        calendar.setGregorianChange(new Date(Long.MIN_VALUE));
        sdf.setCalendar(calendar);
        return sdf;
    }

    private Timestamps() {
    }

    public static Comparator<Timestamp> comparator() {
        return TimestampComparator.INSTANCE;
    }

    public static int compare(Timestamp x, Timestamp y) {
        return TimestampComparator.INSTANCE.compare(x, y);
    }

    public static boolean isValid(Timestamp timestamp) {
        return Timestamps.isValid(timestamp.getSeconds(), timestamp.getNanos());
    }

    public static boolean isValid(long seconds, int nanos) {
        if (seconds < -62135596800L || seconds > 253402300799L) {
            return false;
        }
        return nanos >= 0 && nanos < 1000000000;
    }

    @CanIgnoreReturnValue
    public static Timestamp checkValid(Timestamp timestamp) {
        int nanos;
        long seconds = timestamp.getSeconds();
        if (!Timestamps.isValid(seconds, nanos = timestamp.getNanos())) {
            throw new IllegalArgumentException(String.format("Timestamp is not valid. See proto definition for valid values. Seconds (%s) must be in range [-62,135,596,800, +253,402,300,799]. Nanos (%s) must be in range [0, +999,999,999].", seconds, nanos));
        }
        return timestamp;
    }

    public static Timestamp checkValid(Timestamp.Builder timestampBuilder) {
        return Timestamps.checkValid(timestampBuilder.build());
    }

    public static String toString(Timestamp timestamp) {
        Timestamps.checkValid(timestamp);
        long seconds = timestamp.getSeconds();
        int nanos = timestamp.getNanos();
        StringBuilder result = new StringBuilder();
        Date date = new Date(seconds * 1000L);
        result.append(timestampFormat.get().format(date));
        if (nanos != 0) {
            result.append(".");
            result.append(Timestamps.formatNanos(nanos));
        }
        result.append("Z");
        return result.toString();
    }

    public static Timestamp parse(String value) throws ParseException {
        int nanos;
        String timeValue;
        int dayOffset = value.indexOf(84);
        if (dayOffset == -1) {
            throw new ParseException("Failed to parse timestamp: invalid timestamp \"" + value + "\"", 0);
        }
        int timezoneOffsetPosition = value.indexOf(90, dayOffset);
        if (timezoneOffsetPosition == -1) {
            timezoneOffsetPosition = value.indexOf(43, dayOffset);
        }
        if (timezoneOffsetPosition == -1) {
            timezoneOffsetPosition = value.indexOf(45, dayOffset);
        }
        if (timezoneOffsetPosition == -1) {
            throw new ParseException("Failed to parse timestamp: missing valid timezone offset.", 0);
        }
        String secondValue = timeValue = value.substring(0, timezoneOffsetPosition);
        String nanoValue = "";
        int pointPosition = timeValue.indexOf(46);
        if (pointPosition != -1) {
            secondValue = timeValue.substring(0, pointPosition);
            nanoValue = timeValue.substring(pointPosition + 1);
        }
        Date date = timestampFormat.get().parse(secondValue);
        long seconds = date.getTime() / 1000L;
        int n = nanos = nanoValue.isEmpty() ? 0 : Timestamps.parseNanos(nanoValue);
        if (value.charAt(timezoneOffsetPosition) == 'Z') {
            if (value.length() != timezoneOffsetPosition + 1) {
                throw new ParseException("Failed to parse timestamp: invalid trailing data \"" + value.substring(timezoneOffsetPosition) + "\"", 0);
            }
        } else {
            String offsetValue = value.substring(timezoneOffsetPosition + 1);
            long offset = Timestamps.parseTimezoneOffset(offsetValue);
            seconds = value.charAt(timezoneOffsetPosition) == '+' ? (seconds -= offset) : (seconds += offset);
        }
        try {
            return Timestamps.normalizedTimestamp(seconds, nanos);
        }
        catch (IllegalArgumentException e) {
            ParseException ex = new ParseException("Failed to parse timestamp " + value + " Timestamp is out of range.", 0);
            ex.initCause(e);
            throw ex;
        }
    }

    public static Timestamp parseUnchecked(@CompileTimeConstant String value) {
        try {
            return Timestamps.parse(value);
        }
        catch (ParseException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Nullable
    private static Method instantMethod(String methodName) {
        try {
            return Class.forName("java.time.Instant").getMethod(methodName, new Class[0]);
        }
        catch (Exception e) {
            return null;
        }
    }

    public static Timestamp now() {
        if (INSTANT_NOW != null) {
            try {
                Object now = INSTANT_NOW.invoke(null, new Object[0]);
                long epochSecond = (Long)INSTANT_GET_EPOCH_SECOND.invoke(now, new Object[0]);
                int nanoAdjustment = (Integer)INSTANT_GET_NANO.invoke(now, new Object[0]);
                return Timestamps.normalizedTimestamp(epochSecond, nanoAdjustment);
            }
            catch (Throwable fallThrough) {
                throw new AssertionError((Object)fallThrough);
            }
        }
        return Timestamps.fromMillis(System.currentTimeMillis());
    }

    public static Timestamp fromSeconds(long seconds) {
        return Timestamps.normalizedTimestamp(seconds, 0);
    }

    public static long toSeconds(Timestamp timestamp) {
        return Timestamps.checkValid(timestamp).getSeconds();
    }

    public static Timestamp fromMillis(long milliseconds) {
        return Timestamps.normalizedTimestamp(milliseconds / 1000L, (int)(milliseconds % 1000L * 1000000L));
    }

    public static Timestamp fromDate(Date date) {
        if (date instanceof java.sql.Timestamp) {
            java.sql.Timestamp sqlTimestamp = (java.sql.Timestamp)date;
            long time = sqlTimestamp.getTime();
            long integralSeconds = time < 0L && time % 1000L != 0L ? time / 1000L - 1L : time / 1000L;
            return Timestamp.newBuilder().setSeconds(integralSeconds).setNanos(sqlTimestamp.getNanos()).build();
        }
        return Timestamps.fromMillis(date.getTime());
    }

    public static long toMillis(Timestamp timestamp) {
        Timestamps.checkValid(timestamp);
        return LongMath.checkedAdd(LongMath.checkedMultiply(timestamp.getSeconds(), 1000L), timestamp.getNanos() / 1000000);
    }

    public static Timestamp fromMicros(long microseconds) {
        return Timestamps.normalizedTimestamp(microseconds / 1000000L, (int)(microseconds % 1000000L * 1000L));
    }

    public static long toMicros(Timestamp timestamp) {
        Timestamps.checkValid(timestamp);
        return LongMath.checkedAdd(LongMath.checkedMultiply(timestamp.getSeconds(), 1000000L), timestamp.getNanos() / 1000);
    }

    public static Timestamp fromNanos(long nanoseconds) {
        return Timestamps.normalizedTimestamp(nanoseconds / 1000000000L, (int)(nanoseconds % 1000000000L));
    }

    public static long toNanos(Timestamp timestamp) {
        Timestamps.checkValid(timestamp);
        return LongMath.checkedAdd(LongMath.checkedMultiply(timestamp.getSeconds(), 1000000000L), timestamp.getNanos());
    }

    public static Duration between(Timestamp from, Timestamp to) {
        Timestamps.checkValid(from);
        Timestamps.checkValid(to);
        return Durations.normalizedDuration(LongMath.checkedSubtract(to.getSeconds(), from.getSeconds()), IntMath.checkedSubtract(to.getNanos(), from.getNanos()));
    }

    public static Timestamp add(Timestamp start, Duration length) {
        Timestamps.checkValid(start);
        Durations.checkValid(length);
        return Timestamps.normalizedTimestamp(LongMath.checkedAdd(start.getSeconds(), length.getSeconds()), IntMath.checkedAdd(start.getNanos(), length.getNanos()));
    }

    public static Timestamp subtract(Timestamp start, Duration length) {
        Timestamps.checkValid(start);
        Durations.checkValid(length);
        return Timestamps.normalizedTimestamp(LongMath.checkedSubtract(start.getSeconds(), length.getSeconds()), IntMath.checkedSubtract(start.getNanos(), length.getNanos()));
    }

    static Timestamp normalizedTimestamp(long seconds, int nanos) {
        if (nanos <= -1000000000 || nanos >= 1000000000) {
            seconds = LongMath.checkedAdd(seconds, nanos / 1000000000);
            nanos %= 1000000000;
        }
        if (nanos < 0) {
            nanos += 1000000000;
            seconds = LongMath.checkedSubtract(seconds, 1L);
        }
        Timestamp timestamp = Timestamp.newBuilder().setSeconds(seconds).setNanos(nanos).build();
        return Timestamps.checkValid(timestamp);
    }

    private static long parseTimezoneOffset(String value) throws ParseException {
        int pos = value.indexOf(58);
        if (pos == -1) {
            throw new ParseException("Invalid offset value: " + value, 0);
        }
        String hours = value.substring(0, pos);
        String minutes = value.substring(pos + 1);
        try {
            return (Long.parseLong(hours) * 60L + Long.parseLong(minutes)) * 60L;
        }
        catch (NumberFormatException e) {
            ParseException ex = new ParseException("Invalid offset value: " + value, 0);
            ex.initCause(e);
            throw ex;
        }
    }

    static int parseNanos(String value) throws ParseException {
        int result = 0;
        for (int i = 0; i < 9; ++i) {
            result *= 10;
            if (i >= value.length()) continue;
            if (value.charAt(i) < '0' || value.charAt(i) > '9') {
                throw new ParseException("Invalid nanoseconds.", 0);
            }
            result += value.charAt(i) - 48;
        }
        return result;
    }

    static String formatNanos(int nanos) {
        if (nanos % 1000000 == 0) {
            return String.format(Locale.ENGLISH, "%1$03d", nanos / 1000000);
        }
        if (nanos % 1000 == 0) {
            return String.format(Locale.ENGLISH, "%1$06d", nanos / 1000);
        }
        return String.format(Locale.ENGLISH, "%1$09d", nanos);
    }

    private static enum TimestampComparator implements Comparator<Timestamp>,
    Serializable
    {
        INSTANCE;


        @Override
        public int compare(Timestamp t1, Timestamp t2) {
            Timestamps.checkValid(t1);
            Timestamps.checkValid(t2);
            int secDiff = Long.compare(t1.getSeconds(), t2.getSeconds());
            return secDiff != 0 ? secDiff : Integer.compare(t1.getNanos(), t2.getNanos());
        }
    }
}

