/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.yarn.client;

import java.org.apache.hadoop.shaded.io.EOFException;
import java.org.apache.hadoop.shaded.net.ConnectException;
import java.org.apache.hadoop.shaded.net.InetSocketAddress;
import java.org.apache.hadoop.shaded.net.NoRouteToHostException;
import java.org.apache.hadoop.shaded.net.SocketException;
import java.org.apache.hadoop.shaded.net.UnknownHostException;
import java.security.PrivilegedAction;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience.Public;
import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability.Unstable;
import org.apache.hadoop.shaded.org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.retry.RetryPolicies;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.retry.RetryPolicy;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.retry.RetryProxy;
import org.apache.hadoop.shaded.org.apache.hadoop.ipc.RetriableException;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.net.ConnectTimeoutException;
import org.apache.hadoop.shaded.org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.shaded.org.apache.hadoop.yarn.exceptions.NMNotYetReadyException;
import org.apache.hadoop.shaded.org.apache.hadoop.yarn.ipc.YarnRPC;

import org.apache.hadoop.shaded.com.google.org.apache.hadoop.shaded.com.on.base.Preconditions;

@Public
@Unstable
public class ServerProxy {

  protected static RetryPolicy createRetryPolicy(Configuration conf,
      String maxWaitTimeStr, long defMaxWaitTime,
      String connectRetryIntervalStr, long defRetryInterval) {
    long maxWaitTime = conf.getLong(maxWaitTimeStr, defMaxWaitTime);
    long retryIntervalMS =
        conf.getLong(connectRetryIntervalStr, defRetryInterval);

    Preconditions.checkArgument((maxWaitTime == -1 || maxWaitTime > 0),
        "Invalid Configuration. " + maxWaitTimeStr + " should be either"
            + " positive value or -1.");
    Preconditions.checkArgument(retryIntervalMS > 0, "Invalid Configuration. "
        + connectRetryIntervalStr + "should be a positive value.");

    RetryPolicy retryPolicy = null;
    if (maxWaitTime == -1) {
      // wait forever.
      retryPolicy = RetryPolicies.retryForeverWithFixedSleep(retryIntervalMS,
          TimeUnit.MILLISECONDS);
    } else {
      retryPolicy =
          RetryPolicies.retryUpToMaximumTimeWithFixedSleep(maxWaitTime,
              retryIntervalMS, TimeUnit.MILLISECONDS);
    }

    Map<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap =
        new HashMap<Class<? extends Exception>, RetryPolicy>();
    exceptionToPolicyMap.put(EOFException.class, retryPolicy);
    exceptionToPolicyMap.put(ConnectException.class, retryPolicy);
    exceptionToPolicyMap.put(NoRouteToHostException.class, retryPolicy);
    exceptionToPolicyMap.put(UnknownHostException.class, retryPolicy);
    exceptionToPolicyMap.put(ConnectTimeoutException.class, retryPolicy);
    exceptionToPolicyMap.put(RetriableException.class, retryPolicy);
    exceptionToPolicyMap.put(SocketException.class, retryPolicy);
    
    /*
     * Still keeping this to cover case like newer client talking
     * to an older version of server
     */
    exceptionToPolicyMap.put(NMNotYetReadyException.class, retryPolicy);

    return RetryPolicies.retryByException(RetryPolicies.TRY_ONCE_THEN_FAIL,
      exceptionToPolicyMap);
  }

  @SuppressWarnings("unchecked")
  protected static <T> T createRetriableProxy(final Configuration conf,
      final Class<T> protocol, final UserGroupInformation user,
      final YarnRPC rpc, final InetSocketAddress serverAddress,
      RetryPolicy retryPolicy) {
    T proxy = user.doAs(new PrivilegedAction<T>() {
      @Override
      public T run() {
        return (T) rpc.getProxy(protocol, serverAddress, conf);
      }
    });
    return (T) RetryProxy.create(protocol, proxy, retryPolicy);
  }
}