/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.runtime.locks;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableMap;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import java.io.IOException;
import java.net.URI;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.TimeoutException;
import org.apache.gobblin.runtime.api.JobSpec;
import org.apache.gobblin.runtime.locks.FileBasedJobLock;
import org.apache.gobblin.runtime.locks.JobLockException;
import org.apache.gobblin.runtime.locks.JobLockFactory;
import org.apache.gobblin.util.HadoopUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedJobLockFactory
implements JobLockFactory<FileBasedJobLock> {
    public static final String FS_URI_CONFIG = "fsURI";
    public static final String LOCK_DIR_CONFIG = "lockDir";
    static final String DEFAULT_LOCK_DIR_PREFIX = "/tmp/gobblin-job-locks-";
    static final long DEFAULT_WAIT_MS = 300000L;
    private final FileSystem fs;
    private final Path lockFileDir;
    private final Logger log;
    private final boolean deleteLockDirOnClose;

    public FileBasedJobLockFactory(FileSystem fs, String lockFileDir, Optional<Logger> log) throws IOException {
        this.fs = fs;
        this.lockFileDir = new Path(lockFileDir);
        this.log = (Logger)log.or((Object)LoggerFactory.getLogger((String)(this.getClass().getName() + "-" + lockFileDir)));
        boolean bl = this.deleteLockDirOnClose = !this.fs.exists(this.lockFileDir);
        if (this.deleteLockDirOnClose) {
            FileBasedJobLockFactory.createLockDir(this.fs, this.lockFileDir);
        }
    }

    public FileBasedJobLockFactory(FileSystem fs, String lockFileDir) throws IOException {
        this(fs, lockFileDir, (Optional<Logger>)Optional.absent());
    }

    public static FileBasedJobLockFactory create(Config factoryConfig, Configuration hadoopConf, Optional<Logger> log) throws IOException {
        FileSystem fs = factoryConfig.hasPath(FS_URI_CONFIG) ? FileSystem.get((URI)URI.create(factoryConfig.getString(FS_URI_CONFIG)), (Configuration)hadoopConf) : FileBasedJobLockFactory.getDefaultFileSystem(hadoopConf);
        String lockFilesDir = factoryConfig.hasPath(LOCK_DIR_CONFIG) ? factoryConfig.getString(LOCK_DIR_CONFIG) : FileBasedJobLockFactory.getDefaultLockDir(fs, log);
        return new FileBasedJobLockFactory(fs, lockFilesDir, log);
    }

    public static FileSystem getDefaultFileSystem(Configuration hadoopConf) throws IOException {
        return FileSystem.getLocal((Configuration)hadoopConf);
    }

    public static String getDefaultLockDir(FileSystem fs, Optional<Logger> log) {
        Path dirName;
        Random rng = new Random();
        try {
            while (fs.exists(dirName = new Path(DEFAULT_LOCK_DIR_PREFIX + rng.nextLong()))) {
            }
        }
        catch (IOException | IllegalArgumentException e) {
            throw new RuntimeException("Unable to create job lock directory: " + e, e);
        }
        if (log.isPresent()) {
            ((Logger)log.get()).info("Created default job lock directory: " + dirName);
        }
        return dirName.toString();
    }

    protected static void createLockDir(FileSystem fs, Path dirName) throws IOException {
        if (!fs.mkdirs(dirName, FileBasedJobLockFactory.getDefaultDirPermissions())) {
            throw new RuntimeException("Unable to create job lock directory: " + dirName);
        }
    }

    protected static FsPermission getDefaultDirPermissions() {
        return new FsPermission(FsAction.ALL, FsAction.READ_EXECUTE, FsAction.NONE);
    }

    Path getLockFile(String jobName) {
        return new Path(this.lockFileDir, jobName + ".lock");
    }

    void lock(Path lockFile) throws JobLockException {
        this.log.debug("Creating lock: {}", (Object)lockFile);
        try {
            if (!this.fs.createNewFile(lockFile)) {
                throw new JobLockException("Failed to create lock file " + lockFile.getName());
            }
        }
        catch (IOException e) {
            throw new JobLockException(e);
        }
    }

    void unlock(Path lockFile) throws JobLockException {
        this.log.debug("Removing lock: {}", (Object)lockFile);
        if (!this.isLocked(lockFile)) {
            return;
        }
        try {
            this.fs.delete(lockFile, false);
        }
        catch (IOException e) {
            throw new JobLockException(e);
        }
    }

    boolean tryLock(Path lockFile) throws JobLockException {
        this.log.debug("Attempting lock: {}", (Object)lockFile);
        try {
            return this.fs.createNewFile(lockFile);
        }
        catch (IOException e) {
            throw new JobLockException(e);
        }
    }

    boolean isLocked(Path lockFile) throws JobLockException {
        try {
            return this.fs.exists(lockFile);
        }
        catch (IOException e) {
            throw new JobLockException(e);
        }
    }

    public static Config getConfigForProperties(Properties properties) {
        return ConfigFactory.parseMap((Map)ImmutableMap.builder().put((Object)FS_URI_CONFIG, (Object)properties.getProperty("fs.uri", "file:///")).put((Object)LOCK_DIR_CONFIG, (Object)properties.getProperty("job.lock.dir")).build());
    }

    public static FileBasedJobLockFactory createForProperties(Properties properties) throws JobLockException {
        try {
            FileSystem fs = FileSystem.get((URI)URI.create(properties.getProperty("fs.uri", "file:///")), (Configuration)HadoopUtils.getConfFromProperties((Properties)properties));
            String lockFileDir = properties.getProperty("job.lock.dir");
            return new FileBasedJobLockFactory(fs, lockFileDir);
        }
        catch (IOException e) {
            throw new JobLockException(e);
        }
    }

    @Override
    public FileBasedJobLock getJobLock(JobSpec jobSpec) throws TimeoutException {
        String jobName = FileBasedJobLockFactory.getJobName(jobSpec);
        return new FileBasedJobLock(jobName, this);
    }

    @VisibleForTesting
    static String getJobName(JobSpec jobSpec) {
        return jobSpec.getUri().toString().replaceAll("[/.:]", "_");
    }

    @VisibleForTesting
    FileSystem getFs() {
        return this.fs;
    }

    @VisibleForTesting
    Path getLockFileDir() {
        return this.lockFileDir;
    }

    @Override
    public void close() throws IOException {
        if (this.deleteLockDirOnClose) {
            this.log.info("Delete auto-created lock directory: {}", (Object)this.getLockFileDir());
            if (!this.fs.delete(this.getLockFileDir(), true)) {
                this.log.warn("Failed to delete lock directory: {}", (Object)this.getLockFileDir());
            }
        }
    }
}

