/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.iceberg.Utils;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.iceberg.Utils.TypeInfoToSchemaParser;
import org.apache.gobblin.metadata.IntegerBytesPair;
import org.apache.gobblin.metadata.IntegerLongPair;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.hive.serde2.typeinfo.TypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.TypeInfoUtils;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DataFiles;
import org.apache.iceberg.FileFormat;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.avro.AvroSchemaUtil;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.shaded.org.apache.avro.Schema;
import org.apache.iceberg.types.Conversions;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.TypeUtil;
import org.apache.iceberg.types.Types;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergUtils {
    private static final Logger log = LoggerFactory.getLogger(IcebergUtils.class);
    private static final String AVRO_SCHEMA_URL = "avro.schema.url";
    private static final String AVRO_SCHEMA_LITERAL = "avro.schema.literal";
    private static final String[] RESTRICTED_PROPERTIES = new String[]{"avro.schema.url", "avro.schema.literal"};

    private IcebergUtils() {
    }

    public static PartitionSpec getPartitionSpec(org.apache.iceberg.Schema tableSchema, org.apache.iceberg.Schema partitionSchema) {
        PartitionSpec.Builder builder = PartitionSpec.builderFor((org.apache.iceberg.Schema)tableSchema);
        partitionSchema.asStruct().fields().forEach(f -> builder.identity(f.name()));
        return builder.build();
    }

    public static IcebergDataAndPartitionSchema getIcebergSchema(String schema, Table table) {
        Schema icebergDataSchema = new Schema.Parser().parse(schema);
        Types.StructType dataStructType = AvroSchemaUtil.convert((Schema)icebergDataSchema).asStructType();
        ArrayList dataFields = Lists.newArrayList((Iterable)dataStructType.fields());
        Schema icebergPartitionSchema = IcebergUtils.parseSchemaFromCols(table.getPartitionKeys(), table.getDbName(), table.getTableName(), true);
        Types.StructType partitionStructType = AvroSchemaUtil.convert((Schema)icebergPartitionSchema).asStructType();
        List partitionFields = partitionStructType.fields();
        Preconditions.checkArgument((boolean)partitionFields.stream().allMatch(f -> f.type().isPrimitiveType()), (Object)"Only primitive fields are supported for partition columns");
        dataFields.addAll(partitionFields);
        Types.StructType updatedStructType = Types.StructType.of((List)dataFields);
        updatedStructType = (Types.StructType)TypeUtil.assignFreshIds((Type)updatedStructType, new AtomicInteger(0)::incrementAndGet);
        return new IcebergDataAndPartitionSchema(new org.apache.iceberg.Schema(updatedStructType.fields()), new org.apache.iceberg.Schema(partitionFields));
    }

    private static Schema parseSchemaFromCols(List<FieldSchema> cols, String namespace, String recordName, boolean mkFieldsOptional) {
        ArrayList<String> colNames = new ArrayList<String>(cols.size());
        ArrayList<TypeInfo> colsTypeInfo = new ArrayList<TypeInfo>(cols.size());
        cols.forEach(fs -> {
            colNames.add(fs.getName());
            colsTypeInfo.add(TypeInfoUtils.getTypeInfoFromTypeString((String)fs.getType()));
        });
        TypeInfoToSchemaParser parser = new TypeInfoToSchemaParser(namespace, mkFieldsOptional, Collections.emptyMap());
        return new Schema.Parser().parse(parser.parseSchemaFromFieldsTypeInfo("", recordName, colNames, colsTypeInfo).toString());
    }

    public static Map<String, String> getTableProperties(Table table) {
        Map<String, String> parameters = IcebergUtils.getRawTableProperties(table);
        for (String k : RESTRICTED_PROPERTIES) {
            parameters.remove(k);
        }
        return parameters;
    }

    private static Map<String, String> getRawTableProperties(Table table) {
        HashMap<String, String> parameters = new HashMap<String, String>();
        parameters.putAll(table.getSd().getSerdeInfo().getParameters());
        parameters.putAll(table.getSd().getParameters());
        parameters.putAll(table.getParameters());
        return parameters;
    }

    public static StructLike getPartition(final Types.StructType partitionType, final List<String> partitionValues) {
        return new StructLike(){

            public int size() {
                return partitionValues.size();
            }

            public <T> T get(int pos, Class<T> javaClass) {
                return (T)IcebergUtils.partitionValue((Types.NestedField)partitionType.fields().get(pos), (String)partitionValues.get(pos));
            }

            public <T> void set(int pos, T value) {
                throw new UnsupportedOperationException();
            }
        };
    }

    private static <T> T partitionValue(Types.NestedField partitionField, String colAsString) {
        Preconditions.checkState((boolean)partitionField.type().isPrimitiveType(), (String)"Partition column {} is not of primitive type", (Object[])new Object[]{partitionField});
        return (T)Conversions.fromPartitionString((Type)partitionField.type(), (String)colAsString);
    }

    public static Map<Integer, Long> getMapFromIntegerLongPairs(List<IntegerLongPair> list, Map<Integer, Integer> schemaIdMap) {
        if (list == null || list.size() == 0 || schemaIdMap == null) {
            return null;
        }
        try {
            return list.stream().filter(t -> schemaIdMap.containsKey(t.getKey())).collect(Collectors.toMap(t -> (Integer)schemaIdMap.get(t.getKey()), IntegerLongPair::getValue));
        }
        catch (Exception e) {
            log.warn("get exception {} when calculate metrics", (Throwable)e);
            return null;
        }
    }

    public static Map<Integer, ByteBuffer> getMapFromIntegerBytesPairs(List<IntegerBytesPair> list, Map<Integer, Integer> schemaIdMap) {
        if (list == null || list.size() == 0 || schemaIdMap == null) {
            return null;
        }
        try {
            return list.stream().filter(t -> schemaIdMap.containsKey(t.getKey())).collect(Collectors.toMap(t -> (Integer)schemaIdMap.get(t.getKey()), IntegerBytesPair::getValue));
        }
        catch (Exception e) {
            log.warn("get exception {} when calculate metrics", (Throwable)e);
            return null;
        }
    }

    public static DataFile getIcebergDataFileWithoutMetric(String file, PartitionSpec partitionSpec, StructLike partitionVal) {
        String rawPath = new Path(file).toUri().getRawPath();
        DataFiles.Builder dataFileBuilder = DataFiles.builder((PartitionSpec)partitionSpec).withPath(rawPath).withFileSizeInBytes(0L).withRecordCount(0L);
        if (partitionVal != null) {
            dataFileBuilder.withPartition(partitionVal);
        }
        return dataFileBuilder.build();
    }

    public static DataFile getIcebergDataFileWithMetric(org.apache.gobblin.metadata.DataFile file, PartitionSpec partitionSpec, StructLike partition, Configuration conf, Map<Integer, Integer> schemaIdMap) {
        Path filePath = new Path(file.getFilePath());
        DataFiles.Builder dataFileBuilder = DataFiles.builder((PartitionSpec)partitionSpec);
        try {
            dataFileBuilder.withPath(filePath.toUri().getRawPath()).withFileSizeInBytes(filePath.getFileSystem(conf).getFileStatus(filePath).getLen()).withFormat(file.getFileFormat());
        }
        catch (IOException exception) {
            throw new RuntimeIOException(exception, "Failed to get dataFile for path: %s", new Object[]{filePath});
        }
        if (partition != null) {
            dataFileBuilder.withPartition(partition);
        }
        Metrics metrics = new Metrics(file.getFileMetrics().getRecordCount(), IcebergUtils.getMapFromIntegerLongPairs(file.getFileMetrics().getColumnSizes(), schemaIdMap), IcebergUtils.getMapFromIntegerLongPairs(file.getFileMetrics().getValueCounts(), schemaIdMap), IcebergUtils.getMapFromIntegerLongPairs(file.getFileMetrics().getNullValueCounts(), schemaIdMap), IcebergUtils.getMapFromIntegerBytesPairs(file.getFileMetrics().getLowerBounds(), schemaIdMap), IcebergUtils.getMapFromIntegerBytesPairs(file.getFileMetrics().getUpperBounds(), schemaIdMap));
        return dataFileBuilder.withMetrics(metrics).build();
    }

    public static Map<Integer, Integer> getSchemaIdMap(org.apache.iceberg.Schema schemaWithOriginId, org.apache.iceberg.Schema tableSchema) {
        if (schemaWithOriginId == null || tableSchema == null) {
            return null;
        }
        HashMap<Integer, Integer> map = new HashMap<Integer, Integer>();
        Map originNameIdMap = TypeUtil.indexByName((Types.StructType)schemaWithOriginId.asStruct());
        for (Map.Entry nameIdPair : originNameIdMap.entrySet()) {
            if (tableSchema.findField((String)nameIdPair.getKey()) != null) {
                map.put((Integer)nameIdPair.getValue(), tableSchema.findField((String)nameIdPair.getKey()).fieldId());
                continue;
            }
            log.warn("Cannot find field {}, will skip the metrics for this column", nameIdPair.getKey());
        }
        return map;
    }

    public static FileFormat getIcebergFormat(State state) {
        if (state.getProp("writer.output.format").equalsIgnoreCase("AVRO")) {
            return FileFormat.AVRO;
        }
        if (state.getProp("writer.output.format").equalsIgnoreCase("ORC")) {
            return FileFormat.ORC;
        }
        throw new IllegalArgumentException("Unsupported data format: " + state.getProp("writer.output.format"));
    }

    public static class IcebergDataAndPartitionSchema {
        public org.apache.iceberg.Schema tableSchema;
        public org.apache.iceberg.Schema partitionSchema;

        IcebergDataAndPartitionSchema(org.apache.iceberg.Schema tableSchema, org.apache.iceberg.Schema partitionSchema) {
            this.tableSchema = tableSchema;
            this.partitionSchema = partitionSchema;
        }
    }
}

