/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.compaction.mapreduce;

import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.gobblin.compaction.dataset.Dataset;
import org.apache.gobblin.compaction.dataset.DatasetHelper;
import org.apache.gobblin.compaction.event.CompactionSlaEventHelper;
import org.apache.gobblin.compaction.mapreduce.MRCompactor;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.util.FileListUtils;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MRCompactorJobPropCreator {
    private static final Logger LOG = LoggerFactory.getLogger(MRCompactorJobPropCreator.class);
    protected final Dataset dataset;
    protected final FileSystem fs;
    protected final State state;
    protected final boolean inputDeduplicated;
    protected final boolean outputDeduplicated;
    protected final double lateDataThresholdForRecompact;
    protected final boolean renameSourceDirEnabled;
    protected final boolean recompactFromInputPaths;
    protected final boolean recompactFromOutputPaths;

    private MRCompactorJobPropCreator(Builder builder) {
        this.dataset = builder.dataset;
        this.fs = builder.fs;
        this.state = builder.state;
        this.lateDataThresholdForRecompact = builder.lateDataThresholdForRecompact;
        this.inputDeduplicated = this.state.getPropAsBoolean("compaction.input.deduplicated", false);
        this.outputDeduplicated = this.state.getPropAsBoolean("compaction.output.deduplicated", true);
        this.recompactFromInputPaths = this.state.getPropAsBoolean("compaction.recompact.from.input.for.late.data", false);
        this.recompactFromOutputPaths = this.state.getPropAsBoolean("compaction.recompact.from.dest.paths", false);
        this.renameSourceDirEnabled = this.state.getPropAsBoolean("compaction.rename.source.dir.enabled", false);
    }

    protected List<Dataset> createJobProps() throws IOException {
        if (Iterables.tryFind(this.dataset.inputPaths(), (Predicate)new Predicate<Path>(){

            public boolean apply(Path input) {
                try {
                    return MRCompactorJobPropCreator.this.fs.exists(input);
                }
                catch (IOException e) {
                    LOG.error(String.format("Failed to check if %s exits", input), (Throwable)e);
                    return false;
                }
            }
        }).isPresent()) {
            Optional<Dataset> datasetWithJobProps = this.createJobProps(this.dataset);
            if (datasetWithJobProps.isPresent()) {
                this.setCompactionSLATimestamp((Dataset)datasetWithJobProps.get());
                return ImmutableList.of((Object)datasetWithJobProps.get());
            }
            return ImmutableList.of();
        }
        LOG.warn("Input folders " + this.dataset.inputPaths() + " do not exist. Skipping dataset " + this.dataset);
        return ImmutableList.of();
    }

    private void setCompactionSLATimestamp(Dataset dataset) {
        if ((this.recompactFromOutputPaths || !MRCompactor.datasetAlreadyCompacted(this.fs, dataset, this.renameSourceDirEnabled)) && dataset.jobProps().contains("compaction.input.path.time")) {
            long timeInMills = dataset.jobProps().getPropAsLong("compaction.input.path.time");
            CompactionSlaEventHelper.setUpstreamTimeStamp(this.state, timeInMills + TimeUnit.MILLISECONDS.convert(1L, TimeUnit.DAYS));
        }
    }

    private boolean latePathsFound(Dataset dataset) throws IOException, FileNotFoundException {
        for (Path lateInputPath : dataset.inputLatePaths()) {
            if (!this.fs.exists(lateInputPath) || this.fs.listStatus(lateInputPath).length <= 0) continue;
            return true;
        }
        return false;
    }

    protected Optional<Dataset> createJobProps(Dataset dataset) throws IOException {
        if (this.recompactFromOutputPaths && !this.latePathsFound(dataset)) {
            LOG.info(String.format("Skipping recompaction for %s since there is no late data in %s", dataset.inputPaths(), dataset.inputLatePaths()));
            return Optional.absent();
        }
        State jobProps = new State();
        jobProps.addAll(this.state);
        jobProps.setProp("compaction.fileoutputcommitter.marksuccessfuljobs", (Object)false);
        jobProps.setProp("compaction.input.deduplicated", (Object)this.inputDeduplicated);
        jobProps.setProp("compaction.output.deduplicated", (Object)this.outputDeduplicated);
        jobProps.setProp("compaction.should.deduplicate", (Object)(!this.inputDeduplicated && this.outputDeduplicated ? 1 : 0));
        if (this.recompactFromOutputPaths || !MRCompactor.datasetAlreadyCompacted(this.fs, dataset, this.renameSourceDirEnabled)) {
            if (this.renameSourceDirEnabled) {
                Set<Path> newUnrenamedDirs = MRCompactor.getDeepestLevelUnrenamedDirsWithFileExistence(this.fs, dataset.inputPaths());
                if (this.getAllFilePathsRecursively(newUnrenamedDirs).isEmpty()) {
                    return Optional.absent();
                }
                LOG.info("[{}] has unprocessed directories for first time compaction: {}", (Object)dataset.getDatasetName(), newUnrenamedDirs);
                dataset.overwriteInputPaths(newUnrenamedDirs);
                dataset.setRenamePaths(newUnrenamedDirs);
            } else {
                this.addInputLateFilesForFirstTimeCompaction(jobProps, dataset);
            }
            LOG.info(String.format("Created MR job properties for input %s and output %s.", dataset.inputPaths(), dataset.outputPath()));
            dataset.setJobProps(jobProps);
            return Optional.of((Object)dataset);
        }
        return this.obtainDatasetWithJobProps(jobProps, dataset);
    }

    private void addInputLateFilesForFirstTimeCompaction(State jobProps, Dataset dataset) throws IOException {
        if (this.latePathsFound(dataset) && this.outputDeduplicated) {
            dataset.addAdditionalInputPaths(dataset.inputLatePaths());
            if (this.outputDeduplicated) {
                jobProps.setProp("compaction.should.deduplicate", (Object)true);
            }
        }
    }

    private Set<Path> getAllFilePathsRecursively(Set<Path> paths) throws IOException {
        HashSet allPaths = Sets.newHashSet();
        for (FileStatus fileStatus : FileListUtils.listFilesRecursively((FileSystem)this.fs, paths)) {
            allPaths.add(fileStatus.getPath());
        }
        return allPaths;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private Optional<Dataset> obtainDatasetWithJobProps(State jobProps, Dataset dataset) throws IOException {
        if (this.recompactFromInputPaths) {
            LOG.info(String.format("Will recompact for %s.", dataset.outputPath()));
            this.addInputLateFilesForFirstTimeCompaction(jobProps, dataset);
        } else {
            Set<Object> newDataFiles = new HashSet();
            if (this.renameSourceDirEnabled) {
                Set<Path> newUnrenamedDirs = MRCompactor.getDeepestLevelUnrenamedDirsWithFileExistence(this.fs, dataset.inputPaths());
                if (newUnrenamedDirs.isEmpty()) {
                    LOG.info("[{}] doesn't have unprocessed directories", (Object)dataset.getDatasetName());
                } else {
                    Set<Path> allFiles = this.getAllFilePathsRecursively(newUnrenamedDirs);
                    if (allFiles.isEmpty()) {
                        LOG.info("[{}] has unprocessed directories but all empty: {}", (Object)dataset.getDatasetName(), newUnrenamedDirs);
                    } else {
                        dataset.setRenamePaths(newUnrenamedDirs);
                        newDataFiles.addAll(allFiles);
                        LOG.info("[{}] has unprocessed directories: {}", (Object)dataset.getDatasetName(), newUnrenamedDirs);
                    }
                }
            } else {
                newDataFiles = this.getNewDataInFolder(dataset.inputPaths(), dataset.outputPath());
                Set<Path> newDataFilesInLatePath = this.getNewDataInFolder(dataset.inputLatePaths(), dataset.outputPath());
                newDataFiles.addAll(newDataFilesInLatePath);
                if (!newDataFiles.isEmpty() && !newDataFilesInLatePath.isEmpty()) {
                    dataset.addAdditionalInputPaths(dataset.inputLatePaths());
                }
            }
            if (newDataFiles.isEmpty()) {
                if (!this.isOutputLateDataExists(dataset)) return Optional.absent();
                LOG.info("{} don't have new data, but previous late data still remains, check if it requires to move", (Object)dataset.getDatasetName());
                dataset.setJobProps(jobProps);
                dataset.checkIfNeedToRecompact(new DatasetHelper(dataset, this.fs, Lists.newArrayList((Object[])new String[]{"avro"})));
                if (!dataset.needToRecompact()) return Optional.absent();
                MRCompactor.modifyDatasetStateToRecompact(dataset);
            } else {
                LOG.info(String.format("Will copy %d new data files for %s", newDataFiles.size(), dataset.outputPath()));
                jobProps.setProp("compaction.job.late.data.movement.task", (Object)true);
                jobProps.setProp("compaction.job.late.data.files", (Object)Joiner.on((String)",").join(newDataFiles));
            }
        }
        dataset.setJobProps(jobProps);
        return Optional.of((Object)dataset);
    }

    private boolean isOutputLateDataExists(Dataset dataset) throws IOException {
        if (!this.fs.exists(dataset.outputLatePath())) {
            return false;
        }
        return this.fs.listStatus(dataset.outputLatePath()).length > 0;
    }

    private Set<Path> getNewDataInFolder(Set<Path> inputFolders, Path outputFolder) throws IOException {
        HashSet paths = Sets.newHashSet();
        for (Path inputFolder : inputFolders) {
            paths.addAll(this.getNewDataInFolder(inputFolder, outputFolder));
        }
        return paths;
    }

    private Set<Path> getNewDataInFolder(Path inputFolder, Path outputFolder) throws IOException {
        HashSet newFiles = Sets.newHashSet();
        if (!this.fs.exists(inputFolder) || !this.fs.exists(outputFolder)) {
            return newFiles;
        }
        DateTime lastCompactionTime = new DateTime(MRCompactor.readCompactionTimestamp(this.fs, outputFolder));
        for (FileStatus fstat : FileListUtils.listFilesRecursively((FileSystem)this.fs, (Path)inputFolder)) {
            DateTime fileModificationTime = new DateTime(fstat.getModificationTime());
            if (!fileModificationTime.isAfter((ReadableInstant)lastCompactionTime)) continue;
            LOG.info("[" + fileModificationTime.getMillis() + "] " + fstat.getPath() + " is after " + lastCompactionTime.getMillis());
            newFiles.add(fstat.getPath());
        }
        if (!newFiles.isEmpty()) {
            LOG.info(String.format("Found %d new files within folder %s which are more recent than the previous compaction start time of %s.", newFiles.size(), inputFolder, lastCompactionTime));
        }
        return newFiles;
    }

    public Dataset createFailedJobProps(Throwable t) {
        this.dataset.setJobProps(this.state);
        this.dataset.skip(t);
        return this.dataset;
    }

    static class Builder {
        Dataset dataset;
        FileSystem fs;
        State state;
        double lateDataThresholdForRecompact;

        Builder() {
        }

        Builder withDataset(Dataset dataset) {
            this.dataset = dataset;
            return this;
        }

        Builder withFileSystem(FileSystem fs) {
            this.fs = fs;
            return this;
        }

        Builder withState(State state) {
            this.state = new State();
            this.state.addAll(state);
            return this;
        }

        Builder withLateDataThresholdForRecompact(double thresholdForRecompact) {
            this.lateDataThresholdForRecompact = thresholdForRecompact;
            return this;
        }

        MRCompactorJobPropCreator build() {
            return new MRCompactorJobPropCreator(this);
        }
    }
}

