/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.state.forst.fs.filemapping;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.state.StreamStateHandle;
import org.apache.flink.state.forst.fs.ForStFileSystemUtils;
import org.apache.flink.state.forst.fs.cache.FileBasedCache;
import org.apache.flink.state.forst.fs.filemapping.FileBackedMappingEntrySource;
import org.apache.flink.state.forst.fs.filemapping.FileOwnership;
import org.apache.flink.state.forst.fs.filemapping.FileOwnershipDecider;
import org.apache.flink.state.forst.fs.filemapping.HandleBackedMappingEntrySource;
import org.apache.flink.state.forst.fs.filemapping.MappingEntry;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileMappingManager {
    private static final Logger LOG = LoggerFactory.getLogger(FileMappingManager.class);
    private final FileSystem fileSystem;
    private final HashMap<String, MappingEntry> mappingTable;
    private final String remoteBase;
    private final String localBase;

    public FileMappingManager(FileSystem fileSystem, String remoteBase, String localBase) {
        this.fileSystem = fileSystem;
        this.mappingTable = new HashMap();
        this.remoteBase = remoteBase;
        this.localBase = localBase;
    }

    public MappingEntry createNewFile(Path filePath, boolean overwrite, FileBasedCache cache) {
        String key = filePath.toString();
        boolean isLocal = FileOwnershipDecider.shouldAlwaysBeLocal(filePath);
        if (isLocal) {
            filePath = this.forceLocalPath(filePath);
        }
        return this.addFileToMappingTable(key, this.toUUIDPath(filePath), FileOwnershipDecider.decideForNewFile(filePath), isLocal ? null : cache, true, overwrite);
    }

    public MappingEntry registerReusedRestoredFile(String key, StreamStateHandle stateHandle, Path dbFilePath, FileBasedCache cache) {
        LOG.trace("decide restored file ownership based on dbFilePath: {}", (Object)dbFilePath);
        return this.addHandleBackedFileToMappingTable(key, stateHandle, FileOwnershipDecider.decideForRestoredFile(dbFilePath), cache);
    }

    private MappingEntry addHandleBackedFileToMappingTable(String key, StreamStateHandle stateHandle, FileOwnership fileOwnership, FileBasedCache cache) {
        HandleBackedMappingEntrySource source = new HandleBackedMappingEntrySource(stateHandle);
        MappingEntry existingEntry = this.mappingTable.getOrDefault(key, null);
        if (existingEntry != null) {
            Preconditions.checkState((existingEntry.source.equals(source) && existingEntry.fileOwnership == fileOwnership ? 1 : 0) != 0, (String)"Try to add a file that is already in mappingTable, but with inconsistent entry. Key: %s, source: %s, fileOwnership: %s.  Entry in table: %s", (Object[])new Object[]{key, source, fileOwnership, existingEntry});
            LOG.trace("Skip adding a file that already exists in mapping table: {}", (Object)key);
        }
        return existingEntry == null ? this.addMappingEntry(key, new MappingEntry(1, source, fileOwnership, cache, false, false)) : existingEntry;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private MappingEntry addFileToMappingTable(String key, Path filePath, FileOwnership fileOwnership, FileBasedCache cache, boolean writing, boolean overwrite) {
        FileBackedMappingEntrySource source = new FileBackedMappingEntrySource(filePath);
        MappingEntry existingEntry = this.mappingTable.getOrDefault(key, null);
        if (existingEntry == null) return existingEntry == null ? this.addMappingEntry(key, new MappingEntry(1, source, fileOwnership, cache, false, writing)) : existingEntry;
        if (!existingEntry.source.equals(source) || existingEntry.fileOwnership != fileOwnership) {
            if (!overwrite) throw new IllegalStateException(String.format("Try to add a file that is already in mappingTable, but with inconsistent entry. Key: %s, source: %s, fileOwnership: %s.  Entry in table: %s", new Object[]{key, source, fileOwnership, existingEntry}));
            LOG.trace("Replace the mapping entry for file: {} from {} to {}", new Object[]{key, existingEntry.source, source});
            this.mappingTable.remove(key).release();
            existingEntry = null;
            return existingEntry == null ? this.addMappingEntry(key, new MappingEntry(1, source, fileOwnership, cache, false, writing)) : existingEntry;
        } else {
            LOG.trace("Skip adding a file that already exists in mapping table: {}", (Object)key);
        }
        return existingEntry == null ? this.addMappingEntry(key, new MappingEntry(1, source, fileOwnership, cache, false, writing)) : existingEntry;
    }

    private MappingEntry addMappingEntry(String key, MappingEntry entry) {
        this.mappingTable.put(key, entry);
        LOG.trace("Add entry to mapping table: {} -> {}", (Object)key, (Object)entry);
        return entry;
    }

    public int link(String src, String dst) {
        if (src.equals(dst)) {
            return -1;
        }
        if (this.mappingTable.containsKey(dst)) {
            return -1;
        }
        MappingEntry sourceEntry = this.mappingTable.get(src);
        if (sourceEntry == null) {
            throw new RuntimeException("Unexpected: linking to a file that doesn't exist in ForSt FileMappingManager.");
        }
        sourceEntry.retain();
        this.mappingTable.putIfAbsent(dst, sourceEntry);
        LOG.trace("link: {} -> {}", (Object)dst, (Object)src);
        return 0;
    }

    public List<String> listByPrefix(String path) {
        ArrayList<String> linkedPaths = new ArrayList<String>();
        for (Map.Entry<String, MappingEntry> entry : this.mappingTable.entrySet()) {
            if (!ForStFileSystemUtils.isParentDir(entry.getKey(), path)) continue;
            linkedPaths.add(entry.getKey());
        }
        return linkedPaths;
    }

    public boolean renameFile(String src, String dst) throws IOException {
        if (src.equals(dst)) {
            return true;
        }
        MappingEntry srcEntry = this.mappingTable.get(src);
        if (srcEntry != null) {
            if (this.mappingTable.containsKey(dst)) {
                MappingEntry dstEntry = this.mappingTable.remove(dst);
                dstEntry.release();
            }
            LOG.trace("rename: {} -> {}", (Object)src, (Object)dst);
            this.mappingTable.remove(src);
            this.mappingTable.put(dst, srcEntry);
        } else {
            List<String> toRename = this.listByPrefix(src);
            for (String key : toRename) {
                MappingEntry sourceEntry = this.mappingTable.get(key);
                sourceEntry.retain();
                String renamedDst = key.replace(src, dst);
                LOG.trace("rename: {} -> {}", (Object)key, (Object)renamedDst);
                this.mappingTable.put(renamedDst, sourceEntry);
            }
            Path dstPath = new Path(dst);
            if (!this.fileSystem.exists(dstPath)) {
                this.fileSystem.mkdirs(dstPath);
            }
            this.deleteFileOrDirectory(new Path(src), true);
        }
        return true;
    }

    public boolean deleteFileOrDirectory(Path file, boolean recursive) throws IOException {
        String fileStr = file.toString();
        MappingEntry entry = this.mappingTable.getOrDefault(fileStr, null);
        LOG.trace("Remove from mapping table: {}, entry:{}", (Object)fileStr, (Object)entry);
        if (entry != null) {
            this.mappingTable.remove(fileStr);
            entry.release();
            return true;
        }
        if (!recursive) {
            throw new IOException(fileStr + "is a directory, delete failed.");
        }
        MappingEntry parentEntry = new MappingEntry(0, file, FileOwnership.PRIVATE_OWNED_BY_DB, true);
        for (Map.Entry<String, MappingEntry> currentEntry : this.mappingTable.entrySet()) {
            MappingEntry oldParentDir;
            MappingEntry mappingEntry = currentEntry.getValue();
            if (!ForStFileSystemUtils.isParentDir(mappingEntry.getSourcePath(), fileStr) || (oldParentDir = mappingEntry.parentDir) != null && (!ForStFileSystemUtils.isParentDir(oldParentDir.getSourcePath(), fileStr) || oldParentDir.equals((Object)parentEntry))) continue;
            parentEntry.retain();
            mappingEntry.parentDir = parentEntry;
            if (mappingEntry.fileOwnership != FileOwnership.NOT_OWNED) continue;
            parentEntry.setFileOwnership(FileOwnership.NOT_OWNED);
        }
        parentEntry.setFileOwnership(FileOwnership.NOT_OWNED);
        boolean status = true;
        if (parentEntry.getReferenceCount() == 0 && parentEntry.getFileOwnership() != FileOwnership.NOT_OWNED) {
            status = this.fileSystem.delete(file, recursive);
        }
        List<String> toRelease = this.listByPrefix(fileStr);
        for (String key : toRelease) {
            this.mappingTable.remove(key).release();
        }
        return status;
    }

    @Nullable
    @VisibleForTesting
    public MappingEntry mappingEntry(String path) {
        return this.mappingTable.getOrDefault(path, null);
    }

    public void giveUpOwnership(Path path, StreamStateHandle stateHandle) {
        MappingEntry mappingEntry = this.mappingTable.getOrDefault(path.toString(), null);
        Preconditions.checkArgument((mappingEntry != null ? 1 : 0) != 0, (String)"Try to give up ownership of a file that is not in mapping table: %s", (Object[])new Object[]{path});
        Preconditions.checkArgument((mappingEntry.fileOwnership != FileOwnership.PRIVATE_OWNED_BY_DB ? 1 : 0) != 0, (String)"Try to give up ownership of a file that is not shareable: %s ", (Object[])new Object[]{mappingEntry});
        mappingEntry.setFileOwnership(FileOwnership.NOT_OWNED);
        mappingEntry.setSource(stateHandle);
        LOG.trace("Give up ownership for file: {}, the source is now backed by: {}", (Object)mappingEntry, (Object)stateHandle);
    }

    private Path forceLocalPath(Path filePath) {
        return new Path(this.localBase, filePath.getName());
    }

    private Path toUUIDPath(Path filePath) {
        return new Path(filePath.getParent(), UUID.randomUUID().toString());
    }
}

