/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.formats.parquet.row;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.List;
import org.apache.flink.formats.parquet.ParquetFileFormatFactory;
import org.apache.flink.formats.parquet.utils.ParquetSchemaConverter;
import org.apache.flink.formats.parquet.vector.reader.TimestampColumnReader;
import org.apache.flink.table.data.ArrayData;
import org.apache.flink.table.data.MapData;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.data.StringData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.ArrayType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.IntType;
import org.apache.flink.table.types.logical.LocalZonedTimestampType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.MapType;
import org.apache.flink.table.types.logical.MultisetType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.logical.TimestampType;
import org.apache.flink.util.Preconditions;
import org.apache.hadoop.conf.Configuration;
import org.apache.parquet.io.api.Binary;
import org.apache.parquet.io.api.RecordConsumer;
import org.apache.parquet.schema.GroupType;
import org.apache.parquet.schema.LogicalTypeAnnotation;
import org.apache.parquet.schema.Type;

public class ParquetRowDataWriter {
    private final RowWriter rowWriter;
    private final RecordConsumer recordConsumer;
    private final boolean utcTimestamp;
    private final Configuration conf;
    private boolean useInt64 = false;
    private LogicalTypeAnnotation.TimeUnit timeUnit;

    public ParquetRowDataWriter(RecordConsumer recordConsumer, RowType rowType, GroupType schema, boolean utcTimestamp, Configuration conf) {
        this.recordConsumer = recordConsumer;
        this.utcTimestamp = utcTimestamp;
        this.conf = conf;
        if (this.conf != null) {
            this.useInt64 = this.conf.getBoolean("parquet." + ParquetFileFormatFactory.WRITE_INT64_TIMESTAMP.key(), ((Boolean)ParquetFileFormatFactory.WRITE_INT64_TIMESTAMP.defaultValue()).booleanValue());
            if (this.useInt64) {
                this.timeUnit = LogicalTypeAnnotation.TimeUnit.valueOf(this.conf.get("parquet." + ParquetFileFormatFactory.TIMESTAMP_TIME_UNIT.key(), (String)ParquetFileFormatFactory.TIMESTAMP_TIME_UNIT.defaultValue()).toUpperCase());
            }
        }
        this.rowWriter = new RowWriter(rowType, schema);
    }

    public void write(RowData record) {
        this.recordConsumer.startMessage();
        this.rowWriter.write(record);
        this.recordConsumer.endMessage();
    }

    private FieldWriter createWriter(LogicalType t, Type type) {
        if (type.isPrimitive()) {
            switch (t.getTypeRoot()) {
                case CHAR: 
                case VARCHAR: {
                    return new StringWriter();
                }
                case BOOLEAN: {
                    return new BooleanWriter();
                }
                case BINARY: 
                case VARBINARY: {
                    return new BinaryWriter();
                }
                case DECIMAL: {
                    DecimalType decimalType = (DecimalType)t;
                    return this.createDecimalWriter(decimalType.getPrecision(), decimalType.getScale());
                }
                case TINYINT: {
                    return new ByteWriter();
                }
                case SMALLINT: {
                    return new ShortWriter();
                }
                case DATE: 
                case TIME_WITHOUT_TIME_ZONE: 
                case INTEGER: {
                    return new IntWriter();
                }
                case BIGINT: {
                    return new LongWriter();
                }
                case FLOAT: {
                    return new FloatWriter();
                }
                case DOUBLE: {
                    return new DoubleWriter();
                }
                case TIMESTAMP_WITHOUT_TIME_ZONE: {
                    TimestampType timestampType = (TimestampType)t;
                    return new TimestampWriter(timestampType.getPrecision());
                }
                case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                    LocalZonedTimestampType localZonedTimestampType = (LocalZonedTimestampType)t;
                    return new TimestampWriter(localZonedTimestampType.getPrecision());
                }
            }
            throw new UnsupportedOperationException("Unsupported type: " + String.valueOf(type));
        }
        GroupType groupType = type.asGroupType();
        LogicalTypeAnnotation logicalType = type.getLogicalTypeAnnotation();
        if (t instanceof ArrayType && logicalType instanceof LogicalTypeAnnotation.ListLogicalTypeAnnotation) {
            return new ArrayWriter(((ArrayType)t).getElementType(), groupType);
        }
        if (t instanceof MapType && logicalType instanceof LogicalTypeAnnotation.MapLogicalTypeAnnotation) {
            return new MapWriter(((MapType)t).getKeyType(), ((MapType)t).getValueType(), groupType);
        }
        if (t instanceof MultisetType && logicalType instanceof LogicalTypeAnnotation.MapLogicalTypeAnnotation) {
            return new MapWriter(((MultisetType)t).getElementType(), (LogicalType)new IntType(false), groupType);
        }
        if (t instanceof RowType && type instanceof GroupType) {
            return new RowWriter((RowType)t, groupType);
        }
        throw new UnsupportedOperationException("Unsupported type: " + String.valueOf(type));
    }

    private void writeTimestamp(RecordConsumer recordConsumer, TimestampData timestampData) {
        if (this.useInt64) {
            recordConsumer.addLong(this.timestampToInt64(timestampData));
        } else {
            recordConsumer.addBinary(this.timestampToInt96(timestampData));
        }
    }

    private Long convertInt64ToLong(long mills, long nanosOfMillisecond) {
        switch (this.timeUnit) {
            case NANOS: {
                return mills * TimestampColumnReader.NANOS_PER_MILLISECOND + nanosOfMillisecond;
            }
            case MICROS: {
                return mills * TimestampColumnReader.MICROS_PER_MILLISECOND + nanosOfMillisecond / TimestampColumnReader.NANOS_PER_MICROSECONDS;
            }
            case MILLIS: {
                return mills;
            }
        }
        throw new IllegalArgumentException("Time unit not recognized");
    }

    private Long timestampToInt64(TimestampData timestampData) {
        long mills = 0L;
        long nanosOfMillisecond = 0L;
        if (this.utcTimestamp) {
            mills = timestampData.getMillisecond();
            nanosOfMillisecond = timestampData.getNanoOfMillisecond();
        } else {
            Timestamp timestamp = timestampData.toTimestamp();
            mills = timestamp.getTime();
            nanosOfMillisecond = (long)timestamp.getNanos() % TimestampColumnReader.NANOS_PER_MILLISECOND;
        }
        return this.convertInt64ToLong(mills, nanosOfMillisecond);
    }

    private Binary timestampToInt96(TimestampData timestampData) {
        long nanosOfDay;
        int julianDay;
        if (this.utcTimestamp) {
            long mills = timestampData.getMillisecond();
            julianDay = (int)(mills / TimestampColumnReader.MILLIS_IN_DAY + 2440588L);
            nanosOfDay = mills % TimestampColumnReader.MILLIS_IN_DAY * TimestampColumnReader.NANOS_PER_MILLISECOND + (long)timestampData.getNanoOfMillisecond();
        } else {
            Timestamp timestamp = timestampData.toTimestamp();
            long mills = timestamp.getTime();
            julianDay = (int)(mills / TimestampColumnReader.MILLIS_IN_DAY + 2440588L);
            nanosOfDay = mills % TimestampColumnReader.MILLIS_IN_DAY / 1000L * TimestampColumnReader.NANOS_PER_SECOND + (long)timestamp.getNanos();
        }
        ByteBuffer buf = ByteBuffer.allocate(12);
        buf.order(ByteOrder.LITTLE_ENDIAN);
        buf.putLong(nanosOfDay);
        buf.putInt(julianDay);
        buf.flip();
        return Binary.fromConstantByteBuffer(buf);
    }

    private FieldWriter createDecimalWriter(final int precision, final int scale) {
        Preconditions.checkArgument((precision <= 38 ? 1 : 0) != 0, (String)"Decimal precision %s exceeds max precision %s", (Object[])new Object[]{precision, 38});
        if (ParquetSchemaConverter.is32BitDecimal(precision) || ParquetSchemaConverter.is64BitDecimal(precision)) {
            class LongUnscaledBytesWriter
            implements FieldWriter {
                private final int numBytes;
                private final int initShift;
                private final byte[] decimalBuffer;

                private LongUnscaledBytesWriter() {
                    this.numBytes = ParquetSchemaConverter.computeMinBytesForDecimalPrecision(precision);
                    this.initShift = 8 * (this.numBytes - 1);
                    this.decimalBuffer = new byte[this.numBytes];
                }

                @Override
                public void write(ArrayData arrayData, int ordinal) {
                    long unscaledLong = arrayData.getDecimal(ordinal, precision, scale).toUnscaledLong();
                    this.addRecord(unscaledLong);
                }

                @Override
                public void write(RowData row, int ordinal) {
                    long unscaledLong = row.getDecimal(ordinal, precision, scale).toUnscaledLong();
                    this.addRecord(unscaledLong);
                }

                private void addRecord(long unscaledLong) {
                    int i = 0;
                    int shift = this.initShift;
                    while (i < this.numBytes) {
                        this.decimalBuffer[i] = (byte)(unscaledLong >> shift);
                        ++i;
                        shift -= 8;
                    }
                    ParquetRowDataWriter.this.recordConsumer.addBinary(Binary.fromReusedByteArray(this.decimalBuffer, 0, this.numBytes));
                }
            }
            return new LongUnscaledBytesWriter();
        }
        class UnscaledBytesWriter
        implements FieldWriter {
            private final int numBytes;
            private final byte[] decimalBuffer;

            private UnscaledBytesWriter() {
                this.numBytes = ParquetSchemaConverter.computeMinBytesForDecimalPrecision(precision);
                this.decimalBuffer = new byte[this.numBytes];
            }

            @Override
            public void write(ArrayData arrayData, int ordinal) {
                byte[] bytes = arrayData.getDecimal(ordinal, precision, scale).toUnscaledBytes();
                this.addRecord(bytes);
            }

            @Override
            public void write(RowData row, int ordinal) {
                byte[] bytes = row.getDecimal(ordinal, precision, scale).toUnscaledBytes();
                this.addRecord(bytes);
            }

            private void addRecord(byte[] bytes) {
                byte[] writtenBytes;
                if (bytes.length == this.numBytes) {
                    writtenBytes = bytes;
                } else {
                    byte signByte = bytes[0] < 0 ? (byte)-1 : 0;
                    Arrays.fill(this.decimalBuffer, 0, this.numBytes - bytes.length, signByte);
                    System.arraycopy(bytes, 0, this.decimalBuffer, this.numBytes - bytes.length, bytes.length);
                    writtenBytes = this.decimalBuffer;
                }
                ParquetRowDataWriter.this.recordConsumer.addBinary(Binary.fromReusedByteArray(writtenBytes, 0, this.numBytes));
            }
        }
        return new UnscaledBytesWriter();
    }

    private class RowWriter
    implements FieldWriter {
        private List<LogicalType> logicalTypes;
        private FieldWriter[] fieldWriters;
        private final String[] fieldNames;

        public RowWriter(RowType rowType, GroupType groupType) {
            this.fieldNames = rowType.getFieldNames().toArray(new String[0]);
            this.logicalTypes = rowType.getChildren();
            this.fieldWriters = new FieldWriter[rowType.getFieldCount()];
            for (int i = 0; i < this.fieldWriters.length; ++i) {
                this.fieldWriters[i] = ParquetRowDataWriter.this.createWriter(this.logicalTypes.get(i), groupType.getType(i));
            }
        }

        public void write(RowData row) {
            for (int i = 0; i < this.fieldWriters.length; ++i) {
                if (row.isNullAt(i)) continue;
                String fieldName = this.fieldNames[i];
                FieldWriter writer = this.fieldWriters[i];
                ParquetRowDataWriter.this.recordConsumer.startField(fieldName, i);
                writer.write(row, i);
                ParquetRowDataWriter.this.recordConsumer.endField(fieldName, i);
            }
        }

        @Override
        public void write(RowData row, int ordinal) {
            ParquetRowDataWriter.this.recordConsumer.startGroup();
            RowData rowData = row.getRow(ordinal, this.fieldWriters.length);
            this.write(rowData);
            ParquetRowDataWriter.this.recordConsumer.endGroup();
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            ParquetRowDataWriter.this.recordConsumer.startGroup();
            RowData rowData = arrayData.getRow(ordinal, this.fieldWriters.length);
            this.write(rowData);
            ParquetRowDataWriter.this.recordConsumer.endGroup();
        }
    }

    private class ArrayWriter
    implements FieldWriter {
        private String elementName;
        private FieldWriter elementWriter;
        private String repeatedGroupName;

        private ArrayWriter(LogicalType t, GroupType groupType) {
            GroupType repeatedType = groupType.getType(0).asGroupType();
            this.repeatedGroupName = repeatedType.getName();
            Type elementType = repeatedType.getType(0);
            this.elementName = elementType.getName();
            this.elementWriter = ParquetRowDataWriter.this.createWriter(t, elementType);
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeArrayData(row.getArray(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeArrayData(arrayData.getArray(ordinal));
        }

        private void writeArrayData(ArrayData arrayData) {
            ParquetRowDataWriter.this.recordConsumer.startGroup();
            int listLength = arrayData.size();
            if (listLength > 0) {
                ParquetRowDataWriter.this.recordConsumer.startField(this.repeatedGroupName, 0);
                for (int i = 0; i < listLength; ++i) {
                    ParquetRowDataWriter.this.recordConsumer.startGroup();
                    if (!arrayData.isNullAt(i)) {
                        ParquetRowDataWriter.this.recordConsumer.startField(this.elementName, 0);
                        this.elementWriter.write(arrayData, i);
                        ParquetRowDataWriter.this.recordConsumer.endField(this.elementName, 0);
                    }
                    ParquetRowDataWriter.this.recordConsumer.endGroup();
                }
                ParquetRowDataWriter.this.recordConsumer.endField(this.repeatedGroupName, 0);
            }
            ParquetRowDataWriter.this.recordConsumer.endGroup();
        }
    }

    private class MapWriter
    implements FieldWriter {
        private String repeatedGroupName;
        private String keyName;
        private String valueName;
        private FieldWriter keyWriter;
        private FieldWriter valueWriter;

        private MapWriter(LogicalType keyType, LogicalType valueType, GroupType groupType) {
            GroupType repeatedType = groupType.getType(0).asGroupType();
            this.repeatedGroupName = repeatedType.getName();
            Type type = repeatedType.getType(0);
            this.keyName = type.getName();
            this.keyWriter = ParquetRowDataWriter.this.createWriter(keyType, type);
            Type valuetype = repeatedType.getType(1);
            this.valueName = valuetype.getName();
            this.valueWriter = ParquetRowDataWriter.this.createWriter(valueType, valuetype);
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeMapData(row.getMap(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeMapData(arrayData.getMap(ordinal));
        }

        private void writeMapData(MapData mapData) {
            ParquetRowDataWriter.this.recordConsumer.startGroup();
            if (mapData != null && mapData.size() > 0) {
                ParquetRowDataWriter.this.recordConsumer.startField(this.repeatedGroupName, 0);
                ArrayData keyArray = mapData.keyArray();
                ArrayData valueArray = mapData.valueArray();
                for (int i = 0; i < keyArray.size(); ++i) {
                    ParquetRowDataWriter.this.recordConsumer.startGroup();
                    if (keyArray.isNullAt(i)) {
                        throw new IllegalArgumentException("Parquet does not support null keys in maps. See https://github.com/apache/parquet-format/blob/master/LogicalTypes.md#maps for more details.");
                    }
                    ParquetRowDataWriter.this.recordConsumer.startField(this.keyName, 0);
                    this.keyWriter.write(keyArray, i);
                    ParquetRowDataWriter.this.recordConsumer.endField(this.keyName, 0);
                    if (!valueArray.isNullAt(i)) {
                        ParquetRowDataWriter.this.recordConsumer.startField(this.valueName, 1);
                        this.valueWriter.write(valueArray, i);
                        ParquetRowDataWriter.this.recordConsumer.endField(this.valueName, 1);
                    }
                    ParquetRowDataWriter.this.recordConsumer.endGroup();
                }
                ParquetRowDataWriter.this.recordConsumer.endField(this.repeatedGroupName, 0);
            }
            ParquetRowDataWriter.this.recordConsumer.endGroup();
        }
    }

    private class TimestampWriter
    implements FieldWriter {
        private final int precision;

        private TimestampWriter(int precision) {
            this.precision = precision;
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeTimestamp(row.getTimestamp(ordinal, this.precision));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeTimestamp(arrayData.getTimestamp(ordinal, this.precision));
        }

        private void writeTimestamp(TimestampData value) {
            ParquetRowDataWriter.this.writeTimestamp(ParquetRowDataWriter.this.recordConsumer, value);
        }
    }

    private class IntWriter
    implements FieldWriter {
        private IntWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeInt(row.getInt(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeInt(arrayData.getInt(ordinal));
        }

        private void writeInt(int value) {
            ParquetRowDataWriter.this.recordConsumer.addInteger(value);
        }
    }

    private class BinaryWriter
    implements FieldWriter {
        private BinaryWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeBinary(row.getBinary(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeBinary(arrayData.getBinary(ordinal));
        }

        private void writeBinary(byte[] value) {
            ParquetRowDataWriter.this.recordConsumer.addBinary(Binary.fromReusedByteArray(value));
        }
    }

    private class StringWriter
    implements FieldWriter {
        private StringWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeString(row.getString(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeString(arrayData.getString(ordinal));
        }

        private void writeString(StringData value) {
            ParquetRowDataWriter.this.recordConsumer.addBinary(Binary.fromReusedByteArray(value.toBytes()));
        }
    }

    private class DoubleWriter
    implements FieldWriter {
        private DoubleWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeDouble(row.getDouble(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeDouble(arrayData.getDouble(ordinal));
        }

        private void writeDouble(double value) {
            ParquetRowDataWriter.this.recordConsumer.addDouble(value);
        }
    }

    private class FloatWriter
    implements FieldWriter {
        private FloatWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeFloat(row.getFloat(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeFloat(arrayData.getFloat(ordinal));
        }

        private void writeFloat(float value) {
            ParquetRowDataWriter.this.recordConsumer.addFloat(value);
        }
    }

    private class LongWriter
    implements FieldWriter {
        private LongWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeLong(row.getLong(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeLong(arrayData.getLong(ordinal));
        }

        private void writeLong(long value) {
            ParquetRowDataWriter.this.recordConsumer.addLong(value);
        }
    }

    private class ShortWriter
    implements FieldWriter {
        private ShortWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeShort(row.getShort(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeShort(arrayData.getShort(ordinal));
        }

        private void writeShort(short value) {
            ParquetRowDataWriter.this.recordConsumer.addInteger(value);
        }
    }

    private class ByteWriter
    implements FieldWriter {
        private ByteWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeByte(row.getByte(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeByte(arrayData.getByte(ordinal));
        }

        private void writeByte(byte value) {
            ParquetRowDataWriter.this.recordConsumer.addInteger(value);
        }
    }

    private class BooleanWriter
    implements FieldWriter {
        private BooleanWriter() {
        }

        @Override
        public void write(RowData row, int ordinal) {
            this.writeBoolean(row.getBoolean(ordinal));
        }

        @Override
        public void write(ArrayData arrayData, int ordinal) {
            this.writeBoolean(arrayData.getBoolean(ordinal));
        }

        private void writeBoolean(boolean value) {
            ParquetRowDataWriter.this.recordConsumer.addBoolean(value);
        }
    }

    private static interface FieldWriter {
        public void write(RowData var1, int var2);

        public void write(ArrayData var1, int var2);
    }
}

