/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.util.profiler;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import one.profiler.AsyncProfiler;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.RestOptions;
import org.apache.flink.runtime.rest.messages.ProfilingInfo;
import org.apache.flink.util.CollectionUtil;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;
import org.apache.flink.util.concurrent.ExecutorThreadFactory;
import org.apache.flink.util.concurrent.FutureUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProfilingService
implements Closeable {
    protected static final Logger LOG = LoggerFactory.getLogger(ProfilingService.class);
    private static final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd_HH_mm_ss");
    private static volatile ProfilingService instance;
    private final Map<String, ArrayDeque<ProfilingInfo>> profilingMap = new HashMap<String, ArrayDeque<ProfilingInfo>>();
    private final String profilingResultDir;
    private final int historySizeLimit;
    private final ScheduledExecutorService scheduledExecutor;
    private ProfilingFuture profilingFuture;

    private ProfilingService(Configuration configs) {
        this.historySizeLimit = configs.get(RestOptions.MAX_PROFILING_HISTORY_SIZE);
        Preconditions.checkArgument(this.historySizeLimit > 0, String.format("Configured %s must be positive.", RestOptions.MAX_PROFILING_HISTORY_SIZE.key()));
        this.profilingResultDir = configs.get(RestOptions.PROFILING_RESULT_DIR);
        this.scheduledExecutor = Executors.newSingleThreadScheduledExecutor(new ExecutorThreadFactory.Builder().setPoolName("flink-profiling-service").build());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static ProfilingService getInstance(Configuration configs) {
        if (instance != null) return instance;
        Class<ProfilingService> clazz = ProfilingService.class;
        synchronized (ProfilingService.class) {
            if (instance != null) return instance;
            instance = new ProfilingService(configs);
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return instance;
        }
    }

    public CompletableFuture<ProfilingInfo> requestProfiling(String resourceID, long duration, ProfilingInfo.ProfilingMode mode) {
        if (this.profilingFuture != null && !this.profilingFuture.isDone()) {
            return FutureUtils.completedExceptionally(new IllegalStateException(resourceID + " is still under profiling."));
        }
        ProfilingInfo profilingInfo = ProfilingInfo.create(duration, mode);
        this.profilingMap.putIfAbsent(resourceID, new ArrayDeque());
        this.profilingMap.get(resourceID).addFirst(profilingInfo);
        AsyncProfiler profiler = AsyncProfiler.getInstance();
        try {
            String response = profiler.execute(ProfilerConstants.COMMAND_START.msg + profilingInfo.getProfilingMode().getCode());
            if (StringUtils.isNullOrWhitespaceOnly(response) || !response.startsWith(ProfilerConstants.PROFILER_STARTED_SUCCESS.msg)) {
                return CompletableFuture.completedFuture(profilingInfo.fail("Start profiler failed. " + response));
            }
        }
        catch (Exception e) {
            return CompletableFuture.completedFuture(profilingInfo.fail("Start profiler failed. " + e));
        }
        this.profilingFuture = new ProfilingFuture(duration, () -> this.stopProfiling(resourceID));
        return CompletableFuture.completedFuture(profilingInfo);
    }

    private void stopProfiling(String resourceID) {
        AsyncProfiler profiler = AsyncProfiler.getInstance();
        ArrayDeque<ProfilingInfo> profilingList = this.profilingMap.get(resourceID);
        Preconditions.checkState(!CollectionUtil.isNullOrEmpty(profilingList));
        ProfilingInfo info = profilingList.getFirst();
        try {
            String fileName = this.formatOutputFileName(resourceID, info);
            String outputPath = new File(this.profilingResultDir, fileName).getPath();
            String response = profiler.execute(ProfilerConstants.COMMAND_STOP.msg + outputPath);
            if (!StringUtils.isNullOrWhitespaceOnly(response) && response.startsWith(ProfilerConstants.PROFILER_STOPPED_SUCCESS.msg)) {
                info.success(fileName);
            } else {
                info.fail("Stop profiler failed. " + response);
            }
            this.rollingClearing(profilingList);
        }
        catch (Throwable e) {
            info.fail("Stop profiler failed. " + e);
        }
    }

    private void rollingClearing(ArrayDeque<ProfilingInfo> profilingList) {
        while (profilingList.size() > this.historySizeLimit) {
            String outputFile;
            ProfilingInfo info = profilingList.pollLast();
            String string = outputFile = info != null ? info.getOutputFile() : "";
            if (StringUtils.isNullOrWhitespaceOnly(outputFile)) continue;
            try {
                Files.deleteIfExists(Paths.get(this.profilingResultDir, outputFile));
            }
            catch (Exception e) {
                LOG.error(String.format("Clearing file for %s failed. Skipped.", info), (Throwable)e);
            }
        }
    }

    private String formatOutputFileName(String resourceID, ProfilingInfo info) {
        return String.format("%s_%s_%s.html", new Object[]{resourceID, info.getProfilingMode(), sdf.format(new Date())});
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.profilingFuture != null && !this.profilingFuture.isDone()) {
                this.profilingFuture.cancel();
            }
            if (!this.scheduledExecutor.isShutdown()) {
                this.scheduledExecutor.shutdownNow();
            }
        }
        catch (Exception e) {
            LOG.error("Exception thrown during stopping profiling service. ", (Throwable)e);
        }
        finally {
            instance = null;
        }
    }

    public CompletableFuture<Collection<ProfilingInfo>> getProfilingList(String resourceID) {
        return CompletableFuture.completedFuture((Collection)this.profilingMap.getOrDefault(resourceID, new ArrayDeque()));
    }

    public String getProfilingResultDir() {
        return this.profilingResultDir;
    }

    @VisibleForTesting
    ArrayDeque<ProfilingInfo> getProfilingListForTest(String resourceID) {
        return this.profilingMap.getOrDefault(resourceID, new ArrayDeque());
    }

    @VisibleForTesting
    int getHistorySizeLimit() {
        return this.historySizeLimit;
    }

    @VisibleForTesting
    ProfilingFuture getProfilingFuture() {
        return this.profilingFuture;
    }

    class ProfilingFuture {
        private final ScheduledFuture<?> future;
        private final Runnable handler;

        public ProfilingFuture(long duration, Runnable handler) {
            this.handler = handler;
            this.future = ProfilingService.this.scheduledExecutor.schedule(handler, duration, TimeUnit.SECONDS);
        }

        public boolean isDone() {
            return this.future == null || this.future.isDone();
        }

        public boolean cancel() {
            if (this.isDone()) {
                return true;
            }
            if (!this.future.cancel(true)) {
                return false;
            }
            this.handler.run();
            return true;
        }
    }

    static enum ProfilerConstants {
        PROFILER_STARTED_SUCCESS("Profiling started"),
        PROFILER_STOPPED_SUCCESS("OK"),
        COMMAND_START("start,event="),
        COMMAND_STOP("stop,file=");

        private final String msg;

        private ProfilerConstants(String msg) {
            this.msg = msg;
        }
    }
}

