/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.jobmaster.slotpool;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.jobmaster.SlotInfo;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlot;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlotPool;
import org.apache.flink.runtime.jobmaster.slotpool.SlotInfoWithUtilization;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultAllocatedSlotPool
implements AllocatedSlotPool {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultAllocatedSlotPool.class);
    private final Map<AllocationID, AllocatedSlot> registeredSlots = new HashMap<AllocationID, AllocatedSlot>();
    private final FreeSlots freeSlots;
    private final Map<ResourceID, Set<AllocationID>> slotsPerTaskExecutor = new HashMap<ResourceID, Set<AllocationID>>();

    public DefaultAllocatedSlotPool() {
        this.freeSlots = new FreeSlots();
    }

    @Override
    public void addSlots(Collection<AllocatedSlot> slots, long currentTime) {
        for (AllocatedSlot slot : slots) {
            this.addSlot(slot, currentTime);
        }
    }

    private void addSlot(AllocatedSlot slot, long currentTime) {
        Preconditions.checkState((!this.registeredSlots.containsKey((Object)slot.getAllocationId()) ? 1 : 0) != 0, (String)"The slot pool already contains a slot with id %s", (Object[])new Object[]{slot.getAllocationId()});
        this.addSlotInternal(slot, currentTime);
        this.slotsPerTaskExecutor.computeIfAbsent(slot.getTaskManagerId(), resourceID -> new HashSet()).add(slot.getAllocationId());
    }

    private void addSlotInternal(AllocatedSlot slot, long currentTime) {
        this.registeredSlots.put(slot.getAllocationId(), slot);
        this.freeSlots.addFreeSlot(slot.getAllocationId(), slot.getTaskManagerId(), currentTime);
    }

    @Override
    public Optional<AllocatedSlot> removeSlot(AllocationID allocationId) {
        AllocatedSlot removedSlot = this.removeSlotInternal(allocationId);
        if (removedSlot != null) {
            ResourceID owner = removedSlot.getTaskManagerId();
            this.slotsPerTaskExecutor.computeIfPresent(owner, (resourceID, allocationIds) -> {
                allocationIds.remove((Object)allocationId);
                if (allocationIds.isEmpty()) {
                    return null;
                }
                return allocationIds;
            });
            return Optional.of(removedSlot);
        }
        return Optional.empty();
    }

    @Nullable
    private AllocatedSlot removeSlotInternal(AllocationID allocationId) {
        AllocatedSlot removedSlot = this.registeredSlots.remove((Object)allocationId);
        if (removedSlot != null) {
            this.freeSlots.removeFreeSlot(allocationId, removedSlot.getTaskManagerId());
        }
        return removedSlot;
    }

    @Override
    public AllocatedSlotPool.AllocatedSlotsAndReservationStatus removeSlots(ResourceID owner) {
        Set<AllocationID> slotsOfTaskExecutor = this.slotsPerTaskExecutor.remove(owner);
        if (slotsOfTaskExecutor != null) {
            ArrayList<AllocatedSlot> removedSlots = new ArrayList<AllocatedSlot>();
            HashMap<AllocationID, ReservationStatus> removedSlotsReservationStatus = new HashMap<AllocationID, ReservationStatus>();
            for (AllocationID allocationId : slotsOfTaskExecutor) {
                ReservationStatus reservationStatus = this.containsFreeSlot(allocationId) ? ReservationStatus.FREE : ReservationStatus.RESERVED;
                AllocatedSlot removedSlot = (AllocatedSlot)Preconditions.checkNotNull((Object)this.removeSlotInternal(allocationId));
                removedSlots.add(removedSlot);
                removedSlotsReservationStatus.put(removedSlot.getAllocationId(), reservationStatus);
            }
            return new DefaultAllocatedSlotsAndReservationStatus(removedSlots, removedSlotsReservationStatus);
        }
        return new DefaultAllocatedSlotsAndReservationStatus(Collections.emptyList(), Collections.emptyMap());
    }

    @Override
    public boolean containsSlots(ResourceID owner) {
        return this.slotsPerTaskExecutor.containsKey(owner);
    }

    @Override
    public boolean containsSlot(AllocationID allocationId) {
        return this.registeredSlots.containsKey((Object)allocationId);
    }

    @Override
    public boolean containsFreeSlot(AllocationID allocationId) {
        return this.freeSlots.contains(allocationId);
    }

    @Override
    public AllocatedSlot reserveFreeSlot(AllocationID allocationId) {
        LOG.debug("Reserve free slot with allocation id {}.", (Object)allocationId);
        AllocatedSlot slot = this.registeredSlots.get((Object)allocationId);
        Preconditions.checkNotNull((Object)slot, (String)"The slot with id %s was not exists.", (Object[])new Object[]{allocationId});
        Preconditions.checkState((this.freeSlots.removeFreeSlot(allocationId, slot.getTaskManagerId()) != null ? 1 : 0) != 0, (String)"The slot with id %s was not free.", (Object[])new Object[]{allocationId});
        return this.registeredSlots.get((Object)allocationId);
    }

    @Override
    public Optional<AllocatedSlot> freeReservedSlot(AllocationID allocationId, long currentTime) {
        AllocatedSlot allocatedSlot = this.registeredSlots.get((Object)allocationId);
        if (allocatedSlot != null && !this.freeSlots.contains(allocationId)) {
            this.freeSlots.addFreeSlot(allocationId, allocatedSlot.getTaskManagerId(), currentTime);
            return Optional.of(allocatedSlot);
        }
        return Optional.empty();
    }

    @Override
    public Optional<SlotInfo> getSlotInformation(AllocationID allocationID) {
        return Optional.ofNullable(this.registeredSlots.get((Object)allocationID));
    }

    @Override
    public Collection<AllocatedSlotPool.FreeSlotInfo> getFreeSlotsInformation() {
        ArrayList<AllocatedSlotPool.FreeSlotInfo> freeSlotInfos = new ArrayList<AllocatedSlotPool.FreeSlotInfo>();
        for (Map.Entry<AllocationID, Long> freeSlot : this.freeSlots.getFreeSlotsSince().entrySet()) {
            AllocatedSlot allocatedSlot = (AllocatedSlot)Preconditions.checkNotNull((Object)this.registeredSlots.get((Object)freeSlot.getKey()));
            SlotInfoWithUtilization slotInfoWithUtilization = SlotInfoWithUtilization.from(allocatedSlot, this::getTaskExecutorUtilization);
            freeSlotInfos.add(DefaultFreeSlotInfo.create(slotInfoWithUtilization, freeSlot.getValue()));
        }
        return freeSlotInfos;
    }

    public double getTaskExecutorUtilization(ResourceID resourceId) {
        Set<AllocationID> slots = this.slotsPerTaskExecutor.get(resourceId);
        Preconditions.checkNotNull(slots, (String)"There is no slots on %s", (Object[])new Object[]{resourceId});
        return (double)(slots.size() - this.freeSlots.getFreeSlotsNumberOfTaskExecutor(resourceId)) / (double)slots.size();
    }

    @Override
    public Collection<? extends SlotInfo> getAllSlotsInformation() {
        return this.registeredSlots.values();
    }

    private static enum ReservationStatus {
        FREE,
        RESERVED;

    }

    private static final class DefaultAllocatedSlotsAndReservationStatus
    implements AllocatedSlotPool.AllocatedSlotsAndReservationStatus {
        private final Collection<AllocatedSlot> slots;
        private final Map<AllocationID, ReservationStatus> reservationStatus;

        private DefaultAllocatedSlotsAndReservationStatus(Collection<AllocatedSlot> slots, Map<AllocationID, ReservationStatus> reservationStatus) {
            this.slots = slots;
            this.reservationStatus = reservationStatus;
        }

        @Override
        public boolean wasFree(AllocationID allocatedSlot) {
            return this.reservationStatus.get((Object)allocatedSlot) == ReservationStatus.FREE;
        }

        @Override
        public Collection<AllocatedSlot> getAllocatedSlots() {
            return this.slots;
        }
    }

    private static final class DefaultFreeSlotInfo
    implements AllocatedSlotPool.FreeSlotInfo {
        private final SlotInfoWithUtilization slotInfoWithUtilization;
        private final long freeSince;

        private DefaultFreeSlotInfo(SlotInfoWithUtilization slotInfoWithUtilization, long freeSince) {
            this.slotInfoWithUtilization = slotInfoWithUtilization;
            this.freeSince = freeSince;
        }

        @Override
        public SlotInfoWithUtilization asSlotInfo() {
            return this.slotInfoWithUtilization;
        }

        @Override
        public long getFreeSince() {
            return this.freeSince;
        }

        private static DefaultFreeSlotInfo create(SlotInfoWithUtilization slotInfoWithUtilization, long idleSince) {
            return new DefaultFreeSlotInfo((SlotInfoWithUtilization)Preconditions.checkNotNull((Object)slotInfoWithUtilization), idleSince);
        }
    }

    private static final class FreeSlots {
        private final Map<AllocationID, Long> freeSlotsSince = new HashMap<AllocationID, Long>();
        private final Map<ResourceID, Integer> freeSlotsNumberPerTaskExecutor = new HashMap<ResourceID, Integer>();

        private FreeSlots() {
        }

        public void addFreeSlot(AllocationID allocationId, ResourceID resourceId, long currentTime) {
            if (this.freeSlotsSince.put(allocationId, currentTime) == null) {
                this.freeSlotsNumberPerTaskExecutor.merge(resourceId, 1, Integer::sum);
            }
        }

        public Long removeFreeSlot(AllocationID allocationId, ResourceID resourceId) {
            Long freeSince = this.freeSlotsSince.remove((Object)allocationId);
            if (freeSince != null) {
                this.freeSlotsNumberPerTaskExecutor.computeIfPresent(resourceId, (ignore, count) -> {
                    int newCount = count - 1;
                    return newCount == 0 ? null : Integer.valueOf(newCount);
                });
            }
            return freeSince;
        }

        public boolean contains(AllocationID allocationId) {
            return this.freeSlotsSince.containsKey((Object)allocationId);
        }

        public int getFreeSlotsNumberOfTaskExecutor(ResourceID resourceId) {
            return this.freeSlotsNumberPerTaskExecutor.getOrDefault(resourceId, 0);
        }

        public Map<AllocationID, Long> getFreeSlotsSince() {
            return this.freeSlotsSince;
        }
    }
}

