/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.blob;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.io.FileUtils;
import org.apache.flink.api.common.JobID;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.fs.local.LocalDataOutputStream;
import org.apache.flink.runtime.blob.BlobKey;
import org.apache.flink.runtime.blob.BlobUtils;
import org.apache.flink.runtime.blob.FileSystemBlobStore;
import org.apache.flink.runtime.blob.PermanentBlobKey;
import org.apache.flink.runtime.state.filesystem.TestFs;
import org.apache.flink.testutils.TestFileSystem;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.TestLoggerExtension;
import org.apache.flink.util.function.FunctionWithException;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.api.io.TempDir;

@ExtendWith(value={TestLoggerExtension.class})
class FileSystemBlobStoreTest {
    private FileSystemBlobStore testInstance;
    private java.nio.file.Path storagePath;

    FileSystemBlobStoreTest() {
    }

    @BeforeEach
    public void createTestInstance(@TempDir java.nio.file.Path storagePath) throws IOException {
        this.testInstance = new FileSystemBlobStore((FileSystem)new TestFileSystem(), storagePath.toString());
        this.storagePath = storagePath;
    }

    @AfterEach
    public void finalizeTestInstance() throws IOException {
        this.testInstance.close();
    }

    @Test
    public void testSuccessfulPut() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("put");
        JobID jobId = new JobID();
        BlobKey blobKey = this.createPermanentBlobKeyFromFile(temporaryFile);
        Assertions.assertThat((java.nio.file.Path)this.getBlobDirectoryPath()).doesNotExist();
        boolean successfullyWritten = this.testInstance.put(temporaryFile.toFile(), jobId, blobKey);
        Assertions.assertThat((boolean)successfullyWritten).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId)).isDirectory().exists();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId, blobKey)).isNotEmptyFile().hasSameTextualContentAs(temporaryFile);
    }

    @Test
    public void testMissingFilePut() throws IOException {
        Assertions.assertThatThrownBy(() -> this.testInstance.put(new File("/not/existing/file"), new JobID(), (BlobKey)new PermanentBlobKey())).isInstanceOf(FileNotFoundException.class);
    }

    @Test
    public void testSuccessfulGet() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("get");
        JobID jobId = new JobID();
        BlobKey blobKey = this.createPermanentBlobKeyFromFile(temporaryFile);
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, blobKey)).isTrue();
        java.nio.file.Path targetFile = Files.createTempFile("filesystemblobstoretest-get-target-", "", new FileAttribute[0]);
        Assertions.assertThat((java.nio.file.Path)targetFile).isEmptyFile();
        boolean successfullyGet = this.testInstance.get(jobId, blobKey, targetFile.toFile());
        Assertions.assertThat((boolean)successfullyGet).isTrue();
        Assertions.assertThat((java.nio.file.Path)targetFile).hasSameTextualContentAs(temporaryFile);
    }

    @Test
    public void testGetWithWrongJobId() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("get");
        BlobKey blobKey = this.createPermanentBlobKeyFromFile(temporaryFile);
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), new JobID(), blobKey)).isTrue();
        Assertions.assertThatThrownBy(() -> this.testInstance.get(new JobID(), blobKey, Files.createTempFile("filesystemblobstoretest-get-with-wrong-jobid-", "", new FileAttribute[0]).toFile())).isInstanceOf(FileNotFoundException.class);
    }

    @Test
    public void testGetWithWrongBlobKey() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("get");
        JobID jobId = new JobID();
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, (BlobKey)new PermanentBlobKey())).isTrue();
        Assertions.assertThatThrownBy(() -> this.testInstance.get(jobId, (BlobKey)new PermanentBlobKey(), Files.createTempFile("filesystemblobstoretest-get-with-wrong-blobkey-", "", new FileAttribute[0]).toFile())).isInstanceOf(FileNotFoundException.class);
    }

    @Test
    public void testSuccessfulDeleteOnlyBlob() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("delete");
        JobID jobId = new JobID();
        BlobKey blobKey = this.createPermanentBlobKeyFromFile(temporaryFile);
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, blobKey)).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId)).isDirectory().exists();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId, blobKey)).isNotEmptyFile();
        boolean successfullyDeleted = this.testInstance.delete(jobId, blobKey);
        Assertions.assertThat((boolean)successfullyDeleted).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId)).doesNotExist();
    }

    @Test
    public void testSuccessfulDeleteBlob() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("delete");
        JobID jobId = new JobID();
        BlobKey blobKey = this.createPermanentBlobKeyFromFile(temporaryFile);
        PermanentBlobKey otherBlobKey = new PermanentBlobKey();
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, blobKey)).isTrue();
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, (BlobKey)otherBlobKey)).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId)).isDirectory().exists();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId, blobKey)).isNotEmptyFile();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId, (BlobKey)otherBlobKey)).isNotEmptyFile();
        boolean successfullyDeleted = this.testInstance.delete(jobId, blobKey);
        Assertions.assertThat((boolean)successfullyDeleted).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId, (BlobKey)otherBlobKey)).exists();
    }

    @Test
    public void testDeleteWithNotExistingJobId() {
        Assertions.assertThat((boolean)this.testInstance.delete(new JobID(), (BlobKey)new PermanentBlobKey())).isTrue();
    }

    @Test
    public void testDeleteWithNotExistingBlobKey() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("delete");
        JobID jobId = new JobID();
        BlobKey blobKey = this.createPermanentBlobKeyFromFile(temporaryFile);
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, blobKey)).isTrue();
        Assertions.assertThat((boolean)this.testInstance.delete(jobId, (BlobKey)new PermanentBlobKey())).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId, blobKey)).exists();
    }

    @Test
    public void testDeleteAll() throws IOException {
        java.nio.file.Path temporaryFile = this.createTemporaryFileWithContent("delete");
        JobID jobId = new JobID();
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, (BlobKey)new PermanentBlobKey())).isTrue();
        Assertions.assertThat((boolean)this.testInstance.put(temporaryFile.toFile(), jobId, (BlobKey)new PermanentBlobKey())).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId)).isDirectory().exists();
        Assertions.assertThat((Object[])this.getPath(jobId).toFile().listFiles()).hasSize(2);
        Assertions.assertThat((boolean)this.testInstance.deleteAll(jobId)).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId)).doesNotExist();
    }

    @Test
    public void testDeleteAllWithNotExistingJobId() {
        JobID jobId = new JobID();
        Assertions.assertThat((boolean)this.testInstance.deleteAll(jobId)).isTrue();
        Assertions.assertThat((java.nio.file.Path)this.getPath(jobId)).doesNotExist();
    }

    private java.nio.file.Path createTemporaryFileWithContent(String operationLabel) throws IOException {
        String actualContent = String.format("Content for testing the %s operation", operationLabel);
        java.nio.file.Path temporaryFile = Files.createTempFile(String.format("filesystemblobstoretest-%s-", operationLabel), "", new FileAttribute[0]);
        try (BufferedWriter writer = new BufferedWriter(new FileWriter(temporaryFile.toAbsolutePath().toString()));){
            writer.write(actualContent);
        }
        return temporaryFile;
    }

    private java.nio.file.Path getBlobDirectoryPath() {
        return this.storagePath.resolve("blob");
    }

    private java.nio.file.Path getPath(JobID jobId) {
        return this.getBlobDirectoryPath().resolve(String.format("job_%s", jobId));
    }

    private java.nio.file.Path getPath(JobID jobId, BlobKey blobKey) {
        return this.getPath(jobId).resolve(String.format("blob_%s", blobKey));
    }

    private BlobKey createPermanentBlobKeyFromFile(java.nio.file.Path path) throws IOException {
        Preconditions.checkArgument((!Files.isDirectory(path, new LinkOption[0]) ? 1 : 0) != 0);
        Preconditions.checkArgument((boolean)Files.exists(path, new LinkOption[0]));
        MessageDigest md = BlobUtils.createMessageDigest();
        try (FileInputStream is = new FileInputStream(path.toFile());){
            byte[] buf = new byte[1024];
            int bytesRead = ((InputStream)is).read(buf);
            while (bytesRead >= 0) {
                md.update(buf, 0, bytesRead);
                bytesRead = ((InputStream)is).read(buf);
            }
            BlobKey blobKey = BlobKey.createKey((BlobKey.BlobType)BlobKey.BlobType.PERMANENT_BLOB, (byte[])md.digest());
            return blobKey;
        }
    }

    @Test
    public void fileSystemBlobStoreCallsSyncOnPut(@TempDir java.nio.file.Path storageDirectory) throws IOException {
        java.nio.file.Path blobStoreDirectory = storageDirectory.resolve("blobStore");
        AtomicReference createdOutputStream = new AtomicReference();
        FunctionWithException outputStreamFactory = value -> {
            File file = new File(value.toString());
            FileUtils.createParentDirectories((File)file);
            TestingLocalDataOutputStream outputStream = new TestingLocalDataOutputStream(file);
            createdOutputStream.compareAndSet(null, outputStream);
            return outputStream;
        };
        try (FileSystemBlobStore fileSystemBlobStore = new FileSystemBlobStore((FileSystem)new TestFs((FunctionWithException<Path, FSDataOutputStream, IOException>)outputStreamFactory), blobStoreDirectory.toString());){
            BlobKey blobKey = BlobKey.createKey((BlobKey.BlobType)BlobKey.BlobType.PERMANENT_BLOB);
            File localFile = storageDirectory.resolve("localFile").toFile();
            FileUtils.createParentDirectories((File)localFile);
            FileUtils.writeStringToFile((File)localFile, (String)"foobar", (Charset)StandardCharsets.UTF_8);
            fileSystemBlobStore.put(localFile, new JobID(), blobKey);
            Assertions.assertThat((boolean)((TestingLocalDataOutputStream)((Object)createdOutputStream.get())).hasSyncBeenCalled()).isTrue();
        }
    }

    private static class TestingLocalDataOutputStream
    extends LocalDataOutputStream {
        private boolean hasSyncBeenCalled = false;

        private TestingLocalDataOutputStream(File file) throws IOException {
            super(file);
        }

        public void sync() throws IOException {
            this.hasSyncBeenCalled = true;
            super.sync();
        }

        public boolean hasSyncBeenCalled() {
            return this.hasSyncBeenCalled;
        }
    }
}

