/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.core.api.authn;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import org.apache.directory.server.core.api.authn.EncryptionMethod;
import org.apache.directory.server.core.api.authn.ppolicy.PasswordPolicyConfiguration;
import org.apache.directory.shared.ldap.model.constants.LdapSecurityConstants;
import org.apache.directory.shared.ldap.model.entry.Attribute;
import org.apache.directory.shared.ldap.model.entry.Value;
import org.apache.directory.shared.util.Base64;
import org.apache.directory.shared.util.DateUtils;
import org.apache.directory.shared.util.Strings;
import org.apache.directory.shared.util.UnixCrypt;

public class PasswordUtil {
    public static final int SHA1_LENGTH = 20;
    public static final int SHA256_LENGTH = 32;
    public static final int SHA384_LENGTH = 48;
    public static final int SHA512_LENGTH = 64;
    public static final int MD5_LENGTH = 16;

    public static LdapSecurityConstants findAlgorithm(byte[] credentials) {
        if (credentials == null || credentials.length == 0) {
            return null;
        }
        if (credentials[0] == 123) {
            int pos;
            for (pos = 1; pos < credentials.length && credentials[pos] != 125; ++pos) {
            }
            if (pos < credentials.length) {
                if (pos == 1) {
                    return null;
                }
                String algorithm = Strings.toLowerCase(new String(credentials, 1, pos - 1));
                return LdapSecurityConstants.getAlgorithm(algorithm);
            }
            return null;
        }
        return null;
    }

    public static byte[] createStoragePassword(String credentials, LdapSecurityConstants algorithm) {
        return PasswordUtil.createStoragePassword(Strings.getBytesUtf8(credentials), algorithm);
    }

    public static byte[] createStoragePassword(byte[] credentials, LdapSecurityConstants algorithm) {
        byte[] salt;
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SMD5: {
                salt = new byte[8];
                new SecureRandom().nextBytes(salt);
                break;
            }
            case HASH_METHOD_CRYPT: {
                salt = new byte[2];
                SecureRandom sr = new SecureRandom();
                int i1 = sr.nextInt(64);
                int i2 = sr.nextInt(64);
                salt[0] = (byte)(i1 < 12 ? i1 + 46 : (i1 < 38 ? i1 + 65 - 12 : i1 + 97 - 38));
                salt[1] = (byte)(i2 < 12 ? i2 + 46 : (i2 < 38 ? i2 + 65 - 12 : i2 + 97 - 38));
                break;
            }
            default: {
                salt = null;
            }
        }
        byte[] hashedPassword = PasswordUtil.encryptPassword(credentials, algorithm, salt);
        StringBuffer sb = new StringBuffer();
        if (algorithm != null) {
            sb.append('{').append(algorithm.getName().toUpperCase()).append('}');
            if (algorithm == LdapSecurityConstants.HASH_METHOD_CRYPT) {
                sb.append(Strings.utf8ToString(salt));
                sb.append(Strings.utf8ToString(hashedPassword));
            } else if (salt != null) {
                byte[] hashedPasswordWithSaltBytes = new byte[hashedPassword.length + salt.length];
                PasswordUtil.merge(hashedPasswordWithSaltBytes, hashedPassword, salt);
                sb.append(String.valueOf(Base64.encode(hashedPasswordWithSaltBytes)));
            } else {
                sb.append(String.valueOf(Base64.encode(hashedPassword)));
            }
        } else {
            sb.append(Strings.utf8ToString(hashedPassword));
        }
        return Strings.getBytesUtf8(sb.toString());
    }

    public static boolean compareCredentials(byte[] receivedCredentials, byte[] storedCredentials) {
        LdapSecurityConstants algorithm = PasswordUtil.findAlgorithm(storedCredentials);
        if (algorithm != null) {
            EncryptionMethod encryptionMethod = new EncryptionMethod(algorithm, null);
            byte[] encryptedStored = PasswordUtil.splitCredentials(storedCredentials, encryptionMethod);
            byte[] userPassword = PasswordUtil.encryptPassword(receivedCredentials, encryptionMethod.getAlgorithm(), encryptionMethod.getSalt());
            return Arrays.equals(userPassword, encryptedStored);
        }
        return Arrays.equals(storedCredentials, receivedCredentials);
    }

    public static byte[] encryptPassword(byte[] credentials, LdapSecurityConstants algorithm, byte[] salt) {
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SHA: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA, credentials, salt);
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA256, credentials, salt);
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA384, credentials, salt);
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA512, credentials, salt);
            }
            case HASH_METHOD_SMD5: 
            case HASH_METHOD_MD5: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_MD5, credentials, salt);
            }
            case HASH_METHOD_CRYPT: {
                String saltWithCrypted = UnixCrypt.crypt(Strings.utf8ToString(credentials), Strings.utf8ToString(salt));
                String crypted = saltWithCrypted.substring(2);
                return Strings.getBytesUtf8(crypted);
            }
        }
        return credentials;
    }

    private static byte[] digest(LdapSecurityConstants algorithm, byte[] password, byte[] salt) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(algorithm.getName());
        }
        catch (NoSuchAlgorithmException e1) {
            return null;
        }
        if (salt != null) {
            digest.update(password);
            digest.update(salt);
            return digest.digest();
        }
        return digest.digest(password);
    }

    public static byte[] splitCredentials(byte[] credentials, EncryptionMethod encryptionMethod) {
        int algoLength = encryptionMethod.getAlgorithm().getName().length() + 2;
        int hashLen = 0;
        switch (encryptionMethod.getAlgorithm()) {
            case HASH_METHOD_SHA: 
            case HASH_METHOD_MD5: {
                try {
                    return Base64.decode(new String(credentials, algoLength, credentials.length - algoLength, "UTF-8").toCharArray());
                }
                catch (UnsupportedEncodingException uee) {
                    return credentials;
                }
            }
            case HASH_METHOD_SMD5: {
                try {
                    byte[] passwordAndSalt = Base64.decode(new String(credentials, algoLength, credentials.length - algoLength, "UTF-8").toCharArray());
                    int saltLength = passwordAndSalt.length - 16;
                    encryptionMethod.setSalt(new byte[saltLength]);
                    byte[] password = new byte[16];
                    PasswordUtil.split(passwordAndSalt, 0, password, encryptionMethod.getSalt());
                    return password;
                }
                catch (UnsupportedEncodingException uee) {
                    return credentials;
                }
            }
            case HASH_METHOD_SSHA: {
                hashLen = 20;
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                if (hashLen == 0) {
                    hashLen = 32;
                }
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                if (hashLen == 0) {
                    hashLen = 48;
                }
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                if (hashLen == 0) {
                    hashLen = 64;
                }
                try {
                    byte[] passwordAndSalt = Base64.decode(new String(credentials, algoLength, credentials.length - algoLength, "UTF-8").toCharArray());
                    int saltLength = passwordAndSalt.length - hashLen;
                    encryptionMethod.setSalt(new byte[saltLength]);
                    byte[] password = new byte[hashLen];
                    PasswordUtil.split(passwordAndSalt, 0, password, encryptionMethod.getSalt());
                    return password;
                }
                catch (UnsupportedEncodingException uee) {
                    return credentials;
                }
            }
            case HASH_METHOD_CRYPT: {
                encryptionMethod.setSalt(new byte[2]);
                byte[] password = new byte[credentials.length - encryptionMethod.getSalt().length - algoLength];
                PasswordUtil.split(credentials, algoLength, encryptionMethod.getSalt(), password);
                return password;
            }
        }
        return credentials;
    }

    private static void split(byte[] all, int offset, byte[] left, byte[] right) {
        System.arraycopy(all, offset, left, 0, left.length);
        System.arraycopy(all, offset + left.length, right, 0, right.length);
    }

    private static void merge(byte[] all, byte[] left, byte[] right) {
        System.arraycopy(left, 0, all, 0, left.length);
        System.arraycopy(right, 0, all, left.length, right.length);
    }

    public static boolean isPwdExpired(String pwdChangedZtime, int pwdMaxAgeSec) {
        Date pwdChangeDate = DateUtils.getDate(pwdChangedZtime);
        long time = (long)pwdMaxAgeSec * 1000L;
        Date expiryDate = DateUtils.getDate(DateUtils.getGeneralizedTime(time += pwdChangeDate.getTime()));
        Date now = DateUtils.getDate(DateUtils.getGeneralizedTime());
        boolean expired = false;
        if (expiryDate.equals(now) || expiryDate.before(now)) {
            expired = true;
        }
        return expired;
    }

    public static void purgeFailureTimes(PasswordPolicyConfiguration config, Attribute pwdFailTimeAt) {
        long interval = config.getPwdFailureCountInterval();
        if (interval == 0L) {
            return;
        }
        interval *= 1000L;
        long currentTime = DateUtils.getDate(DateUtils.getGeneralizedTime()).getTime();
        Iterator itr = pwdFailTimeAt.iterator();
        while (itr.hasNext()) {
            Value value = (Value)itr.next();
            String failureTime = value.getString();
            long time = DateUtils.getDate(failureTime).getTime();
            if (currentTime < (time += interval)) continue;
            itr.remove();
        }
    }
}

