/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.mavibot.btree;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.directory.mavibot.btree.AbstractPage;
import org.apache.directory.mavibot.btree.BTree;
import org.apache.directory.mavibot.btree.BTreeConfiguration;
import org.apache.directory.mavibot.btree.BTreeFactory;
import org.apache.directory.mavibot.btree.BTreeTypeEnum;
import org.apache.directory.mavibot.btree.DuplicateKeyMemoryHolder;
import org.apache.directory.mavibot.btree.ElementHolder;
import org.apache.directory.mavibot.btree.Leaf;
import org.apache.directory.mavibot.btree.MemoryHolder;
import org.apache.directory.mavibot.btree.Node;
import org.apache.directory.mavibot.btree.Page;
import org.apache.directory.mavibot.btree.PageIO;
import org.apache.directory.mavibot.btree.ReferenceHolder;
import org.apache.directory.mavibot.btree.RevisionName;
import org.apache.directory.mavibot.btree.RevisionNameSerializer;
import org.apache.directory.mavibot.btree.exception.BTreeAlreadyManagedException;
import org.apache.directory.mavibot.btree.exception.EndOfFileExceededException;
import org.apache.directory.mavibot.btree.exception.KeyNotFoundException;
import org.apache.directory.mavibot.btree.serializer.ElementSerializer;
import org.apache.directory.mavibot.btree.serializer.IntSerializer;
import org.apache.directory.mavibot.btree.serializer.LongArraySerializer;
import org.apache.directory.mavibot.btree.serializer.LongSerializer;
import org.apache.directory.mavibot.btree.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RecordManager {
    protected static final Logger LOG = LoggerFactory.getLogger(RecordManager.class);
    protected static final Logger LOG_CHECK = LoggerFactory.getLogger("RM_CHECK");
    private File file;
    private FileChannel fileChannel;
    private int nbBtree;
    private long firstFreePage;
    private long lastFreePage;
    List<PageIO> freePages = new ArrayList<PageIO>();
    public AtomicLong nbFreedPages = new AtomicLong(0L);
    public AtomicLong nbCreatedPages = new AtomicLong(0L);
    public AtomicLong nbReusedPages = new AtomicLong(0L);
    private long endOfFileOffset;
    private BTree<RevisionName, long[]> copiedPageBTree;
    private BTree<RevisionName, Long> revisionBTree;
    private static final long NO_PAGE = -1L;
    private static final int NB_TREE_SIZE = 4;
    private static final int PAGE_SIZE = 4;
    private static final int DATA_SIZE = 4;
    private static final int LINK_SIZE = 8;
    private static final int BYTE_SIZE = 1;
    private static final int INT_SIZE = 4;
    private static final int LONG_SIZE = 8;
    private static final int FIRST_FREE_PAGE_SIZE = 8;
    private static final int LAST_FREE_PAGE_SIZE = 8;
    private static final int DEFAULT_PAGE_SIZE = 512;
    private static int HEADER_SIZE = 512;
    private static ByteBuffer HEADER_BUFFER;
    private int pageSize = 512;
    private Map<String, BTree<?, ?>> managedBTrees = new LinkedHashMap();
    private long lastAddedBTreeOffset = -1L;
    private static final String DEFAULT_FILE_NAME = "mavibot.db";
    private static final LongSerializer OFFSET_SERIALIZER;
    private static final String REVISION_BTREE_NAME = "_revisionBTree_";
    private static final String COPIED_PAGE_BTREE_NAME = "_copiedPageBTree_";
    private boolean keepRevisions;

    public RecordManager(String fileName) {
        this(fileName, 512);
    }

    public RecordManager(String fileName, int pageSize) {
        HEADER_BUFFER = ByteBuffer.allocate(pageSize);
        HEADER_SIZE = pageSize;
        File tmpFile = new File(fileName);
        boolean isNewFile = false;
        if (tmpFile.isDirectory()) {
            File mavibotFile = new File(tmpFile, DEFAULT_FILE_NAME);
            if (!mavibotFile.exists()) {
                try {
                    mavibotFile.createNewFile();
                    isNewFile = true;
                }
                catch (IOException ioe) {
                    LOG.error("Cannot create the file {}", (Object)mavibotFile.getName());
                    return;
                }
            }
            this.file = mavibotFile;
        } else {
            if (!tmpFile.exists() || tmpFile.length() == 0L) {
                isNewFile = true;
                try {
                    tmpFile.createNewFile();
                }
                catch (IOException ioe) {
                    LOG.error("Cannot create the file {}", (Object)tmpFile.getName());
                    return;
                }
            }
            this.file = tmpFile;
        }
        try {
            RandomAccessFile randomFile = new RandomAccessFile(this.file, "rw");
            this.fileChannel = randomFile.getChannel();
            this.endOfFileOffset = this.fileChannel.size();
            if (isNewFile) {
                this.pageSize = pageSize;
                this.initRecordManager();
            } else {
                this.loadRecordManager();
            }
        }
        catch (Exception e) {
            LOG.error("Error while initializing the RecordManager : {}", (Object)e.getMessage());
            LOG.error("", e);
            throw new RuntimeException(e);
        }
    }

    private void initRecordManager() throws IOException {
        this.nbBtree = 0;
        this.firstFreePage = -1L;
        this.lastFreePage = -1L;
        this.updateRecordManagerHeader();
        this.endOfFileOffset = this.fileChannel.size();
        this.copiedPageBTree = new BTree<RevisionName, long[]>(COPIED_PAGE_BTREE_NAME, new RevisionNameSerializer(), new LongArraySerializer());
        this.revisionBTree = new BTree<RevisionName, Long>(REVISION_BTREE_NAME, new RevisionNameSerializer(), new LongSerializer());
        try {
            this.manage(this.copiedPageBTree);
            this.manage(this.revisionBTree);
        }
        catch (BTreeAlreadyManagedException bTreeAlreadyManagedException) {
            // empty catch block
        }
    }

    private void loadRecordManager() throws IOException, ClassNotFoundException, IllegalAccessException, InstantiationException {
        if (this.fileChannel.size() != 0L) {
            ByteBuffer header = ByteBuffer.allocate(HEADER_SIZE);
            this.fileChannel.read(header);
            header.rewind();
            this.pageSize = header.getInt();
            this.nbBtree = header.getInt();
            this.firstFreePage = header.getLong();
            this.lastFreePage = header.getLong();
            long btreeOffset = HEADER_SIZE;
            PageIO[] pageIos = this.readPageIOs(HEADER_SIZE, Long.MAX_VALUE);
            this.copiedPageBTree = BTreeFactory.createBTree();
            this.copiedPageBTree.setBtreeOffset(btreeOffset);
            this.loadBTree(pageIos, this.copiedPageBTree);
            long nextBtreeOffset = this.copiedPageBTree.getNextBTreeOffset();
            pageIos = this.readPageIOs(nextBtreeOffset, Long.MAX_VALUE);
            this.revisionBTree = BTreeFactory.createBTree();
            this.revisionBTree.setBtreeOffset(nextBtreeOffset);
            this.loadBTree(pageIos, this.revisionBTree);
            nextBtreeOffset = this.revisionBTree.getNextBTreeOffset();
            for (int i = 2; i < this.nbBtree; ++i) {
                BTree btree = BTreeFactory.createBTree();
                btree.setBtreeOffset(nextBtreeOffset);
                this.lastAddedBTreeOffset = nextBtreeOffset;
                pageIos = this.readPageIOs(nextBtreeOffset, Long.MAX_VALUE);
                this.loadBTree(pageIos, btree);
                nextBtreeOffset = btree.getNextBTreeOffset();
                this.managedBTrees.put(btree.getName(), btree);
            }
            this.endOfFileOffset = this.fileChannel.size();
        }
    }

    private PageIO[] readPageIOs(long position, long limit) throws IOException, EndOfFileExceededException {
        long dataRead;
        LOG.debug("Read PageIOs at position {}", (Object)position);
        if (limit <= 0L) {
            limit = Long.MAX_VALUE;
        }
        PageIO firstPage = this.fetchPage(position);
        firstPage.setSize();
        ArrayList<PageIO> listPages = new ArrayList<PageIO>();
        listPages.add(firstPage);
        long nextPage = firstPage.getNextPage();
        if (dataRead < limit && nextPage != -1L) {
            for (dataRead = (long)(this.pageSize - 8 - 4); dataRead < limit; dataRead += (long)(this.pageSize - 8)) {
                PageIO page = this.fetchPage(nextPage);
                listPages.add(page);
                nextPage = page.getNextPage();
                if (nextPage != -1L) continue;
                page.setNextPage(-1L);
                break;
            }
        }
        LOG.debug("Nb of PageIOs read : {}", (Object)listPages.size());
        return listPages.toArray(new PageIO[0]);
    }

    private void loadBTree(PageIO[] pageIos, BTree<?, ?> btree) throws EndOfFileExceededException, IOException, ClassNotFoundException, IllegalAccessException, InstantiationException {
        long dataPos = 0L;
        long revision = this.readLong(pageIos, dataPos);
        BTreeFactory.setRevision(btree, revision);
        long nbElems = this.readLong(pageIos, dataPos += 8L);
        BTreeFactory.setNbElems(btree, nbElems);
        long rootPageOffset = this.readLong(pageIos, dataPos += 8L);
        BTreeFactory.setRootPageOffset(btree, rootPageOffset);
        long nextBTreeOffset = this.readLong(pageIos, dataPos += 8L);
        BTreeFactory.setNextBTreeOffset(btree, nextBTreeOffset);
        int btreePageSize = this.readInt(pageIos, dataPos += 8L);
        BTreeFactory.setPageSize(btree, btreePageSize);
        byte[] btreeNameBytes = this.readBytes(pageIos, dataPos += 4L);
        String btreeName = Strings.utf8ToString(btreeNameBytes);
        BTreeFactory.setName(btree, btreeName);
        byte[] keySerializerBytes = this.readBytes(pageIos, dataPos += (long)(4 + btreeNameBytes.length));
        String keySerializerFqcn = null;
        keySerializerFqcn = keySerializerBytes != null ? Strings.utf8ToString(keySerializerBytes) : "";
        BTreeFactory.setKeySerializer(btree, keySerializerFqcn);
        byte[] valueSerializerBytes = this.readBytes(pageIos, dataPos += (long)(4 + keySerializerBytes.length));
        String valueSerializerFqcn = null;
        valueSerializerFqcn = valueSerializerBytes != null ? Strings.utf8ToString(valueSerializerBytes) : "";
        BTreeFactory.setValueSerializer(btree, valueSerializerFqcn);
        int allowDuplicates = this.readInt(pageIos, dataPos += (long)(4 + valueSerializerBytes.length));
        btree.setAllowDuplicates(allowDuplicates != 0);
        dataPos += 4L;
        btree.init();
        PageIO[] rootPageIos = this.readPageIOs(rootPageOffset, Long.MAX_VALUE);
        Page btreeRoot = this.readPage(btree, rootPageIos);
        BTreeFactory.setRecordManager(btree, this);
        BTreeFactory.setRoot(btree, btreeRoot);
    }

    private Page readNode(BTree btree, long offset, long revision, int nbElems) throws IOException {
        Node node = BTreeFactory.createNode(btree, revision, nbElems);
        PageIO[] pageIos = this.readPageIOs(offset, Long.MAX_VALUE);
        return node;
    }

    public Page deserialize(BTree btree, long offset) throws EndOfFileExceededException, IOException {
        PageIO[] rootPageIos = this.readPageIOs(offset, Long.MAX_VALUE);
        Page page = this.readPage(btree, rootPageIos);
        return page;
    }

    private Page readPage(BTree btree, PageIO[] pageIos) throws IOException {
        long position = 0L;
        long revision = this.readLong(pageIos, position);
        int nbElems = this.readInt(pageIos, position += 8L);
        AbstractPage page = null;
        ByteBuffer byteBuffer = null;
        byte[] data = this.readBytes(pageIos, position += 4L);
        if (data != null) {
            byteBuffer = ByteBuffer.allocate(data.length);
            byteBuffer.put(data);
            byteBuffer.rewind();
        }
        if (nbElems >= 0) {
            page = BTreeFactory.createLeaf(btree, revision, nbElems);
            ((AbstractPage)page).setOffset(pageIos[0].getOffset());
            ((AbstractPage)page).setLastOffset(pageIos[pageIos.length - 1].getOffset());
            for (int i = 0; i < nbElems; ++i) {
                ElementHolder valueHolder;
                if (btree.isAllowDuplicates()) {
                    long value = OFFSET_SERIALIZER.deserialize(byteBuffer);
                    BTree dupValueContainer = this.loadDupsBTree(value);
                    valueHolder = new DuplicateKeyMemoryHolder(btree, dupValueContainer);
                } else {
                    Object value = btree.getValueSerializer().deserialize(byteBuffer);
                    valueHolder = new MemoryHolder(btree, value);
                }
                BTreeFactory.setValue(page, i, valueHolder);
                Object key = btree.getKeySerializer().deserialize(byteBuffer);
                BTreeFactory.setKey(page, i, key);
            }
        } else {
            int nodeNbElems = -nbElems;
            page = BTreeFactory.createNode(btree, revision, nodeNbElems);
            for (int i = 0; i < nodeNbElems; ++i) {
                long offset = OFFSET_SERIALIZER.deserialize(byteBuffer);
                long lastOffset = OFFSET_SERIALIZER.deserialize(byteBuffer);
                ReferenceHolder valueHolder = new ReferenceHolder(btree, null, offset, lastOffset);
                ((Node)page).setValue(i, valueHolder);
                Object key = btree.getKeySerializer().deserialize(byteBuffer);
                BTreeFactory.setKey(page, i, key);
            }
            long offset = OFFSET_SERIALIZER.deserialize(byteBuffer);
            long lastOffset = OFFSET_SERIALIZER.deserialize(byteBuffer);
            ReferenceHolder valueHolder = new ReferenceHolder(btree, null, offset, lastOffset);
            ((Node)page).setValue(nodeNbElems, valueHolder);
        }
        return page;
    }

    private byte[] readBytes(PageIO[] pageIos, long position) {
        int length = this.readInt(pageIos, position);
        int pageNb = this.computePageNb(position += 4L);
        int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
        ByteBuffer pageData = pageIos[pageNb].getData();
        int remaining = pageData.capacity() - pagePos;
        if (length == 0) {
            return null;
        }
        byte[] bytes = new byte[length];
        int bytesPos = 0;
        while (length > 0) {
            if (length <= remaining) {
                pageData.mark();
                pageData.position(pagePos);
                pageData.get(bytes, bytesPos, length);
                pageData.reset();
                return bytes;
            }
            pageData.mark();
            pageData.position(pagePos);
            pageData.get(bytes, bytesPos, remaining);
            pageData.reset();
            pagePos = 8;
            bytesPos += remaining;
            pageData = pageIos[++pageNb].getData();
            length -= remaining;
            remaining = pageData.capacity() - pagePos;
        }
        return bytes;
    }

    private int readInt(PageIO[] pageIos, long position) {
        int pageNb = this.computePageNb(position);
        int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
        ByteBuffer pageData = pageIos[pageNb].getData();
        int remaining = pageData.capacity() - pagePos;
        int value = 0;
        if (remaining >= 4) {
            value = pageData.getInt(pagePos);
        } else {
            value = 0;
            switch (remaining) {
                case 3: {
                    value += (pageData.get(pagePos + 2) & 0xFF) << 8;
                }
                case 2: {
                    value += (pageData.get(pagePos + 1) & 0xFF) << 16;
                }
                case 1: {
                    value += pageData.get(pagePos) << 24;
                }
            }
            pageData = pageIos[pageNb + 1].getData();
            pagePos = 8;
            switch (remaining) {
                case 1: {
                    value += (pageData.get(pagePos) & 0xFF) << 16;
                }
                case 2: {
                    value += (pageData.get(pagePos + 2 - remaining) & 0xFF) << 8;
                }
                case 3: {
                    value += pageData.get(pagePos + 3 - remaining) & 0xFF;
                }
            }
        }
        return value;
    }

    private byte readByte(PageIO[] pageIos, long position) {
        int pageNb = this.computePageNb(position);
        int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
        ByteBuffer pageData = pageIos[pageNb].getData();
        byte value = 0;
        value = pageData.get(pagePos);
        return value;
    }

    private long readLong(PageIO[] pageIos, long position) {
        int pageNb = this.computePageNb(position);
        int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
        ByteBuffer pageData = pageIos[pageNb].getData();
        int remaining = pageData.capacity() - pagePos;
        long value = 0L;
        if (remaining >= 8) {
            value = pageData.getLong(pagePos);
        } else {
            switch (remaining) {
                case 7: {
                    value += ((long)pageData.get(pagePos + 6) & 0xFFL) << 8;
                }
                case 6: {
                    value += ((long)pageData.get(pagePos + 5) & 0xFFL) << 16;
                }
                case 5: {
                    value += ((long)pageData.get(pagePos + 4) & 0xFFL) << 24;
                }
                case 4: {
                    value += ((long)pageData.get(pagePos + 3) & 0xFFL) << 32;
                }
                case 3: {
                    value += ((long)pageData.get(pagePos + 2) & 0xFFL) << 40;
                }
                case 2: {
                    value += ((long)pageData.get(pagePos + 1) & 0xFFL) << 48;
                }
                case 1: {
                    value += (long)pageData.get(pagePos) << 56;
                }
            }
            pageData = pageIos[pageNb + 1].getData();
            pagePos = 8;
            switch (remaining) {
                case 1: {
                    value += ((long)pageData.get(pagePos) & 0xFFL) << 48;
                }
                case 2: {
                    value += ((long)pageData.get(pagePos + 2 - remaining) & 0xFFL) << 40;
                }
                case 3: {
                    value += ((long)pageData.get(pagePos + 3 - remaining) & 0xFFL) << 32;
                }
                case 4: {
                    value += ((long)pageData.get(pagePos + 4 - remaining) & 0xFFL) << 24;
                }
                case 5: {
                    value += ((long)pageData.get(pagePos + 5 - remaining) & 0xFFL) << 16;
                }
                case 6: {
                    value += ((long)pageData.get(pagePos + 6 - remaining) & 0xFFL) << 8;
                }
                case 7: {
                    value += (long)pageData.get(pagePos + 7 - remaining) & 0xFFL;
                }
            }
        }
        return value;
    }

    public synchronized void manage(BTree<?, ?> btree) throws BTreeAlreadyManagedException, IOException {
        this.manage(btree, false);
    }

    public synchronized void manage(BTree<?, ?> btree, boolean internalTree) throws BTreeAlreadyManagedException, IOException {
        LOG.debug("Managing the btree {} which is an internam tree : {}", (Object)btree.getName(), (Object)internalTree);
        BTreeFactory.setRecordManager(btree, this);
        String name = btree.getName();
        if (this.managedBTrees.containsKey(name)) {
            LOG.error("There is already a BTree named '{}' managed by this recordManager", (Object)name);
            throw new BTreeAlreadyManagedException(name);
        }
        this.managedBTrees.put(name, btree);
        byte[] btreeNameBytes = Strings.getBytesUtf8(name);
        byte[] keySerializerBytes = Strings.getBytesUtf8(btree.getKeySerializerFQCN());
        byte[] valueSerializerBytes = Strings.getBytesUtf8(btree.getValueSerializerFQCN());
        int bufferSize = 4 + btreeNameBytes.length + 4 + keySerializerBytes.length + 4 + valueSerializerBytes.length + 4 + 8 + 8 + 8 + 8 + 4;
        PageIO[] pageIos = this.getFreePageIOs(bufferSize);
        long btreeOffset = pageIos[0].getOffset();
        btree.setBtreeOffset(btreeOffset);
        long position = 0L;
        position = this.store(position, btree.getRevision(), pageIos);
        position = this.store(position, btree.getNbElems(), pageIos);
        Page rootPage = BTreeFactory.getRoot(btree);
        PageIO[] rootPageIos = this.serializePage(btree, btree.getRevision(), rootPage);
        PageIO rootPageIo = rootPageIos[0];
        position = this.store(position, rootPageIo.getOffset(), pageIos);
        btree.setRootPageOffset(rootPageIo.getOffset());
        ((Leaf)rootPage).setOffset(rootPageIo.getOffset());
        position = this.store(position, -1L, pageIos);
        position = this.store(position, btree.getPageSize(), pageIos);
        position = this.store(position, btreeNameBytes, pageIos);
        position = this.store(position, keySerializerBytes, pageIos);
        position = this.store(position, valueSerializerBytes, pageIos);
        position = this.store(position, btree.isAllowDuplicates() ? 1 : 0, pageIos);
        LOG.debug("Flushing the newly managed '{}' btree header", (Object)btree.getName());
        this.flushPages(pageIos);
        LOG.debug("Flushing the newly managed '{}' btree rootpage", (Object)btree.getName());
        this.flushPages(rootPageIos);
        if (!internalTree) {
            ++this.nbBtree;
            if (this.lastAddedBTreeOffset != -1L) {
                pageIos = this.readPageIOs(this.lastAddedBTreeOffset, 32L);
                this.store(24L, btreeOffset, pageIos);
                LOG.debug("Updated the previous btree pointer on the added BTree {}", (Object)btree.getName());
                this.flushPages(pageIos);
            }
            this.lastAddedBTreeOffset = btreeOffset;
            this.updateRecordManagerHeader();
        }
        if (LOG_CHECK.isDebugEnabled()) {
            this.check();
        }
    }

    private PageIO[] serializePage(BTree btree, long revision, Page page) throws IOException {
        int nbElems = page.getNbElems();
        if (nbElems == 0) {
            PageIO[] pageIos = new PageIO[1];
            PageIO newPage = this.fetchNewPage();
            long position = 0L;
            position = this.store(position, revision, newPage);
            position = this.store(position, nbElems, newPage);
            newPage.setSize((int)position);
            pageIos[0] = newPage;
            return pageIos;
        }
        int nbBuffers = 3 + nbElems * 2;
        int dataSize = 0;
        int serializedSize = 0;
        if (page instanceof Node) {
            ++nbBuffers;
        }
        ArrayList<byte[]> serializedData = new ArrayList<byte[]>(nbBuffers);
        byte[] buffer = LongSerializer.serialize(revision);
        serializedData.add(buffer);
        serializedSize += buffer.length;
        int pageNbElems = nbElems;
        if (page instanceof Node) {
            pageNbElems = -nbElems;
        }
        buffer = IntSerializer.serialize(pageNbElems);
        serializedData.add(buffer);
        serializedSize += buffer.length;
        for (int pos = 0; pos < nbElems; ++pos) {
            if (page instanceof Node) {
                Page child = ((Node)page).getReference(pos);
                buffer = LongSerializer.serialize(child.getOffset());
                serializedData.add(buffer);
                dataSize += buffer.length;
                buffer = LongSerializer.serialize(child.getLastOffset());
                serializedData.add(buffer);
                dataSize += buffer.length;
            } else {
                DuplicateKeyMemoryHolder value;
                if (btree.isAllowDuplicates()) {
                    value = (DuplicateKeyMemoryHolder)((Leaf)page).getValue(pos);
                    long duplicateContainerOffset = ((BTree)value.getValue(btree)).getBtreeOffset();
                    buffer = LongSerializer.serialize(duplicateContainerOffset);
                } else {
                    value = ((Leaf)page).getValue(pos);
                    buffer = btree.getValueSerializer().serialize(value.getValue(btree));
                }
                serializedData.add(buffer);
                dataSize += buffer.length;
            }
            buffer = btree.getKeySerializer().serialize(page.getKey(pos));
            serializedData.add(buffer);
            dataSize += buffer.length;
        }
        if (page instanceof Node) {
            Page child = ((Node)page).getReference(nbElems);
            buffer = LongSerializer.serialize(child.getOffset());
            serializedData.add(buffer);
            dataSize += buffer.length;
            buffer = LongSerializer.serialize(child.getLastOffset());
            serializedData.add(buffer);
            dataSize += buffer.length;
        }
        buffer = IntSerializer.serialize(dataSize);
        serializedData.add(2, buffer);
        serializedSize += buffer.length;
        PageIO[] pageIos = this.getFreePageIOs(serializedSize += dataSize);
        long position = 0L;
        for (byte[] bytes : serializedData) {
            position = this.storeRaw(position, bytes, pageIos);
        }
        return pageIos;
    }

    private void updateRecordManagerHeader() throws IOException {
        HEADER_BUFFER.clear();
        HEADER_BUFFER.putInt(this.pageSize);
        HEADER_BUFFER.putInt(this.nbBtree);
        HEADER_BUFFER.putLong(this.firstFreePage);
        HEADER_BUFFER.putLong(this.lastFreePage);
        HEADER_BUFFER.limit(this.pageSize);
        HEADER_BUFFER.rewind();
        LOG.debug("Update RM header, FF : {}, LF : {}", (Object)this.firstFreePage, (Object)this.lastFreePage);
        this.fileChannel.write(HEADER_BUFFER, 0L);
    }

    void updateBtreeHeader(BTree btree, long rootPageOffset) throws EndOfFileExceededException, IOException {
        long offset = btree.getBtreeOffset();
        long headerSize = 24L;
        PageIO[] pageIos = this.readPageIOs(offset, headerSize);
        long position = 0L;
        position = this.store(position, btree.getRevision(), pageIos);
        position = this.store(position, btree.getNbElems(), pageIos);
        position = this.store(position, rootPageOffset, pageIos);
        if (LOG.isDebugEnabled()) {
            LOG.debug("-----> Flushing the '{}' BTreeHeader", (Object)btree.getName());
            LOG.debug("  revision : " + btree.getRevision() + ", NbElems : " + btree.getNbElems() + ", root offset : " + rootPageOffset);
        }
        this.flushPages(pageIos);
        if (LOG_CHECK.isDebugEnabled()) {
            this.check();
        }
    }

    private void flushPages(PageIO ... pageIos) throws IOException {
        for (PageIO pageIo : pageIos) {
            pageIo.getData().rewind();
            if (this.fileChannel.size() < pageIo.getOffset() + (long)this.pageSize) {
                LOG.debug("Adding a page at the end of the file");
                this.fileChannel.write(pageIo.getData(), this.fileChannel.size());
            } else {
                LOG.debug("Writing a page at position {}", (Object)pageIo.getOffset());
                this.fileChannel.write(pageIo.getData(), pageIo.getOffset());
            }
            pageIo.getData().rewind();
        }
    }

    private int computePageNb(long offset) {
        long pageNb = 0L;
        if ((offset -= (long)(this.pageSize - 8 - 4)) < 0L) {
            return (int)pageNb;
        }
        pageNb = 1L + offset / (long)(this.pageSize - 8);
        return (int)pageNb;
    }

    private long store(long position, byte[] bytes, PageIO ... pageIos) {
        if (bytes != null) {
            position = this.store(position, bytes.length, pageIos);
            int pageNb = this.computePageNb(position);
            ByteBuffer pageData = pageIos[pageNb].getData();
            int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
            int remaining = pageData.capacity() - pagePos;
            int nbStored = bytes.length;
            while (nbStored > 0) {
                if (remaining > nbStored) {
                    pageData.mark();
                    pageData.position(pagePos);
                    pageData.put(bytes, bytes.length - nbStored, nbStored);
                    pageData.reset();
                    nbStored = 0;
                    continue;
                }
                pageData.mark();
                pageData.position(pagePos);
                pageData.put(bytes, bytes.length - nbStored, remaining);
                pageData.reset();
                pageData = pageIos[++pageNb].getData();
                pagePos = 8;
                nbStored -= remaining;
                remaining = pageData.capacity() - pagePos;
            }
            position += (long)bytes.length;
        } else {
            position = this.store(position, 0, pageIos);
        }
        return position;
    }

    private long storeRaw(long position, byte[] bytes, PageIO ... pageIos) {
        if (bytes != null) {
            int pageNb = this.computePageNb(position);
            ByteBuffer pageData = pageIos[pageNb].getData();
            int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
            int remaining = pageData.capacity() - pagePos;
            int nbStored = bytes.length;
            while (nbStored > 0) {
                if (remaining > nbStored) {
                    pageData.mark();
                    pageData.position(pagePos);
                    pageData.put(bytes, bytes.length - nbStored, nbStored);
                    pageData.reset();
                    nbStored = 0;
                    continue;
                }
                pageData.mark();
                pageData.position(pagePos);
                pageData.put(bytes, bytes.length - nbStored, remaining);
                pageData.reset();
                if (++pageNb == pageIos.length) break;
                pageData = pageIos[pageNb].getData();
                pagePos = 8;
                nbStored -= remaining;
                remaining = pageData.capacity() - pagePos;
            }
            position += (long)bytes.length;
        } else {
            position = this.store(position, 0, pageIos);
        }
        return position;
    }

    private long store(long position, int value, PageIO ... pageIos) {
        int pageNb = this.computePageNb(position);
        int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
        ByteBuffer pageData = pageIos[pageNb].getData();
        int remaining = pageData.capacity() - pagePos;
        if (remaining < 4) {
            switch (remaining) {
                case 3: {
                    pageData.put(pagePos + 2, (byte)(value >>> 8));
                }
                case 2: {
                    pageData.put(pagePos + 1, (byte)(value >>> 16));
                }
                case 1: {
                    pageData.put(pagePos, (byte)(value >>> 24));
                }
            }
            pageData = pageIos[pageNb + 1].getData();
            pagePos = 8;
            switch (remaining) {
                case 1: {
                    pageData.put(pagePos, (byte)(value >>> 16));
                }
                case 2: {
                    pageData.put(pagePos + 2 - remaining, (byte)(value >>> 8));
                }
                case 3: {
                    pageData.put(pagePos + 3 - remaining, (byte)value);
                }
            }
        } else {
            pageData.putInt(pagePos, value);
        }
        return position += 4L;
    }

    private long store(long position, long value, PageIO ... pageIos) {
        int pageNb = this.computePageNb(position);
        int pagePos = (int)(position + (long)((pageNb + 1) * 8) + 4L) - pageNb * this.pageSize;
        ByteBuffer pageData = pageIos[pageNb].getData();
        int remaining = pageData.capacity() - pagePos;
        if (remaining < 8) {
            switch (remaining) {
                case 7: {
                    pageData.put(pagePos + 6, (byte)(value >>> 8));
                }
                case 6: {
                    pageData.put(pagePos + 5, (byte)(value >>> 16));
                }
                case 5: {
                    pageData.put(pagePos + 4, (byte)(value >>> 24));
                }
                case 4: {
                    pageData.put(pagePos + 3, (byte)(value >>> 32));
                }
                case 3: {
                    pageData.put(pagePos + 2, (byte)(value >>> 40));
                }
                case 2: {
                    pageData.put(pagePos + 1, (byte)(value >>> 48));
                }
                case 1: {
                    pageData.put(pagePos, (byte)(value >>> 56));
                }
            }
            pageData = pageIos[pageNb + 1].getData();
            pagePos = 8;
            switch (remaining) {
                case 1: {
                    pageData.put(pagePos, (byte)(value >>> 48));
                }
                case 2: {
                    pageData.put(pagePos + 2 - remaining, (byte)(value >>> 40));
                }
                case 3: {
                    pageData.put(pagePos + 3 - remaining, (byte)(value >>> 32));
                }
                case 4: {
                    pageData.put(pagePos + 4 - remaining, (byte)(value >>> 24));
                }
                case 5: {
                    pageData.put(pagePos + 5 - remaining, (byte)(value >>> 16));
                }
                case 6: {
                    pageData.put(pagePos + 6 - remaining, (byte)(value >>> 8));
                }
                case 7: {
                    pageData.put(pagePos + 7 - remaining, (byte)value);
                }
            }
        } else {
            pageData.putLong(pagePos, value);
        }
        return position += 8L;
    }

    ElementHolder writePage(BTree btree, Page newPage, long newRevision) throws IOException {
        PageIO[] pageIos = this.serializePage(btree, newRevision, newPage);
        LOG.debug("Write data for '{}' btree ", (Object)btree.getName());
        this.flushPages(pageIos);
        long offset = pageIos[0].getOffset();
        long lastOffset = pageIos[pageIos.length - 1].getOffset();
        ReferenceHolder valueHolder = new ReferenceHolder(btree, newPage, offset, lastOffset);
        if (LOG_CHECK.isDebugEnabled()) {
            this.check();
        }
        return valueHolder;
    }

    private int computeNbPages(int dataSize) {
        if (dataSize <= 0) {
            return 0;
        }
        int availableSize = this.pageSize - 8;
        int nbNeededPages = 1;
        if (dataSize > availableSize - 4) {
            int remainingSize = dataSize - (availableSize - 4);
            nbNeededPages += remainingSize / availableSize;
            int remain = remainingSize % availableSize;
            if (remain > 0) {
                ++nbNeededPages;
            }
        }
        return nbNeededPages;
    }

    private PageIO[] getFreePageIOs(int dataSize) throws IOException {
        if (dataSize == 0) {
            return new PageIO[0];
        }
        int nbNeededPages = this.computeNbPages(dataSize);
        PageIO[] pageIOs = new PageIO[nbNeededPages];
        pageIOs[0] = this.fetchNewPage();
        pageIOs[0].setSize(dataSize);
        for (int i = 1; i < nbNeededPages; ++i) {
            pageIOs[i] = this.fetchNewPage();
            pageIOs[i - 1].setNextPage(pageIOs[i].getOffset());
        }
        return pageIOs;
    }

    private PageIO fetchNewPage() throws IOException {
        if (this.firstFreePage == -1L) {
            this.nbCreatedPages.incrementAndGet();
            PageIO newPage = new PageIO(this.endOfFileOffset);
            this.endOfFileOffset += (long)this.pageSize;
            ByteBuffer data = ByteBuffer.allocateDirect(this.pageSize);
            newPage.setData(data);
            newPage.setNextPage(-1L);
            newPage.setSize(0);
            LOG.debug("Requiring a new page at offset {}", (Object)newPage.getOffset());
            return newPage;
        }
        this.nbReusedPages.incrementAndGet();
        PageIO pageIo = this.fetchPage(this.firstFreePage);
        this.firstFreePage = pageIo.getNextPage();
        ByteBuffer data = ByteBuffer.allocateDirect(this.pageSize);
        pageIo.setData(data);
        pageIo.setNextPage(-1L);
        pageIo.setSize(0);
        LOG.debug("Reused page at offset {}", (Object)pageIo.getOffset());
        if (this.firstFreePage == -1L) {
            this.lastFreePage = -1L;
        }
        this.updateRecordManagerHeader();
        return pageIo;
    }

    private PageIO fetchPage(long offset) throws IOException, EndOfFileExceededException {
        if (this.fileChannel.size() < offset + (long)this.pageSize) {
            throw new EndOfFileExceededException("We are fetching a page on " + offset + " when the file's size is " + this.fileChannel.size());
        }
        this.fileChannel.position(offset);
        ByteBuffer data = ByteBuffer.allocate(this.pageSize);
        this.fileChannel.read(data);
        data.rewind();
        PageIO readPage = new PageIO(offset);
        readPage.setData(data);
        return readPage;
    }

    public int getPageSize() {
        return this.pageSize;
    }

    public void setPageSize(int pageSize) {
        if (this.pageSize == -1) {
            this.pageSize = pageSize;
        }
    }

    public void close() throws IOException {
        for (BTree<?, ?> tree : this.managedBTrees.values()) {
            tree.close();
        }
        this.managedBTrees.clear();
        this.fileChannel.force(true);
        this.fileChannel.close();
    }

    public void dump() throws IOException {
        RandomAccessFile randomFile = new RandomAccessFile(this.file, "r");
        FileChannel fileChannel = randomFile.getChannel();
        ByteBuffer header = ByteBuffer.allocate(HEADER_SIZE);
        fileChannel.read(header);
        header.rewind();
        int pageSize = header.getInt();
        int nbBTree = header.getInt();
        long firstFreePage = header.getLong();
        long lastFreePage = header.getLong();
        if (LOG.isDebugEnabled()) {
            LOG.debug("RecordManager");
            LOG.debug("-------------");
            LOG.debug("  Header ");
            LOG.debug("    '{}'", (Object)Strings.dumpBytes(header.array()));
            LOG.debug("    page size : {}", (Object)pageSize);
            LOG.debug("    nbTree : {}", (Object)nbBTree);
            LOG.debug("    firstFreePage : {}", (Object)firstFreePage);
            LOG.debug("    lastFreePage : {}", (Object)lastFreePage);
        }
        long position = HEADER_SIZE;
        for (int i = 0; i < nbBTree; ++i) {
            PageIO[] pageIos;
            LOG.debug("  Btree[{}]", (Object)i);
            for (PageIO pageIo : pageIos = this.readPageIOs(position, Long.MAX_VALUE)) {
                LOG.debug("    {}", (Object)pageIo);
            }
        }
        randomFile.close();
    }

    public int getNbManagedTrees() {
        return this.nbBtree - 2;
    }

    public Set<String> getManagedTrees() {
        HashSet<String> btrees = new HashSet<String>(this.managedBTrees.keySet());
        btrees.remove(COPIED_PAGE_BTREE_NAME);
        btrees.remove(REVISION_BTREE_NAME);
        return btrees;
    }

    void storeRootPage(BTree btree, Page rootPage) throws IOException {
        if (!this.isKeepRevisions()) {
            return;
        }
        if (btree == this.copiedPageBTree || btree == this.revisionBTree) {
            return;
        }
        RevisionName revisionName = new RevisionName(rootPage.getRevision(), btree.getName());
        this.revisionBTree.insert(revisionName, rootPage.getOffset(), 0L);
        if (LOG_CHECK.isDebugEnabled()) {
            this.check();
        }
    }

    Page getRootPage(BTree btree, long revision) throws KeyNotFoundException, IOException {
        if (btree.getRevision() == revision) {
            return btree.rootPage;
        }
        RevisionName revisionName = new RevisionName(revision, btree.getName());
        long rootPageOffset = this.revisionBTree.get(revisionName);
        PageIO[] rootPageIos = this.readPageIOs(rootPageOffset, Long.MAX_VALUE);
        Page btreeRoot = this.readPage(btree, rootPageIos);
        return btreeRoot;
    }

    public BTree getManagedTree(String name) {
        return this.managedBTrees.get(name);
    }

    void addFreePages(BTree btree, List<Page> pages) throws EndOfFileExceededException, IOException {
        if (btree == this.copiedPageBTree || btree == this.revisionBTree) {
            return;
        }
        if (pages == null || pages.isEmpty()) {
            return;
        }
        if (!this.keepRevisions) {
            for (Page page : pages) {
                long firstOffset = page.getOffset();
                if (firstOffset == -1L) continue;
                long lastOffset = page.getLastOffset();
                if (this.firstFreePage == -1L) {
                    this.firstFreePage = firstOffset;
                    continue;
                }
                long offset = page.getLastOffset();
                if (offset == -1L) {
                    offset = page.getOffset();
                }
                PageIO pageIo = this.fetchPage(offset);
                pageIo.setNextPage(this.firstFreePage);
                LOG.debug("Flushing the first free page");
                this.flushPages(pageIo);
                this.firstFreePage = firstOffset;
            }
            this.updateRecordManagerHeader();
        } else {
            LOG.debug("We should not get there");
            for (Page p : pages) {
                this.addFreePage(btree, p);
            }
        }
    }

    private void addFreePage(BTree btree, Page freePage) {
        try {
            RevisionName revision = new RevisionName(freePage.getRevision(), btree.getName());
            long[] offsetArray = null;
            if (this.copiedPageBTree.hasKey(revision)) {
                offsetArray = this.copiedPageBTree.get(revision);
                long[] tmp = new long[offsetArray.length + 1];
                System.arraycopy(offsetArray, 0, tmp, 0, offsetArray.length);
                offsetArray = tmp;
            } else {
                offsetArray = new long[1];
            }
            offsetArray[offsetArray.length - 1] = freePage.getOffset();
            this.copiedPageBTree.insert(revision, offsetArray, 0L);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public boolean isKeepRevisions() {
        return this.keepRevisions;
    }

    public void setKeepRevisions(boolean keepRevisions) {
        this.keepRevisions = keepRevisions;
    }

    public BTree addBTree(String name, ElementSerializer<?> keySerializer, ElementSerializer<?> valueSerializer, boolean allowDuplicates) throws IOException, BTreeAlreadyManagedException {
        BTreeConfiguration config = new BTreeConfiguration();
        config.setName(name);
        config.setKeySerializer(keySerializer);
        config.setValueSerializer(valueSerializer);
        config.setAllowDuplicates(allowDuplicates);
        config.setType(BTreeTypeEnum.MANAGED);
        BTree btree = new BTree(config);
        this.manage(btree);
        if (LOG_CHECK.isDebugEnabled()) {
            this.check();
        }
        return btree;
    }

    private void setCheckedPage(long[] checkedPages, long offset, int pageSize) {
        long pageOffset = (offset - (long)HEADER_SIZE) / (long)pageSize;
        int index = (int)(pageOffset / 64L);
        long bits = checkedPages[index];
        long mask = 1L << (int)(pageOffset % 64L);
        if ((bits & mask) == 1L) {
            throw new RuntimeException("The page at : " + offset + " has already been checked");
        }
        int n = index;
        checkedPages[n] = checkedPages[n] | mask;
    }

    private void checkFreePages(long[] checkedPages, int pageSize, long firstFreePage, long lastFreePage) throws IOException {
        if (firstFreePage == -1L) {
            if (lastFreePage == -1L) {
                return;
            }
            throw new RuntimeException("Wrong last free page : " + lastFreePage);
        }
        if (lastFreePage != -1L) {
            throw new RuntimeException("Wrong last free page : " + lastFreePage);
        }
        long currentOffset = firstFreePage;
        long fileSize = this.fileChannel.size();
        while (currentOffset != -1L) {
            if (currentOffset > fileSize) {
                throw new RuntimeException("Wrong free page offset, above file size : " + currentOffset);
            }
            try {
                long newOffset;
                PageIO pageIo = this.fetchPage(currentOffset);
                if (currentOffset != pageIo.getOffset()) {
                    throw new RuntimeException("PageIO offset is incorrect : " + currentOffset + "-" + pageIo.getOffset());
                }
                this.setCheckedPage(checkedPages, currentOffset, pageSize);
                currentOffset = newOffset = pageIo.getNextPage();
            }
            catch (IOException ioe) {
                throw new RuntimeException("Cannot fetch page at : " + currentOffset);
            }
        }
    }

    private void checkRoot(long[] checkedPages, long offset, int pageSize, long nbBTreeElems, ElementSerializer keySerializer, ElementSerializer valueSerializer, boolean allowDuplicates) throws EndOfFileExceededException, IOException {
        PageIO[] rootPageIos = this.readPageIOs(offset, Long.MAX_VALUE);
        long position = 0L;
        long revision = this.readLong(rootPageIos, position);
        int nbElems = this.readInt(rootPageIos, position += 8L);
        ByteBuffer byteBuffer = null;
        byte[] data = this.readBytes(rootPageIos, position += 4L);
        if (data != null) {
            byteBuffer = ByteBuffer.allocate(data.length);
            byteBuffer.put(data);
            byteBuffer.rewind();
        }
        if (nbElems >= 0) {
            long pageOffset = rootPageIos[0].getOffset();
            if (pageOffset < 0L || pageOffset > this.fileChannel.size()) {
                throw new RuntimeException("The page offset is incorrect : " + pageOffset);
            }
            long pageLastOffset = rootPageIos[rootPageIos.length - 1].getOffset();
            if (pageLastOffset <= 0L || pageLastOffset > this.fileChannel.size()) {
                throw new RuntimeException("The page last offset is incorrect : " + pageLastOffset);
            }
            for (int i = 0; i < nbElems; ++i) {
                if (!allowDuplicates) {
                    valueSerializer.deserialize(byteBuffer);
                }
                keySerializer.deserialize(byteBuffer);
            }
        }
    }

    private long checkBTree(long[] checkedPages, PageIO[] pageIos, int pageSize, boolean isLast) throws EndOfFileExceededException, IOException, InstantiationException, IllegalAccessException, ClassNotFoundException {
        long dataPos = 0L;
        long revision = this.readLong(pageIos, dataPos);
        long nbElems = this.readLong(pageIos, dataPos += 8L);
        long rootPageOffset = this.readLong(pageIos, dataPos += 8L);
        if (rootPageOffset < 0L || rootPageOffset > this.fileChannel.size()) {
            throw new RuntimeException("The rootpage is incorrect : " + rootPageOffset);
        }
        long nextBTreeOffset = this.readLong(pageIos, dataPos += 8L);
        if (rootPageOffset < 0L && !isLast || nextBTreeOffset > this.fileChannel.size()) {
            throw new RuntimeException("The rootpage is incorrect : " + rootPageOffset);
        }
        int btreePageSize = this.readInt(pageIos, dataPos += 8L);
        if (btreePageSize < 2 || (btreePageSize & ~btreePageSize + 1) != btreePageSize) {
            throw new RuntimeException("The BTree page size is not a power of 2 : " + btreePageSize);
        }
        byte[] btreeNameBytes = this.readBytes(pageIos, dataPos += 4L);
        dataPos += 4L;
        String btreeName = Strings.utf8ToString(btreeNameBytes);
        byte[] keySerializerBytes = this.readBytes(pageIos, dataPos += (long)btreeNameBytes.length);
        String keySerializerFqcn = null;
        dataPos += 4L;
        if (keySerializerBytes != null) {
            dataPos += (long)keySerializerBytes.length;
            keySerializerFqcn = Strings.utf8ToString(keySerializerBytes);
        } else {
            keySerializerFqcn = "";
        }
        byte[] valueSerializerBytes = this.readBytes(pageIos, dataPos);
        String valueSerializerFqcn = null;
        dataPos += 4L;
        if (valueSerializerBytes != null) {
            dataPos += (long)valueSerializerBytes.length;
            valueSerializerFqcn = Strings.utf8ToString(valueSerializerBytes);
        } else {
            valueSerializerFqcn = "";
        }
        int allowDuplicates = this.readInt(pageIos, dataPos);
        dataPos += 4L;
        Class<?> valueSerializer = Class.forName(valueSerializerFqcn);
        Class<?> keySerializer = Class.forName(keySerializerFqcn);
        return nextBTreeOffset;
    }

    private void checkBTrees(long[] checkedPages, int pageSize, int nbBTrees) throws EndOfFileExceededException, IOException, InstantiationException, IllegalAccessException, ClassNotFoundException {
        long position = HEADER_SIZE;
        for (int i = 0; i < nbBTrees; ++i) {
            PageIO[] pageIos = this.readPageIOs(position, Long.MAX_VALUE);
            int pageNb = 0;
            for (PageIO currentPageIo : pageIos) {
                long nextPageOffset = currentPageIo.getNextPage();
                if (pageNb == pageIos.length - 1) {
                    if (nextPageOffset != -1L) {
                        throw new RuntimeException("The pointer to the next page is not valid, expected NO_PAGE");
                    }
                } else if (nextPageOffset == -1L) {
                    throw new RuntimeException("The pointer to the next page is not valid, NO_PAGE");
                }
                if (nextPageOffset != -1L && (nextPageOffset - (long)HEADER_SIZE) % (long)pageSize != 0L) {
                    throw new RuntimeException("The pointer to the next page is not valid");
                }
                this.setCheckedPage(checkedPages, currentPageIo.getOffset(), pageSize);
            }
            long nextBTree = this.checkBTree(checkedPages, pageIos, pageSize, i == nbBTrees - 1);
            if (nextBTree == -1L && i < nbBTrees - 1) {
                throw new RuntimeException("The pointer to the next BTree is incorrect");
            }
            position = nextBTree;
        }
    }

    private void check() {
        try {
            ByteBuffer header = ByteBuffer.allocate(HEADER_SIZE);
            long fileSize = this.fileChannel.size();
            if (fileSize < (long)HEADER_SIZE) {
                throw new RuntimeException("File size too small : " + fileSize);
            }
            this.fileChannel.read(header, 0L);
            header.flip();
            int pageSize = header.getInt();
            if (pageSize < 0 || pageSize < 32 || (pageSize & ~pageSize + 1) != pageSize) {
                throw new RuntimeException("Wrong page size : " + pageSize);
            }
            long nbPages = (fileSize - (long)HEADER_SIZE) / (long)pageSize;
            int nbBTrees = header.getInt();
            if (nbBTrees < 0) {
                throw new RuntimeException("Wrong nb trees : " + nbBTrees);
            }
            long firstFreePage = header.getLong();
            if (firstFreePage > fileSize) {
                throw new RuntimeException("First free page pointing after the end of the file : " + firstFreePage);
            }
            if (firstFreePage != -1L && (firstFreePage - (long)HEADER_SIZE) % (long)pageSize != 0L) {
                throw new RuntimeException("First free page not pointing to a correct offset : " + firstFreePage);
            }
            long lastFreePage = header.getLong();
            if (lastFreePage != -1L && (lastFreePage - (long)HEADER_SIZE) % (long)pageSize != 0L) {
                throw new RuntimeException("Invalid last free page : " + lastFreePage);
            }
            int nbPageBits = (int)(nbPages / 64L);
            long[] checkedPages = new long[nbPageBits + 1];
            this.checkFreePages(checkedPages, pageSize, firstFreePage, lastFreePage);
            this.checkBTrees(checkedPages, pageSize, nbBTrees);
        }
        catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Error : " + e.getMessage());
        }
    }

    BTree loadDupsBTree(long offset) {
        try {
            PageIO[] pageIos = this.readPageIOs(offset, Long.MAX_VALUE);
            BTree dupValueContainer = BTreeFactory.createBTree();
            dupValueContainer.setBtreeOffset(offset);
            this.loadBTree(pageIos, dupValueContainer);
            return dupValueContainer;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("RM free pages : [");
        if (this.firstFreePage != -1L) {
            long current = this.firstFreePage;
            boolean isFirst = true;
            while (current != -1L) {
                if (isFirst) {
                    isFirst = false;
                } else {
                    sb.append(", ");
                }
                try {
                    PageIO pageIo = this.fetchPage(current);
                    sb.append(pageIo.getOffset());
                    current = pageIo.getNextPage();
                }
                catch (EndOfFileExceededException e) {
                    e.printStackTrace();
                }
                catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        sb.append("]");
        return sb.toString();
    }

    static {
        OFFSET_SERIALIZER = new LongSerializer();
    }
}

