/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.rng.examples.stress;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Formatter;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.rng.UniformRandomProvider;
import org.apache.commons.rng.core.source64.RandomLongSource;
import org.apache.commons.rng.examples.stress.ApplicationException;
import org.apache.commons.rng.examples.stress.Hex;
import org.apache.commons.rng.examples.stress.ListCommand;
import org.apache.commons.rng.examples.stress.LogUtils;
import org.apache.commons.rng.examples.stress.ProcessUtils;
import org.apache.commons.rng.examples.stress.RNGUtils;
import org.apache.commons.rng.examples.stress.RngDataOutput;
import org.apache.commons.rng.examples.stress.StandardOptions;
import org.apache.commons.rng.examples.stress.StressTestData;
import org.apache.commons.rng.examples.stress.StressTestDataList;
import org.apache.commons.rng.simple.RandomSource;
import picocli.CommandLine;

@CommandLine.Command(name="stress", description={"Run repeat trials of random data generators using a provided test application.", "Data is transferred to the application sub-process via standard input."})
class StressTestCommand
implements Callable<Void> {
    private static final int ONE_THOUSAND = 1000;
    @CommandLine.Mixin
    private StandardOptions reusableOptions;
    @CommandLine.Parameters(index="0", description={"The stress test executable."})
    private File executable;
    @CommandLine.Parameters(index="1..*", description={"The arguments to pass to the executable."}, paramLabel="<argument>")
    private List<String> executableArguments = new ArrayList<String>();
    @CommandLine.Option(names={"--prefix"}, description={"Results file prefix (default: ${DEFAULT-VALUE})."})
    private File fileOutputPrefix = new File("test_");
    @CommandLine.Option(names={"--stop-file"}, description={"Stop file (default: <Results file prefix>.stop).", "When created it will prevent new tasks from starting but running tasks will complete."})
    private File stopFile;
    @CommandLine.Option(names={"-o", "--output-mode"}, description={"Output mode for existing files (default: ${DEFAULT-VALUE}).", "Valid values: ${COMPLETION-CANDIDATES}."})
    private OutputMode outputMode = OutputMode.ERROR;
    @CommandLine.Option(names={"-l", "--list"}, description={"List of random generators.", "The default list is all known generators."}, paramLabel="<genList>")
    private File generatorsListFile;
    @CommandLine.Option(names={"-t", "--trials"}, description={"The number of trials for each random generator.", "Used only for the default list (default: ${DEFAULT-VALUE})."})
    private int trials = 1;
    @CommandLine.Option(names={"--trial-offset"}, description={"Offset to add to the trial number for output files (default: ${DEFAULT-VALUE}).", "Use for parallel tests with the same output prefix."})
    private int trialOffset;
    @CommandLine.Option(names={"-p", "--processors"}, description={"Number of available processors (default: ${DEFAULT-VALUE}).", "Number of concurrent tasks = ceil(processors / threadsPerTask)", "threadsPerTask = applicationThreads + (ignoreJavaThread ? 0 : 1)"})
    private int processors = Math.max(1, Runtime.getRuntime().availableProcessors());
    @CommandLine.Option(names={"--ignore-java-thread"}, description={"Ignore the java RNG thread when computing concurrent tasks."})
    private boolean ignoreJavaThread;
    @CommandLine.Option(names={"--threads"}, description={"Number of threads to use for each application (default: ${DEFAULT-VALUE}).", "Total threads per task includes an optional java thread."})
    private int applicationThreads = 1;
    @CommandLine.Option(names={"--buffer-size"}, description={"Byte-buffer size for the transferred data (default: ${DEFAULT-VALUE})."})
    private int bufferSize = 8192;
    @CommandLine.Option(names={"-b", "--byte-order"}, description={"Byte-order of the transferred data (default: ${DEFAULT-VALUE}).", "Valid values: BIG_ENDIAN, LITTLE_ENDIAN."})
    private ByteOrder byteOrder = ByteOrder.nativeOrder();
    @CommandLine.Option(names={"-r", "--reverse-bits"}, description={"Reverse the bits in the data (default: ${DEFAULT-VALUE}).", "Note: Generators may fail tests for a reverse sequence when passing using the standard sequence."})
    private boolean reverseBits;
    @CommandLine.Option(names={"--high-bits"}, description={"Use the upper 32-bits from the 64-bit long output.", "Takes precedent over --low-bits."})
    private boolean longHighBits;
    @CommandLine.Option(names={"--low-bits"}, description={"Use the lower 32-bits from the 64-bit long output."})
    private boolean longLowBits;
    @CommandLine.Option(names={"--raw64"}, description={"Use 64-bit output (default is 32-bit).", "This requires a 64-bit testing application and native 64-bit generators.", "In 32-bit mode the output uses the upper then lower bits of 64-bit generators sequentially, each appropriately byte reversed for the platform."})
    private boolean raw64;
    @CommandLine.Option(names={"-x", "--hex-seed"}, description={"The hex-encoded random seed.", "Seed conversion for multi-byte primitives use little-endian format.", "Use to repeat tests. Not recommended for batch testing."})
    private String byteSeed;
    @CommandLine.Option(names={"--hashcode"}, description={"Combine the bits with a hashcode (default: ${DEFAULT-VALUE}).", "System.identityHashCode(new Object()) ^ rng.nextInt()."})
    private boolean xorHashCode;
    @CommandLine.Option(names={"--local-random"}, description={"Combine the bits with ThreadLocalRandom (default: ${DEFAULT-VALUE}).", "ThreadLocalRandom.current().nextInt() ^ rng.nextInt()."})
    private boolean xorThreadLocalRandom;
    @CommandLine.Option(names={"--xor-rng"}, description={"Combine the bits with a second generator.", "xorRng.nextInt() ^ rng.nextInt().", "Valid values: Any known RandomSource enum value."})
    private RandomSource xorRandomSource;
    @CommandLine.Option(names={"--dry-run"}, description={"Perform a dry run where the generators and output files are created but the stress test is not executed."})
    private boolean dryRun;
    private ReentrantLock stopFileLock = new ReentrantLock(false);
    private boolean stopFileExists;
    private long stopFileTimestamp;

    StressTestCommand() {
    }

    @Override
    public Void call() {
        LogUtils.setLogLevel(this.reusableOptions.logLevel);
        ProcessUtils.checkExecutable(this.executable);
        ProcessUtils.checkOutputDirectory(this.fileOutputPrefix);
        this.checkStopFileDoesNotExist();
        Iterable<StressTestData> stressTestData = this.createStressTestData();
        StressTestCommand.printStressTestData(stressTestData);
        this.runStressTest(stressTestData);
        return null;
    }

    private void checkStopFileDoesNotExist() {
        if (this.stopFile == null) {
            this.stopFile = new File(this.fileOutputPrefix + ".stop");
        }
        if (this.stopFile.exists()) {
            throw new ApplicationException("Stop file exists: " + this.stopFile);
        }
    }

    private boolean isStopFileExists() {
        this.stopFileLock.lock();
        try {
            long timestamp;
            if (!this.stopFileExists && (timestamp = System.currentTimeMillis()) > this.stopFileTimestamp) {
                this.checkStopFile(timestamp);
            }
            boolean bl = this.stopFileExists;
            return bl;
        }
        finally {
            this.stopFileLock.unlock();
        }
    }

    private void checkStopFile(long timestamp) {
        this.stopFileTimestamp = timestamp + TimeUnit.SECONDS.toMillis(2L);
        this.stopFileExists = this.stopFile.exists();
        if (this.stopFileExists) {
            LogUtils.info("Stop file detected: %s", this.stopFile);
            LogUtils.info("No further tasks will start");
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Iterable<StressTestData> createStressTestData() {
        if (this.generatorsListFile == null) {
            return new StressTestDataList("", this.trials);
        }
        try (BufferedReader reader = Files.newBufferedReader(this.generatorsListFile.toPath());){
            Iterable<StressTestData> iterable = ListCommand.readStressTestData(reader);
            return iterable;
        }
        catch (IOException ex) {
            throw new ApplicationException("Failed to read generators list: " + this.generatorsListFile, ex);
        }
    }

    private static void printStressTestData(Iterable<StressTestData> stressTestData) {
        if (!LogUtils.isLoggable(LogUtils.LogLevel.DEBUG)) {
            return;
        }
        try {
            StringBuilder sb = new StringBuilder("Testing generators").append(System.lineSeparator());
            ListCommand.writeStressTestData(sb, stressTestData);
            LogUtils.debug(sb.toString());
        }
        catch (IOException ex) {
            throw new ApplicationException("Failed to show list of generators", ex);
        }
    }

    private void runStressTest(Iterable<StressTestData> stressTestData) {
        List<String> command = ProcessUtils.buildSubProcessCommand(this.executable, this.executableArguments);
        LogUtils.info("Set-up stress test ...");
        String basePath = this.fileOutputPrefix.getAbsolutePath();
        this.checkExistingOutputFiles(basePath, stressTestData);
        int parallelTasks = this.getParallelTasks();
        ProgressTracker progressTracker = new ProgressTracker(parallelTasks);
        List<Runnable> tasks = this.createTasks(command, basePath, stressTestData, progressTracker);
        ExecutorService service = Executors.newFixedThreadPool(parallelTasks);
        LogUtils.info("Running stress test ...");
        LogUtils.info("Shutdown by creating stop file: %s", this.stopFile);
        progressTracker.setTotal(tasks.size());
        List<Future<?>> taskList = StressTestCommand.submitTasks(service, tasks);
        try {
            for (Future<?> f : taskList) {
                try {
                    f.get();
                }
                catch (ExecutionException ex) {
                    LogUtils.error(ex.getCause(), ex.getMessage());
                }
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new ApplicationException("Unexpected interruption: " + ex.getMessage(), ex);
        }
        finally {
            service.shutdown();
        }
        LogUtils.info("Finished stress test");
    }

    private void checkExistingOutputFiles(String basePath, Iterable<StressTestData> stressTestData) {
        if (this.outputMode == OutputMode.ERROR) {
            for (StressTestData testData : stressTestData) {
                for (int trial = 1; trial <= testData.getTrials(); ++trial) {
                    File output = this.createOutputFile(basePath, testData, trial);
                    if (!output.exists()) continue;
                    throw new ApplicationException(StressTestCommand.createExistingFileMessage(output));
                }
            }
        }
    }

    private File createOutputFile(String basePath, StressTestData testData, int trial) {
        return new File(String.format("%s%s_%d", basePath, testData.getId(), trial + this.trialOffset));
    }

    private static String createExistingFileMessage(File output) {
        return "Existing output file: " + output;
    }

    private int getParallelTasks() {
        int availableProcessors = Math.max(1, this.processors);
        int threadsPerTask = Math.max(1, this.applicationThreads + (this.ignoreJavaThread ? 0 : 1));
        int parallelTasks = (int)Math.ceil((double)availableProcessors / (double)threadsPerTask);
        LogUtils.debug("Parallel tasks = %d (%d / %d)", parallelTasks, availableProcessors, threadsPerTask);
        return parallelTasks;
    }

    private List<Runnable> createTasks(List<String> command, String basePath, Iterable<StressTestData> stressTestData, ProgressTracker progressTracker) {
        ArrayList<Runnable> tasks = new ArrayList<Runnable>();
        for (StressTestData testData : stressTestData) {
            for (int trial = 1; trial <= testData.getTrials(); ++trial) {
                File output = this.createOutputFile(basePath, testData, trial);
                if (output.exists()) {
                    if (this.outputMode == OutputMode.ERROR) {
                        throw new ApplicationException(StressTestCommand.createExistingFileMessage(output));
                    }
                    LogUtils.info("%s existing output file: %s", new Object[]{this.outputMode, output});
                    if (this.outputMode == OutputMode.SKIP) continue;
                }
                byte[] seed = this.createSeed(testData.getRandomSource());
                UniformRandomProvider rng = testData.createRNG(seed);
                if (this.longHighBits) {
                    rng = RNGUtils.createLongUpperBitsIntProvider(rng);
                } else if (this.longLowBits) {
                    rng = RNGUtils.createLongLowerBitsIntProvider(rng);
                }
                if (this.xorHashCode) {
                    rng = RNGUtils.createHashCodeProvider(rng);
                }
                if (this.xorThreadLocalRandom) {
                    rng = RNGUtils.createThreadLocalRandomProvider(rng);
                }
                if (this.xorRandomSource != null) {
                    rng = RNGUtils.createXorProvider((UniformRandomProvider)RandomSource.create((RandomSource)this.xorRandomSource), rng);
                }
                if (this.reverseBits) {
                    rng = RNGUtils.createReverseBitsProvider(rng);
                }
                StressTestTask r = new StressTestTask(testData.getRandomSource(), rng, seed, output, command, this, progressTracker);
                tasks.add(r);
            }
        }
        return tasks;
    }

    private byte[] createSeed(RandomSource randomSource) {
        if (this.byteSeed != null) {
            try {
                return Hex.decodeHex(this.byteSeed);
            }
            catch (IllegalArgumentException ex) {
                throw new ApplicationException("Invalid hex seed: " + ex.getMessage(), ex);
            }
        }
        return randomSource.createSeed();
    }

    private static List<Future<?>> submitTasks(ExecutorService service, List<Runnable> tasks) {
        ArrayList taskList = new ArrayList(tasks.size());
        tasks.forEach(r -> taskList.add(service.submit((Runnable)r)));
        return taskList;
    }

    private static class StressTestTask
    implements Runnable {
        private static final String C = "# ";
        private static final String N = System.lineSeparator();
        private static final String DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";
        private static final String[] SI_UNITS = new String[]{"B", "kB", "MB", "GB", "TB", "PB", "EB"};
        private static final long SI_UNIT_BASE = 1000L;
        private final RandomSource randomSource;
        private final UniformRandomProvider rng;
        private final byte[] seed;
        private final File output;
        private final List<String> command;
        private final StressTestCommand cmd;
        private final ProgressTracker progressTracker;
        private long bytesUsed;

        StressTestTask(RandomSource randomSource, UniformRandomProvider rng, byte[] seed, File output, List<String> command, StressTestCommand cmd, ProgressTracker progressTracker) {
            this.randomSource = randomSource;
            this.rng = rng;
            this.seed = seed;
            this.output = output;
            this.command = command;
            this.cmd = cmd;
            this.progressTracker = progressTracker;
        }

        @Override
        public void run() {
            if (this.cmd.isStopFileExists()) {
                return;
            }
            try {
                long millis;
                Object exitValue;
                this.printHeader();
                int taskId = this.progressTracker.submitTask();
                if (this.cmd.dryRun) {
                    exitValue = "N/A";
                    this.progressTracker.endTask(taskId);
                    millis = 0L;
                } else {
                    exitValue = this.runSubProcess();
                    millis = this.progressTracker.endTask(taskId);
                }
                this.printFooter(millis, exitValue);
            }
            catch (IOException ex) {
                throw new ApplicationException("Failed to run task: " + ex.getMessage(), ex);
            }
        }

        private Integer runSubProcess() throws IOException {
            ProcessBuilder builder = new ProcessBuilder(this.command);
            builder.redirectOutput(ProcessBuilder.Redirect.appendTo(this.output));
            builder.redirectErrorStream(true);
            Process testingProcess = builder.start();
            try {
                RngDataOutput sink = RNGUtils.createDataOutput(this.rng, this.cmd.raw64, testingProcess.getOutputStream(), this.cmd.bufferSize, this.cmd.byteOrder);
                Throwable throwable = null;
                try {
                    try {
                        while (true) {
                            sink.write(this.rng);
                            ++this.bytesUsed;
                        }
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                }
                catch (Throwable throwable3) {
                    if (sink != null) {
                        StressTestTask.$closeResource(throwable, sink);
                    }
                    throw throwable3;
                }
            }
            catch (IOException iOException) {
                this.bytesUsed *= (long)this.cmd.bufferSize;
                return ProcessUtils.getExitValue(testingProcess, TimeUnit.SECONDS.toMillis(60L));
            }
        }

        private void printHeader() throws IOException {
            StringBuilder sb = new StringBuilder(200);
            sb.append(C).append(N).append(C).append("RandomSource: ").append(this.randomSource.name()).append(N).append(C).append("RNG: ").append(this.rng.toString()).append(N).append(C).append("Seed: ").append(Hex.encodeHex(this.seed)).append(N).append(C).append(N).append(C).append("Java: ").append(System.getProperty("java.version")).append(N);
            StressTestTask.appendNameAndVersion(sb, "Runtime", "java.runtime.name", "java.runtime.version", new String[0]);
            StressTestTask.appendNameAndVersion(sb, "JVM", "java.vm.name", "java.vm.version", "java.vm.info");
            sb.append(C).append("OS: ").append(System.getProperty("os.name")).append(' ').append(System.getProperty("os.version")).append(' ').append(System.getProperty("os.arch")).append(N).append(C).append("Native byte-order: ").append(ByteOrder.nativeOrder()).append(N).append(C).append("Output byte-order: ").append(this.cmd.byteOrder).append(N);
            if (this.rng instanceof RandomLongSource) {
                sb.append(C).append("64-bit output: ").append(this.cmd.raw64).append(N);
            }
            sb.append(C).append(N).append(C).append("Analyzer: ");
            for (String s : this.command) {
                sb.append(s).append(' ');
            }
            sb.append(N).append(C).append(N);
            StressTestTask.appendDate(sb, "Start").append(C).append(N);
            StressTestTask.write(sb, this.output, this.cmd.outputMode == OutputMode.APPEND);
        }

        private void printFooter(long millis, Object exitValue) throws IOException {
            StringBuilder sb = new StringBuilder(200);
            sb.append(C).append(N);
            StressTestTask.appendDate(sb, "End").append(C).append(N);
            sb.append(C).append("Exit value: ").append(exitValue).append(N).append(C).append("Bytes used: ").append(this.bytesUsed).append(" >= 2^").append(StressTestTask.log2(this.bytesUsed)).append(" (").append(StressTestTask.bytesToString(this.bytesUsed)).append(')').append(N).append(C).append(N);
            double duration = (double)millis * 0.001 / 60.0;
            sb.append(C).append("Test duration: ").append(duration).append(" minutes").append(N).append(C).append(N);
            StressTestTask.write(sb, this.output, true);
        }

        private static void write(StringBuilder sb, File output, boolean append) throws IOException {
            try (BufferedWriter w = append ? Files.newBufferedWriter(output.toPath(), StandardOpenOption.APPEND) : Files.newBufferedWriter(output.toPath(), new OpenOption[0]);){
                w.write(sb.toString());
            }
        }

        private static StringBuilder appendNameAndVersion(StringBuilder sb, String prefix, String nameKey, String versionKey, String ... infoKeys) {
            StressTestTask.appendPrefix(sb, prefix).append(System.getProperty(nameKey, "?")).append(" (build ").append(System.getProperty(versionKey, "?"));
            for (String key : infoKeys) {
                String value = System.getProperty(key, "");
                if (value.isEmpty()) continue;
                sb.append(", ").append(value);
            }
            return sb.append(')').append(N);
        }

        private static StringBuilder appendDate(StringBuilder sb, String prefix) {
            SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT, Locale.US);
            return StressTestTask.appendPrefix(sb, prefix).append(dateFormat.format(new Date())).append(N);
        }

        private static StringBuilder appendPrefix(StringBuilder sb, String prefix) {
            return sb.append(C).append(prefix).append(": ");
        }

        static String bytesToString(long bytes) {
            if (bytes < 1000L) {
                return bytes + " " + SI_UNITS[0];
            }
            int exponent = (int)(Math.log(bytes) / Math.log(1000.0));
            String unit = SI_UNITS[exponent];
            return String.format(Locale.US, "%.1f %s", (double)bytes / Math.pow(1000.0, exponent), unit);
        }

        static int log2(long x) {
            return 63 - Long.numberOfLeadingZeros(x);
        }
    }

    static class ProgressTracker {
        private static final long PROGRESS_INTERVAL = 500L;
        private int total;
        private final int parallelTasks;
        private int taskId;
        private long[] startTimes;
        private long[] sortedDurations;
        private int completed;
        private long nextReportTimestamp;

        ProgressTracker(int parallelTasks) {
            this.parallelTasks = parallelTasks;
        }

        void setTotal(int total) {
            this.total = total;
            this.startTimes = new long[total];
            this.sortedDurations = new long[total];
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        int submitTask() {
            int id;
            ProgressTracker progressTracker = this;
            synchronized (progressTracker) {
                long current = System.currentTimeMillis();
                id = this.taskId++;
                this.startTimes[id] = current;
                this.reportProgress(current);
            }
            return id;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        long endTask(int id) {
            long duration;
            ProgressTracker progressTracker = this;
            synchronized (progressTracker) {
                long current = System.currentTimeMillis();
                duration = current - this.startTimes[id];
                this.sortedDurations[this.completed++] = duration;
                this.reportProgress(current);
            }
            return duration;
        }

        private void reportProgress(long current) {
            int pending = this.total - this.taskId;
            int running = this.taskId - this.completed;
            if (this.completed >= this.total || current >= this.nextReportTimestamp && (running == this.parallelTasks || pending == 0)) {
                this.nextReportTimestamp = current + 500L;
                StringBuilder sb = ProgressTracker.createStringBuilderWithTimestamp(current, pending, running, this.completed);
                try (Formatter formatter = new Formatter(sb);){
                    formatter.format(" (%.2f%%)", 100.0 * (double)this.completed / (double)this.total);
                    this.appendRemaining(sb, current, pending, running);
                    LogUtils.info(sb.toString());
                }
            }
        }

        private static StringBuilder createStringBuilderWithTimestamp(long current, int pending, int running, int completed) {
            StringBuilder sb = new StringBuilder(80);
            LocalDateTime time = LocalDateTime.ofInstant(Instant.ofEpochMilli(current), ZoneId.systemDefault());
            sb.append('[');
            ProgressTracker.append00(sb, time.getHour()).append(':');
            ProgressTracker.append00(sb, time.getMinute()).append(':');
            ProgressTracker.append00(sb, time.getSecond());
            return sb.append("] Pending ").append(pending).append(". Running ").append(running).append(". Completed ").append(completed);
        }

        private StringBuilder appendRemaining(StringBuilder sb, long current, int pending, int running) {
            long millis = this.getRemainingTime(current, pending, running);
            if (millis == 0L) {
                return sb;
            }
            sb.append(". Remaining = ");
            ProgressTracker.hms(sb, millis);
            return sb;
        }

        private long getRemainingTime(long current, int pending, int running) {
            long taskTime = this.getEstimatedTaskTime();
            if (taskTime == 0L) {
                return 0L;
            }
            int id = Math.max(0, this.taskId - 1);
            long millis = running == 0 ? 0L : ProgressTracker.getTimeRemaining(taskTime, current, this.startTimes[id]);
            int batches = pending / this.parallelTasks;
            millis += (long)batches * taskTime;
            int remainder = pending % this.parallelTasks;
            if (remainder != 0) {
                int nthOldest = Math.max(0, id - this.parallelTasks + remainder);
                millis += ProgressTracker.getTimeRemaining(taskTime, current, this.startTimes[nthOldest]);
            }
            return millis;
        }

        private long getEstimatedTaskTime() {
            Arrays.sort(this.sortedDurations, 0, this.completed);
            if (this.completed < 4) {
                return this.sortedDurations[this.completed / 2];
            }
            int upper = this.completed - 1;
            long halfMax = this.sortedDurations[upper] / 2L;
            int lower = 0;
            while (lower + 1 < upper) {
                int mid = lower + upper >>> 1;
                if (this.sortedDurations[mid] < halfMax) {
                    lower = mid;
                    continue;
                }
                upper = mid;
            }
            return this.sortedDurations[(upper + this.completed - 1) / 2];
        }

        private static long getTimeRemaining(long taskTime, long current, long startTime) {
            long endTime = startTime + taskTime;
            return Math.max(0L, endTime - current);
        }

        static StringBuilder hms(StringBuilder sb, long millis) {
            long hours = TimeUnit.MILLISECONDS.toHours(millis);
            long minutes = TimeUnit.MILLISECONDS.toMinutes(millis);
            long seconds = TimeUnit.MILLISECONDS.toSeconds(millis);
            seconds -= TimeUnit.MINUTES.toSeconds(minutes);
            ProgressTracker.append00(sb, hours).append(':');
            ProgressTracker.append00(sb, minutes -= TimeUnit.HOURS.toMinutes(hours)).append(':');
            return ProgressTracker.append00(sb, seconds);
        }

        static StringBuilder append00(StringBuilder sb, long ticks) {
            if (ticks == 0L) {
                sb.append("00");
            } else {
                if (ticks < 10L) {
                    sb.append('0');
                }
                sb.append(ticks);
            }
            return sb;
        }
    }

    static enum OutputMode {
        ERROR,
        SKIP,
        APPEND,
        OVERWRITE;

    }
}

