/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.collections4.functors;

import java.io.Serializable;
import java.util.Objects;

import org.apache.commons.collections4.Predicate;

/**
 * Predicate implementation that returns true if the input is an instanceof
 * the type stored in this predicate.
 *
 * @since 3.0
 */
public final class InstanceofPredicate extends AbstractPredicate<Object> implements Serializable {

    /** Serial version UID */
    private static final long serialVersionUID = -6682656911025165584L;

    /**
     * Creates the identity predicate.
     *
     * @param type  the type to check for, may not be null
     * @return the predicate
     * @throws NullPointerException if the class is null
     */
    public static Predicate<Object> instanceOfPredicate(final Class<?> type) {
        return new InstanceofPredicate(Objects.requireNonNull(type, "type"));
    }

    /** The type to compare to */
    private final Class<?> iType;

    /**
     * Constructor that performs no validation.
     * Use {@code instanceOfPredicate} if you want that.
     *
     * @param type  the type to check for
     */
    public InstanceofPredicate(final Class<?> type) {
        iType = type;
    }

    /**
     * Gets the type to compare to.
     *
     * @return the type
     * @since 3.1
     */
    public Class<?> getType() {
        return iType;
    }

    /**
     * Evaluates the predicate returning true if the input object is of the correct type.
     *
     * @param object  the input object
     * @return true if input is of stored type
     */
    @Override
    public boolean test(final Object object) {
        return iType.isInstance(object);
    }

}
