/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.streamhist;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.math.IntMath;
import java.lang.invoke.CallSite;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.utils.streamhist.HistogramDataConsumer;
import org.apache.cassandra.utils.streamhist.TombstoneHistogram;
import org.apache.commons.lang3.StringUtils;

public class StreamingTombstoneHistogramBuilder {
    private final DataHolder bin;
    private Spool spool;
    private final int roundSeconds;

    public StreamingTombstoneHistogramBuilder(int maxBinSize, int maxSpoolSize, int roundSeconds) {
        assert (maxBinSize > 0 && maxSpoolSize >= 0 && roundSeconds > 0) : "Invalid arguments: maxBinSize:" + maxBinSize + " maxSpoolSize:" + maxSpoolSize + " delta:" + roundSeconds;
        this.roundSeconds = roundSeconds;
        this.bin = new DataHolder(maxBinSize + 1, roundSeconds);
        this.spool = new Spool(maxSpoolSize);
    }

    public void update(long point) {
        this.update(point, 1);
    }

    public void update(long point, int value) {
        assert (this.spool != null) : "update is being called after releaseBuffers. This could be functionally okay, but this assertion is a canary to alert about unintended use before it is necessary.";
        point = StreamingTombstoneHistogramBuilder.ceilKey(point, this.roundSeconds);
        if (this.spool.capacity > 0) {
            if (!this.spool.tryAddOrAccumulate(point, value)) {
                this.flushHistogram();
                boolean success = this.spool.tryAddOrAccumulate(point, value);
                assert (success) : "Can not add value to spool";
            }
        } else {
            this.flushValue(point, value);
        }
    }

    public void flushHistogram() {
        Spool spool = this.spool;
        if (spool != null) {
            spool.forEach(this::flushValue);
            spool.clear();
        }
    }

    public void releaseBuffers() {
        this.flushHistogram();
        this.spool = null;
    }

    private void flushValue(long key, int spoolValue) {
        this.bin.addValue(key, spoolValue);
        if (this.bin.isFull()) {
            this.bin.mergeNearestPoints();
        }
    }

    public TombstoneHistogram build() {
        this.flushHistogram();
        return new TombstoneHistogram(this.bin);
    }

    private static long ceilKey(long point, int bucketSize) {
        long delta = point % (long)bucketSize;
        if (delta == 0L) {
            return point;
        }
        return StreamingTombstoneHistogramBuilder.saturatingCastToMaxDeletionTime(point + (long)bucketSize - delta);
    }

    public static int saturatingCastToInt(long value) {
        return (int)(value > Integer.MAX_VALUE ? Integer.MAX_VALUE : value);
    }

    public static long saturatingCastToLong(long value) {
        return value < 0L ? Long.MAX_VALUE : value;
    }

    public static long saturatingCastToMaxDeletionTime(long value) {
        return value < 0L || value > Cell.MAX_DELETION_TIME ? Cell.MAX_DELETION_TIME : value;
    }

    static class Spool {
        final long[] points;
        final int[] values;
        final int capacity;
        int size;

        Spool(int requestedCapacity) {
            if (requestedCapacity < 0) {
                throw new IllegalArgumentException("Illegal capacity " + requestedCapacity);
            }
            this.capacity = this.getPowerOfTwoCapacity(requestedCapacity);
            this.points = new long[this.capacity * 2];
            this.values = new int[this.capacity * 2];
            this.clear();
        }

        private int getPowerOfTwoCapacity(int requestedCapacity) {
            return requestedCapacity == 0 ? 0 : IntMath.pow((int)2, (int)IntMath.log2((int)requestedCapacity, (RoundingMode)RoundingMode.CEILING));
        }

        void clear() {
            Arrays.fill(this.points, -1L);
            this.size = 0;
        }

        boolean tryAddOrAccumulate(long point, int delta) {
            if (this.size > this.capacity) {
                return false;
            }
            int cell = this.capacity - 1 & this.hash(point);
            for (int attempt = 0; attempt < 100; ++attempt) {
                if (!this.tryCell(cell + attempt, point, delta)) continue;
                return true;
            }
            return false;
        }

        private int hash(long i) {
            long largePrime = 948701839L;
            return (int)(i * largePrime);
        }

        <E extends Exception> void forEach(HistogramDataConsumer<E> consumer) throws E {
            for (int i = 0; i < this.points.length; ++i) {
                if (this.points[i] == -1L) continue;
                consumer.consume(this.points[i], this.values[i]);
            }
        }

        private boolean tryCell(int cell, long point, int delta) {
            assert (cell >= 0 && point >= 0L && delta >= 0) : "Invalid arguments: cell:" + cell + " point:" + point + " delta:" + delta;
            if (this.points[cell %= this.points.length] == -1L) {
                this.points[cell] = point;
                this.values[cell] = delta;
                ++this.size;
                return true;
            }
            if (this.points[cell] == point) {
                this.values[cell] = StreamingTombstoneHistogramBuilder.saturatingCastToInt((long)this.values[cell] + (long)delta);
                return true;
            }
            return false;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append('[');
            for (int i = 0; i < this.points.length; ++i) {
                if (this.points[i] == -1L) continue;
                if (sb.length() > 1) {
                    sb.append(", ");
                }
                sb.append('[').append(this.points[i]).append(',').append(this.values[i]).append(']');
            }
            sb.append(']');
            return sb.toString();
        }
    }

    static class DataHolder {
        private static final long EMPTY = Long.MAX_VALUE;
        private final long[] points;
        private final int[] values;
        private final int roundSeconds;

        DataHolder(int maxCapacity, int roundSeconds) {
            this.points = new long[maxCapacity];
            this.values = new int[maxCapacity];
            Arrays.fill(this.points, Long.MAX_VALUE);
            Arrays.fill(this.values, 0);
            this.roundSeconds = roundSeconds;
        }

        DataHolder(DataHolder holder) {
            this.points = Arrays.copyOf(holder.points, holder.points.length);
            this.values = Arrays.copyOf(holder.values, holder.values.length);
            this.roundSeconds = holder.roundSeconds;
        }

        @VisibleForTesting
        int getValue(long point) {
            int index = Arrays.binarySearch(this.points, point);
            if (index < 0) {
                index = -index - 1;
            }
            if (index >= this.points.length) {
                return -1;
            }
            if (this.points[index] != point) {
                return -2;
            }
            return this.values[index];
        }

        boolean addValue(long point, int delta) {
            int index = Arrays.binarySearch(this.points, point);
            if (index < 0) {
                index = -index - 1;
                assert (index < this.points.length) : "No more space in array";
                if (this.points[index] != point) {
                    assert (this.points[this.points.length - 1] == Long.MAX_VALUE) : "No more space in array";
                    System.arraycopy(this.points, index, this.points, index + 1, this.points.length - index - 1);
                    System.arraycopy(this.values, index, this.values, index + 1, this.values.length - index - 1);
                    this.points[index] = point;
                    this.values[index] = StreamingTombstoneHistogramBuilder.saturatingCastToInt(delta);
                    return true;
                }
                this.values[index] = StreamingTombstoneHistogramBuilder.saturatingCastToInt((long)this.values[index] + (long)delta);
            } else {
                this.values[index] = StreamingTombstoneHistogramBuilder.saturatingCastToInt((long)this.values[index] + (long)delta);
            }
            return false;
        }

        @VisibleForTesting
        void mergeNearestPoints() {
            assert (this.isFull()) : "DataHolder must be full in order to merge two points";
            long[] smallestDifference = this.findPointPairWithSmallestDistance();
            long point1 = smallestDifference[0];
            long point2 = smallestDifference[1];
            int index = Arrays.binarySearch(this.points, point1);
            if (index < 0) {
                index = -index - 1;
                assert (index < this.points.length) : "Not found in array";
                assert (this.points[index] == point1) : "Not found in array";
            }
            long value1 = this.values[index];
            long value2 = this.values[index + 1];
            assert (this.points[index + 1] == point2) : "point2 should follow point1";
            long a = StreamingTombstoneHistogramBuilder.saturatingCastToLong(point1 * value1);
            long b = StreamingTombstoneHistogramBuilder.saturatingCastToLong(point2 * value2);
            long sum = StreamingTombstoneHistogramBuilder.saturatingCastToLong(value1 + value2);
            long newPoint = StreamingTombstoneHistogramBuilder.saturatingCastToMaxDeletionTime(StreamingTombstoneHistogramBuilder.saturatingCastToLong(a + b) / sum);
            newPoint = newPoint <= point1 ? StreamingTombstoneHistogramBuilder.saturatingCastToMaxDeletionTime(point1 + 1L) : newPoint;
            newPoint = Math.min(newPoint, point2);
            this.points[index] = newPoint = StreamingTombstoneHistogramBuilder.ceilKey(newPoint, this.roundSeconds);
            this.values[index] = StreamingTombstoneHistogramBuilder.saturatingCastToInt(sum);
            System.arraycopy(this.points, index + 2, this.points, index + 1, this.points.length - index - 2);
            System.arraycopy(this.values, index + 2, this.values, index + 1, this.values.length - index - 2);
            this.points[this.points.length - 1] = Long.MAX_VALUE;
            this.values[this.values.length - 1] = 0;
        }

        private long[] findPointPairWithSmallestDistance() {
            assert (this.isFull()) : "The DataHolder must be full in order to find the closest pair of points";
            long point1 = 0L;
            long point2 = Long.MAX_VALUE;
            for (int i = 0; i < this.points.length - 1; ++i) {
                long pointA = this.points[i];
                long pointB = this.points[i + 1];
                assert (pointB > pointA) : "DataHolder not sorted, p2(" + pointB + ") < p1(" + pointA + ") for " + this;
                if (point2 - point1 <= pointB - pointA) continue;
                point1 = pointA;
                point2 = pointB;
            }
            return new long[]{point1, point2};
        }

        public String toString() {
            ArrayList<CallSite> entries = new ArrayList<CallSite>();
            for (int i = 0; i < this.points.length && this.points[i] != Long.MAX_VALUE; ++i) {
                entries.add((CallSite)((Object)("[" + this.points[i] + "], [" + this.values[i] + "]")));
            }
            return StringUtils.join(entries, (String)",");
        }

        public boolean isFull() {
            return this.points[this.points.length - 1] != Long.MAX_VALUE;
        }

        public <E extends Exception> void forEach(HistogramDataConsumer<E> histogramDataConsumer) throws E {
            for (int i = 0; i < this.points.length && this.points[i] != Long.MAX_VALUE; ++i) {
                histogramDataConsumer.consume(this.points[i], this.values[i]);
            }
        }

        public int size() {
            int[] accumulator = new int[1];
            this.forEach((point, value) -> {
                accumulator[0] = accumulator[0] + 1;
            });
            return accumulator[0];
        }

        public double sum(int b) {
            long point;
            double sum = 0.0;
            for (int i = 0; i < this.points.length && (point = this.points[i]) != Long.MAX_VALUE; ++i) {
                int value = this.values[i];
                if (point > (long)b) {
                    if (i == 0) {
                        return 0.0;
                    }
                    long prevPoint = this.points[i - 1];
                    int prevValue = this.values[i - 1];
                    double weight = (double)((long)b - prevPoint) / (double)(point - prevPoint);
                    double mb = (double)prevValue + (double)(value - prevValue) * weight;
                    sum -= (double)prevValue;
                    sum += ((double)prevValue + mb) * weight / 2.0;
                    return sum += (double)prevValue / 2.0;
                }
                sum += (double)value;
            }
            return sum;
        }

        public int hashCode() {
            return Arrays.hashCode(this.points);
        }

        public boolean equals(Object o) {
            if (!(o instanceof DataHolder)) {
                return false;
            }
            DataHolder other = (DataHolder)o;
            if (this.size() != other.size()) {
                return false;
            }
            for (int i = 0; i < this.size(); ++i) {
                if (this.points[i] == other.points[i] && this.values[i] == other.values[i]) continue;
                return false;
            }
            return true;
        }
    }
}

