/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.utils.AbstractIterator;
import org.apache.cassandra.utils.AsymmetricOrdering;
import org.apache.cassandra.utils.Interval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IntervalTree<C extends Comparable<? super C>, D extends Comparable<? super D>, I extends Interval<C, D>>
implements Iterable<I> {
    public static final boolean EXPENSIVE_CHECKS = CassandraRelevantProperties.TEST_INTERVAL_TREE_EXPENSIVE_CHECKS.getBoolean();
    private static final Logger logger = LoggerFactory.getLogger(IntervalTree.class);
    public static final Interval[] EMPTY_ARRAY = new Interval[0];
    private static final IntervalTree EMPTY_TREE = new IntervalTree(null);
    private final IntervalNode head;
    private final I[] intervalsByMinOrder;
    private final I[] intervalsByMaxOrder;

    protected IntervalTree(Collection<I> intervals) {
        if (intervals == null || intervals.isEmpty()) {
            this.head = null;
            this.intervalsByMaxOrder = EMPTY_ARRAY;
            this.intervalsByMinOrder = EMPTY_ARRAY;
        } else if (intervals.size() == 1) {
            this.intervalsByMaxOrder = new Interval[]{(Interval)intervals.iterator().next()};
            this.intervalsByMinOrder = this.intervalsByMaxOrder;
            this.head = new IntervalNode(intervals);
        } else {
            this.intervalsByMinOrder = intervals.toArray(EMPTY_ARRAY);
            Arrays.sort(this.intervalsByMinOrder, Interval.minOrdering());
            this.intervalsByMaxOrder = intervals.toArray(EMPTY_ARRAY);
            Arrays.sort(this.intervalsByMaxOrder, Interval.maxOrdering());
            this.head = new IntervalNode(Arrays.asList(this.intervalsByMinOrder), Arrays.asList(this.intervalsByMaxOrder));
        }
    }

    protected IntervalTree(I[] minSortedIntervals, I[] maxSortedIntervals) {
        if (minSortedIntervals == null || minSortedIntervals.length == 0) {
            this.head = null;
            this.intervalsByMaxOrder = EMPTY_ARRAY;
            this.intervalsByMinOrder = EMPTY_ARRAY;
        } else if (minSortedIntervals.length == 1) {
            this.intervalsByMaxOrder = minSortedIntervals;
            this.intervalsByMinOrder = minSortedIntervals;
            List<I> intervals = Collections.singletonList(minSortedIntervals[0]);
            this.head = new IntervalNode(intervals, intervals);
        } else {
            this.intervalsByMinOrder = minSortedIntervals;
            this.intervalsByMaxOrder = maxSortedIntervals;
            this.head = new IntervalNode(Arrays.asList(minSortedIntervals), Arrays.asList(maxSortedIntervals));
        }
    }

    protected IntervalTree<C, D, I> create(I[] minOrder, I[] maxOrder) {
        return new IntervalTree(minOrder, maxOrder);
    }

    public static <C extends Comparable<? super C>, D extends Comparable<? super D>, I extends Interval<C, D>> IntervalTree<C, D, I> build(Collection<I> intervals) {
        if (intervals == null || intervals.isEmpty()) {
            return IntervalTree.emptyTree();
        }
        return new IntervalTree<C, D, I>(intervals);
    }

    public static <C extends Comparable<? super C>, D extends Comparable<? super D>, I extends Interval<C, D>> IntervalTree<C, D, I> emptyTree() {
        return EMPTY_TREE;
    }

    public int intervalCount() {
        return this.intervalsByMinOrder.length;
    }

    public boolean isEmpty() {
        return this.head == null;
    }

    public C max() {
        if (this.head == null) {
            throw new IllegalStateException();
        }
        return this.head.high;
    }

    public C min() {
        if (this.head == null) {
            throw new IllegalStateException();
        }
        return this.head.low;
    }

    public List<D> search(Interval<C, D> searchInterval) {
        if (this.head == null) {
            return Collections.emptyList();
        }
        ArrayList results = new ArrayList();
        this.head.searchInternal(searchInterval, results);
        return results;
    }

    public List<D> search(C point) {
        return this.search((C)Interval.create(point, point, null));
    }

    public IntervalTree<C, D, I> update(I[] removals, I[] additions) {
        if (removals == null) {
            removals = EMPTY_ARRAY;
        }
        if (additions == null) {
            additions = EMPTY_ARRAY;
        }
        if (removals.length == 0 && additions.length == 0) {
            return this;
        }
        Arrays.sort(removals, Interval.minOrdering());
        Arrays.sort(additions, Interval.minOrdering());
        for (int i = 1; i < additions.length; ++i) {
            Preconditions.checkState((Interval.minOrdering().compare(additions[i], additions[i - 1]) != 0 ? 1 : 0) != 0, (String)"Duplicate interval in additions %s", additions[i]);
        }
        Interval[] newByMin = this.buildUpdatedArray((Interval[])this.intervalsByMinOrder, (Interval[])removals, (Interval[])additions, Interval.minOrdering());
        Arrays.sort(removals, Interval.maxOrdering());
        Arrays.sort(additions, Interval.maxOrdering());
        Interval[] newByMax = this.buildUpdatedArray((Interval[])this.intervalsByMaxOrder, (Interval[])removals, (Interval[])additions, Interval.maxOrdering());
        return this.create(newByMin, newByMax);
    }

    private I[] buildUpdatedArray(I[] existingSorted, I[] removalsSorted, I[] additionsSorted, AsymmetricOrdering<Interval<C, D>, C> cmp) {
        int finalSize = existingSorted.length + additionsSorted.length - removalsSorted.length;
        Interval[] result = new Interval[finalSize];
        int removalsIndex = 0;
        int additionsIndex = 0;
        int resultIndex = 0;
        for (int existingIndex = 0; existingIndex < existingSorted.length; ++existingIndex) {
            int c;
            I currentExisting = existingSorted[existingIndex];
            while (removalsIndex < removalsSorted.length && (c = cmp.compare(removalsSorted[removalsIndex], currentExisting)) <= 0) {
                if (c < 0) {
                    throw new IllegalStateException("Removal interval not found in the existing tree: " + removalsSorted[removalsIndex]);
                }
                ++removalsIndex;
                if (++existingIndex >= existingSorted.length) break;
                currentExisting = existingSorted[existingIndex];
            }
            if (existingIndex >= existingSorted.length) break;
            while (additionsIndex < additionsSorted.length) {
                int additionCmp = cmp.compare(additionsSorted[additionsIndex], currentExisting);
                if (additionCmp == 0) {
                    throw new IllegalStateException("Attempting to add duplicate interval: " + additionsSorted[additionsIndex]);
                }
                if (additionCmp >= 0) break;
                result[resultIndex++] = additionsSorted[additionsIndex++];
            }
            result[resultIndex++] = currentExisting;
        }
        if (removalsIndex < removalsSorted.length) {
            throw new IllegalStateException("Removal interval not found in the existing tree: " + removalsSorted[removalsIndex]);
        }
        while (additionsIndex < additionsSorted.length) {
            result[resultIndex++] = additionsSorted[additionsIndex++];
        }
        if (EXPENSIVE_CHECKS && result.length > 1) {
            for (int i = 1; i < result.length; ++i) {
                Preconditions.checkState((cmp.compare(result[i - 1], result[i]) < 0 ? 1 : 0) != 0, (String)"%s and %s out of order", (Object)result[i - 1], (Object)result[i]);
            }
        }
        return result;
    }

    @Override
    public Iterator<I> iterator() {
        if (this.head == null) {
            return Collections.emptyIterator();
        }
        return new TreeIterator(this.head);
    }

    public String toString() {
        return "<" + Joiner.on((String)", ").join(Iterables.limit((Iterable)this, (int)100)) + ">";
    }

    public boolean equals(Object o) {
        if (!(o instanceof IntervalTree)) {
            return false;
        }
        IntervalTree that = (IntervalTree)o;
        return Iterators.elementsEqual(this.iterator(), that.iterator());
    }

    public final int hashCode() {
        int result = 0;
        for (Interval interval : this) {
            result = 31 * result + interval.hashCode();
        }
        return result;
    }

    private class TreeIterator
    extends AbstractIterator<I> {
        private final Deque<IntervalNode> stack = new ArrayDeque<IntervalNode>();
        private Iterator<I> current;

        TreeIterator(IntervalNode node) {
            this.gotoMinOf(node);
        }

        @Override
        protected I computeNext() {
            while (this.current == null || !this.current.hasNext()) {
                IntervalNode node = this.stack.pollFirst();
                if (node == null) {
                    return (Interval)this.endOfData();
                }
                this.current = node.intersectsLeft.iterator();
                this.gotoMinOf(node.right);
            }
            return (Interval)this.current.next();
        }

        private void gotoMinOf(IntervalNode node) {
            while (node != null) {
                this.stack.offerFirst(node);
                node = node.left;
            }
        }
    }

    protected class IntervalNode {
        final C center;
        final C low;
        final C high;
        final List<I> intersectsLeft;
        final List<I> intersectsRight;
        final IntervalNode left;
        final IntervalNode right;

        public IntervalNode(Collection<I> toBisect) {
            assert (toBisect.size() == 1);
            Interval interval = (Interval)toBisect.iterator().next();
            this.low = (Comparable)interval.min;
            this.center = (Comparable)interval.max;
            this.high = (Comparable)interval.max;
            List<Interval> l = Collections.singletonList(interval);
            this.intersectsLeft = l;
            this.intersectsRight = l;
            this.left = null;
            this.right = null;
        }

        public IntervalNode(List<I> minOrder, List<I> maxOrder) {
            assert (!minOrder.isEmpty());
            logger.trace("Creating IntervalNode from {}", minOrder);
            if (minOrder.size() == 1) {
                Interval interval = (Interval)minOrder.iterator().next();
                this.low = (Comparable)interval.min;
                this.center = (Comparable)interval.max;
                this.high = (Comparable)interval.max;
                List<Interval> l = Collections.singletonList(interval);
                this.intersectsLeft = l;
                this.intersectsRight = l;
                this.left = null;
                this.right = null;
                return;
            }
            this.low = (Comparable)((Interval)minOrder.get((int)0)).min;
            this.high = (Comparable)((Interval)maxOrder.get((int)(maxOrder.size() - 1))).max;
            int i = 0;
            int j = 0;
            for (int count = 0; count < minOrder.size(); ++count) {
                if (i < minOrder.size() && (j >= maxOrder.size() || ((Comparable)((Interval)minOrder.get((int)i)).min).compareTo(((Interval)maxOrder.get((int)j)).max) <= 0)) {
                    ++i;
                    continue;
                }
                ++j;
            }
            this.center = i < minOrder.size() && (j >= maxOrder.size() || ((Comparable)((Interval)minOrder.get((int)i)).min).compareTo(((Interval)maxOrder.get((int)j)).max) < 0) ? (Comparable)((Interval)minOrder.get((int)i)).min : (Comparable)((Interval)maxOrder.get((int)j)).max;
            if (EXPENSIVE_CHECKS) {
                ArrayList<Comparable> allEndpoints = new ArrayList<Comparable>(minOrder.size() * 2);
                for (Interval interval : minOrder) {
                    allEndpoints.add((Comparable)interval.min);
                    allEndpoints.add((Comparable)interval.max);
                }
                Collections.sort(allEndpoints);
                Comparable expectedCenter = (Comparable)allEndpoints.get(minOrder.size());
                Preconditions.checkState((boolean)expectedCenter.equals(this.center));
            }
            int initialIntersectionSize = i - j + 1;
            this.intersectsLeft = new ArrayList(initialIntersectionSize);
            this.intersectsRight = new ArrayList(initialIntersectionSize);
            int initialChildSize = Math.min(i, j);
            ArrayList<Interval> leftSegmentMinOrder = new ArrayList<Interval>(initialChildSize);
            ArrayList<Interval> leftSegmentMaxOrder = new ArrayList<Interval>(initialChildSize);
            ArrayList<Interval> rightSegmentMinOrder = new ArrayList<Interval>(initialChildSize);
            ArrayList<Interval> rightSegmentMaxOrder = new ArrayList<Interval>(initialChildSize);
            for (Interval candidate : minOrder) {
                if (((Comparable)candidate.max).compareTo(this.center) < 0) {
                    leftSegmentMinOrder.add(candidate);
                    continue;
                }
                if (((Comparable)candidate.min).compareTo(this.center) > 0) {
                    rightSegmentMinOrder.add(candidate);
                    continue;
                }
                this.intersectsLeft.add(candidate);
            }
            for (Interval candidate : maxOrder) {
                if (((Comparable)candidate.max).compareTo(this.center) < 0) {
                    leftSegmentMaxOrder.add(candidate);
                    continue;
                }
                if (((Comparable)candidate.min).compareTo(this.center) > 0) {
                    rightSegmentMaxOrder.add(candidate);
                    continue;
                }
                this.intersectsRight.add(candidate);
            }
            this.left = leftSegmentMinOrder.isEmpty() ? null : new IntervalNode(leftSegmentMinOrder, leftSegmentMaxOrder);
            IntervalNode intervalNode = this.right = rightSegmentMinOrder.isEmpty() ? null : new IntervalNode(rightSegmentMinOrder, rightSegmentMaxOrder);
            assert (this.intersectsLeft.size() == this.intersectsRight.size());
            assert (this.intersectsLeft.size() + leftSegmentMinOrder.size() + rightSegmentMinOrder.size() == minOrder.size()) : "intersects (" + String.valueOf(this.intersectsLeft.size()) + ") + leftSegment (" + String.valueOf(leftSegmentMinOrder.size()) + ") + rightSegment (" + String.valueOf(rightSegmentMinOrder.size()) + ") != toBisect (" + String.valueOf(minOrder.size()) + ")";
        }

        void searchInternal(Interval<C, D> searchInterval, List<D> results) {
            if (this.center.compareTo(searchInterval.min) < 0) {
                int i = Interval.maxOrdering().binarySearchAsymmetric(this.intersectsRight, (Comparable)searchInterval.min, AsymmetricOrdering.Op.CEIL);
                if (i == this.intersectsRight.size() && this.high.compareTo(searchInterval.min) < 0) {
                    return;
                }
                while (i < this.intersectsRight.size()) {
                    results.add((Comparable)((Interval)this.intersectsRight.get((int)i++)).data);
                }
                if (this.right != null) {
                    this.right.searchInternal(searchInterval, results);
                }
            } else if (this.center.compareTo(searchInterval.max) > 0) {
                int j = Interval.minOrdering().binarySearchAsymmetric(this.intersectsLeft, (Comparable)searchInterval.max, AsymmetricOrdering.Op.HIGHER);
                if (j == 0 && this.low.compareTo(searchInterval.max) > 0) {
                    return;
                }
                for (int i = 0; i < j; ++i) {
                    results.add((Comparable)((Interval)this.intersectsLeft.get((int)i)).data);
                }
                if (this.left != null) {
                    this.left.searchInternal(searchInterval, results);
                }
            } else {
                for (Interval interval : this.intersectsLeft) {
                    results.add((Comparable)interval.data);
                }
                if (this.left != null) {
                    this.left.searchInternal(searchInterval, results);
                }
                if (this.right != null) {
                    this.right.searchInternal(searchInterval, results);
                }
            }
        }
    }
}

