/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction.unified;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.UnifiedCompactionStrategy;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.MonotonicClock;
import org.apache.cassandra.utils.Overlaps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Controller {
    protected static final Logger logger = LoggerFactory.getLogger(Controller.class);
    static final String SCALING_PARAMETERS_OPTION = "scaling_parameters";
    private static final String DEFAULT_SCALING_PARAMETERS = CassandraRelevantProperties.UCS_SCALING_PARAMETER.getString();
    static final String MIN_SSTABLE_SIZE_OPTION = "min_sstable_size";
    private static final String DEFAULT_MIN_SSTABLE_SIZE = CassandraRelevantProperties.UCS_MIN_SSTABLE_SIZE.getString();
    static final String FLUSH_SIZE_OVERRIDE_OPTION = "flush_size_override";
    static final String BASE_SHARD_COUNT_OPTION = "base_shard_count";
    public static final int DEFAULT_BASE_SHARD_COUNT = CassandraRelevantProperties.UCS_BASE_SHARD_COUNT.getInt();
    static final String TARGET_SSTABLE_SIZE_OPTION = "target_sstable_size";
    public static final long DEFAULT_TARGET_SSTABLE_SIZE = CassandraRelevantProperties.UCS_TARGET_SSTABLE_SIZE.getSizeInBytes();
    static final long MIN_TARGET_SSTABLE_SIZE = 0x100000L;
    static final String SSTABLE_GROWTH_OPTION = "sstable_growth";
    private static final double DEFAULT_SSTABLE_GROWTH = CassandraRelevantProperties.UCS_SSTABLE_GROWTH.getDouble();
    static final double DEFAULT_SURVIVAL_FACTOR = CassandraRelevantProperties.UCS_SURVIVAL_FACTOR.getDouble();
    static final double[] DEFAULT_SURVIVAL_FACTORS = new double[]{DEFAULT_SURVIVAL_FACTOR};
    static final String MAX_SSTABLES_TO_COMPACT_OPTION = "max_sstables_to_compact";
    static final String ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION_OPTION = "unsafe_aggressive_sstable_expiration";
    static final boolean ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION = CassandraRelevantProperties.ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION.getBoolean();
    static final boolean DEFAULT_ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION = false;
    static final int DEFAULT_EXPIRED_SSTABLE_CHECK_FREQUENCY_SECONDS = 600;
    static final String EXPIRED_SSTABLE_CHECK_FREQUENCY_SECONDS_OPTION = "expired_sstable_check_frequency_seconds";
    static final int MAX_SHARD_SHIFT = 20;
    static final double MAX_SHARD_SPLIT = Math.scalb(1.0f, 20);
    static final String OVERLAP_INCLUSION_METHOD_OPTION = "overlap_inclusion_method";
    static final Overlaps.InclusionMethod DEFAULT_OVERLAP_INCLUSION_METHOD = CassandraRelevantProperties.UCS_OVERLAP_INCLUSION_METHOD.getEnum(Overlaps.InclusionMethod.TRANSITIVE);
    protected final ColumnFamilyStore cfs;
    protected final MonotonicClock clock;
    private final int[] scalingParameters;
    protected final double[] survivalFactors;
    protected volatile long minSSTableSize;
    protected final long flushSizeOverride;
    protected volatile long currentFlushSize;
    protected final int maxSSTablesToCompact;
    protected final long expiredSSTableCheckFrequency;
    protected final boolean ignoreOverlapsInExpirationCheck;
    protected final int baseShardCount;
    protected final double targetSSTableSize;
    protected final double sstableGrowthModifier;
    static final double INVERSE_SQRT_2 = Math.sqrt(0.5);
    private static final double INVERSE_LOG_2 = 1.0 / Math.log(2.0);
    protected final Overlaps.InclusionMethod overlapInclusionMethod;

    Controller(ColumnFamilyStore cfs, MonotonicClock clock, int[] scalingParameters, double[] survivalFactors, long minSSTableSize, long flushSizeOverride, int maxSSTablesToCompact, long expiredSSTableCheckFrequency, boolean ignoreOverlapsInExpirationCheck, int baseShardCount, double targetSStableSize, double sstableGrowthModifier, Overlaps.InclusionMethod overlapInclusionMethod) {
        this.cfs = cfs;
        this.clock = clock;
        this.scalingParameters = scalingParameters;
        this.survivalFactors = survivalFactors;
        this.minSSTableSize = minSSTableSize;
        this.flushSizeOverride = flushSizeOverride;
        this.currentFlushSize = flushSizeOverride;
        this.expiredSSTableCheckFrequency = TimeUnit.MILLISECONDS.convert(expiredSSTableCheckFrequency, TimeUnit.SECONDS);
        this.baseShardCount = baseShardCount;
        this.targetSSTableSize = targetSStableSize;
        this.overlapInclusionMethod = overlapInclusionMethod;
        this.sstableGrowthModifier = sstableGrowthModifier;
        if (maxSSTablesToCompact <= 0) {
            maxSSTablesToCompact = Integer.MAX_VALUE;
        }
        this.maxSSTablesToCompact = maxSSTablesToCompact;
        if (ignoreOverlapsInExpirationCheck && !ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION) {
            logger.warn("Not enabling aggressive SSTable expiration, as the system property '" + CassandraRelevantProperties.ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION.name() + "' is set to 'false'. Set it to 'true' to enable aggressive SSTable expiration.");
        }
        this.ignoreOverlapsInExpirationCheck = ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION && ignoreOverlapsInExpirationCheck;
    }

    public int getScalingParameter(int index) {
        if (index < 0) {
            throw new IllegalArgumentException("Index should be >= 0: " + index);
        }
        return index < this.scalingParameters.length ? this.scalingParameters[index] : this.scalingParameters[this.scalingParameters.length - 1];
    }

    public String toString() {
        return String.format("Controller, m: %s, o: %s, Ws: %s", FBUtilities.prettyPrintBinary(this.targetSSTableSize, "B", ""), Arrays.toString(this.survivalFactors), Controller.printScalingParameters(this.scalingParameters));
    }

    public int getFanout(int index) {
        int W = this.getScalingParameter(index);
        return UnifiedCompactionStrategy.fanoutFromScalingParameter(W);
    }

    public int getThreshold(int index) {
        int W = this.getScalingParameter(index);
        return UnifiedCompactionStrategy.thresholdFromScalingParameter(W);
    }

    public int getNumShards(double localDensity) {
        double count;
        if (this.minSSTableSize > 0L && !((count = localDensity / (double)this.minSSTableSize) >= (double)this.baseShardCount)) {
            int shards = Math.min(Integer.highestOneBit((int)count | 1), this.baseShardCount & -this.baseShardCount);
            if (logger.isDebugEnabled()) {
                logger.debug("Shard count {} for density {}, {} times min size {}", new Object[]{shards, FBUtilities.prettyPrintBinary(localDensity, "B", " "), localDensity / (double)this.minSSTableSize, FBUtilities.prettyPrintBinary(this.minSSTableSize, "B", " ")});
            }
            return shards;
        }
        if (this.sstableGrowthModifier == 1.0) {
            int shards = this.baseShardCount;
            logger.debug("Shard count {} for density {} in fixed shards mode", (Object)shards, (Object)FBUtilities.prettyPrintBinary(localDensity, "B", " "));
            return shards;
        }
        if (this.sstableGrowthModifier == 0.0) {
            count = localDensity / (this.targetSSTableSize * INVERSE_SQRT_2 * (double)this.baseShardCount);
            if (count > MAX_SHARD_SPLIT) {
                count = MAX_SHARD_SPLIT;
            }
            assert (!(count < 0.0));
            int shards = this.baseShardCount * Integer.highestOneBit((int)count | 1);
            if (logger.isDebugEnabled()) {
                logger.debug("Shard count {} for density {}, {} times target {}", new Object[]{shards, FBUtilities.prettyPrintBinary(localDensity, "B", " "), localDensity / this.targetSSTableSize, FBUtilities.prettyPrintBinary(this.targetSSTableSize, "B", " ")});
            }
            return shards;
        }
        count = localDensity / (this.targetSSTableSize * (double)this.baseShardCount);
        double countLog = Math.log(count);
        double pow = countLog * INVERSE_LOG_2 * (1.0 - this.sstableGrowthModifier) + 0.5;
        int shards = pow >= 20.0 ? this.baseShardCount << 20 : (pow >= 0.0 ? this.baseShardCount << (int)pow : this.baseShardCount);
        if (logger.isDebugEnabled()) {
            long targetSize = (long)(this.targetSSTableSize * Math.exp(countLog * this.sstableGrowthModifier));
            logger.debug("Shard count {} for density {}, {} times target {}", new Object[]{shards, FBUtilities.prettyPrintBinary(localDensity, "B", " "), localDensity / (double)targetSize, FBUtilities.prettyPrintBinary(targetSize, "B", " ")});
        }
        return shards;
    }

    public double getSurvivalFactor(int index) {
        if (index < 0) {
            throw new IllegalArgumentException("Index should be >= 0: " + index);
        }
        return index < this.survivalFactors.length ? this.survivalFactors[index] : this.survivalFactors[this.survivalFactors.length - 1];
    }

    public long getFlushSizeBytes() {
        if (this.flushSizeOverride > 0L) {
            return this.flushSizeOverride;
        }
        double envFlushSize = this.cfs.metric.flushSizeOnDisk.get();
        if (this.currentFlushSize == 0L || Math.abs(1.0 - (double)this.currentFlushSize / envFlushSize) > 0.5) {
            this.currentFlushSize = (long)Math.ceil(Math.scalb(envFlushSize, -20)) << 20;
        }
        return this.currentFlushSize;
    }

    public boolean getIgnoreOverlapsInExpirationCheck() {
        return this.ignoreOverlapsInExpirationCheck;
    }

    public long getExpiredSSTableCheckFrequency() {
        return this.expiredSSTableCheckFrequency;
    }

    public static Controller fromOptions(ColumnFamilyStore cfs, Map<String, String> options) {
        int[] Ws = Controller.parseScalingParameters(options.getOrDefault(SCALING_PARAMETERS_OPTION, DEFAULT_SCALING_PARAMETERS));
        long flushSizeOverride = FBUtilities.parseHumanReadableBytes(options.getOrDefault(FLUSH_SIZE_OVERRIDE_OPTION, "0MiB"));
        int maxSSTablesToCompact = Integer.parseInt(options.getOrDefault(MAX_SSTABLES_TO_COMPACT_OPTION, "0"));
        long expiredSSTableCheckFrequency = options.containsKey(EXPIRED_SSTABLE_CHECK_FREQUENCY_SECONDS_OPTION) ? Long.parseLong(options.get(EXPIRED_SSTABLE_CHECK_FREQUENCY_SECONDS_OPTION)) : 600L;
        boolean ignoreOverlapsInExpirationCheck = options.containsKey(ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION_OPTION) ? Boolean.parseBoolean(options.get(ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION_OPTION)) : false;
        int baseShardCount = options.containsKey(BASE_SHARD_COUNT_OPTION) ? Integer.parseInt(options.get(BASE_SHARD_COUNT_OPTION)) : DEFAULT_BASE_SHARD_COUNT;
        long targetSStableSize = options.containsKey(TARGET_SSTABLE_SIZE_OPTION) ? FBUtilities.parseHumanReadableBytes(options.get(TARGET_SSTABLE_SIZE_OPTION)) : DEFAULT_TARGET_SSTABLE_SIZE;
        long minSSTableSize = options.containsKey(MIN_SSTABLE_SIZE_OPTION) ? FBUtilities.parseHumanReadableBytes(options.get(MIN_SSTABLE_SIZE_OPTION)) : FBUtilities.parseHumanReadableBytes(DEFAULT_MIN_SSTABLE_SIZE);
        double sstableGrowthModifier = DEFAULT_SSTABLE_GROWTH;
        if (options.containsKey(SSTABLE_GROWTH_OPTION)) {
            sstableGrowthModifier = FBUtilities.parsePercent(options.get(SSTABLE_GROWTH_OPTION));
        }
        Overlaps.InclusionMethod inclusionMethod = options.containsKey(OVERLAP_INCLUSION_METHOD_OPTION) ? Overlaps.InclusionMethod.valueOf(options.get(OVERLAP_INCLUSION_METHOD_OPTION).toUpperCase()) : DEFAULT_OVERLAP_INCLUSION_METHOD;
        return new Controller(cfs, MonotonicClock.Global.preciseTime, Ws, DEFAULT_SURVIVAL_FACTORS, minSSTableSize, flushSizeOverride, maxSSTablesToCompact, expiredSSTableCheckFrequency, ignoreOverlapsInExpirationCheck, baseShardCount, targetSStableSize, sstableGrowthModifier, inclusionMethod);
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        String s = (options = new HashMap<String, String>(options)).remove(SCALING_PARAMETERS_OPTION);
        if (s != null) {
            Controller.parseScalingParameters(s);
        }
        if ((s = options.remove(BASE_SHARD_COUNT_OPTION)) != null) {
            try {
                int numShards = Integer.parseInt(s);
                if (numShards <= 0) {
                    throw new ConfigurationException(String.format("Invalid configuration, %s should be positive: %d", BASE_SHARD_COUNT_OPTION, numShards));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", s, BASE_SHARD_COUNT_OPTION), e);
            }
        }
        long targetSSTableSize = DEFAULT_TARGET_SSTABLE_SIZE;
        s = options.remove(TARGET_SSTABLE_SIZE_OPTION);
        if (s != null) {
            try {
                targetSSTableSize = FBUtilities.parseHumanReadableBytes(s);
                if (targetSSTableSize < 0x100000L) {
                    throw new ConfigurationException(String.format("%s %s is not acceptable, size must be at least %s", TARGET_SSTABLE_SIZE_OPTION, s, FBUtilities.prettyPrintMemory(0x100000L)));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s %s is not a valid size in bytes: %s", TARGET_SSTABLE_SIZE_OPTION, s, e.getMessage()), e);
            }
        }
        if ((s = options.remove(FLUSH_SIZE_OVERRIDE_OPTION)) != null) {
            try {
                long flushSize = FBUtilities.parseHumanReadableBytes(s);
                if (flushSize < 0x100000L) {
                    throw new ConfigurationException(String.format("%s %s is not acceptable, size must be at least %s", FLUSH_SIZE_OVERRIDE_OPTION, s, FBUtilities.prettyPrintMemory(0x100000L)));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s %s is not a valid size in bytes: %s", FLUSH_SIZE_OVERRIDE_OPTION, s, e.getMessage()), e);
            }
        }
        if ((s = options.remove(MAX_SSTABLES_TO_COMPACT_OPTION)) != null) {
            try {
                Integer.parseInt(s);
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", s, MAX_SSTABLES_TO_COMPACT_OPTION), e);
            }
        }
        if ((s = options.remove(EXPIRED_SSTABLE_CHECK_FREQUENCY_SECONDS_OPTION)) != null) {
            try {
                long expiredSSTableCheckFrequency = Long.parseLong(s);
                if (expiredSSTableCheckFrequency <= 0L) {
                    throw new ConfigurationException(String.format("Invalid configuration, %s should be positive: %d", EXPIRED_SSTABLE_CHECK_FREQUENCY_SECONDS_OPTION, expiredSSTableCheckFrequency));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s is not a parsable long (base10) for %s", s, EXPIRED_SSTABLE_CHECK_FREQUENCY_SECONDS_OPTION), e);
            }
        }
        if ((s = options.remove(ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION_OPTION)) != null && !s.equalsIgnoreCase("true") && !s.equalsIgnoreCase("false")) {
            throw new ConfigurationException(String.format("%s should either be 'true' or 'false', not %s", ALLOW_UNSAFE_AGGRESSIVE_SSTABLE_EXPIRATION_OPTION, s));
        }
        s = options.remove(OVERLAP_INCLUSION_METHOD_OPTION);
        if (s != null) {
            try {
                Overlaps.InclusionMethod.valueOf(s.toUpperCase());
            }
            catch (IllegalArgumentException e) {
                throw new ConfigurationException(String.format("Invalid overlap inclusion method %s. The valid options are %s.", s, Arrays.toString((Object[])Overlaps.InclusionMethod.values())));
            }
        }
        if ((s = options.remove(MIN_SSTABLE_SIZE_OPTION)) != null) {
            try {
                long sizeInBytes = FBUtilities.parseHumanReadableBytes(s);
                if (sizeInBytes < 0L) {
                    throw new ConfigurationException(String.format("Invalid configuration, %s should be greater than or equal to 0 (zero)", MIN_SSTABLE_SIZE_OPTION));
                }
                int limit = (int)Math.ceil((double)targetSSTableSize * INVERSE_SQRT_2);
                if (sizeInBytes >= (long)limit) {
                    throw new ConfigurationException(String.format("Invalid configuration, %s (%s) should be less than the target size minimum: %s", MIN_SSTABLE_SIZE_OPTION, FBUtilities.prettyPrintMemory(sizeInBytes), FBUtilities.prettyPrintMemory(limit)));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s is not a valid size in bytes for %s", s, MIN_SSTABLE_SIZE_OPTION), e);
            }
        }
        if ((s = options.remove(SSTABLE_GROWTH_OPTION)) != null) {
            try {
                double targetSSTableGrowth = FBUtilities.parsePercent(s);
                if (targetSSTableGrowth < 0.0 || targetSSTableGrowth > 1.0) {
                    throw new ConfigurationException(String.format("%s %s must be between 0 and 1", SSTABLE_GROWTH_OPTION, s));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s is not a valid number between 0 and 1: %s", SSTABLE_GROWTH_OPTION, e.getMessage()), e);
            }
        }
        return options;
    }

    public double getBaseSstableSize(int F) {
        return (double)Math.max(0x100000L, this.getFlushSizeBytes()) * (1.0 - 0.9 / (double)F);
    }

    public double getMaxLevelDensity(int index, double minSize) {
        return Math.floor(minSize * (double)this.getFanout(index) * this.getSurvivalFactor(index));
    }

    public double maxThroughput() {
        double compactionThroughputMbPerSec = DatabaseDescriptor.getCompactionThroughputMebibytesPerSec();
        if (compactionThroughputMbPerSec <= 0.0) {
            return Double.MAX_VALUE;
        }
        return Math.scalb(compactionThroughputMbPerSec, 20);
    }

    public int maxConcurrentCompactions() {
        return DatabaseDescriptor.getConcurrentCompactors();
    }

    public int maxSSTablesToCompact() {
        return this.maxSSTablesToCompact;
    }

    public Random random() {
        return ThreadLocalRandom.current();
    }

    public Overlaps.InclusionMethod overlapInclusionMethod() {
        return this.overlapInclusionMethod;
    }

    public static int[] parseScalingParameters(String str) {
        String[] vals = str.split(",");
        int[] ret = new int[vals.length];
        for (int i = 0; i < vals.length; ++i) {
            int W;
            String value = vals[i].trim();
            ret[i] = W = UnifiedCompactionStrategy.parseScalingParameter(value);
        }
        return ret;
    }

    public static String printScalingParameters(int[] parameters) {
        int i;
        StringBuilder builder = new StringBuilder();
        for (i = 0; i < parameters.length - 1; ++i) {
            builder.append(UnifiedCompactionStrategy.printScalingParameter(parameters[i]));
            builder.append(", ");
        }
        builder.append(UnifiedCompactionStrategy.printScalingParameter(parameters[i]));
        return builder.toString();
    }
}

