/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.Iterator;
import java.util.Queue;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.zip.CRC32;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.io.util.DataOutputBufferFixed;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.utils.AbstractIterator;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.concurrent.OpOrder;

final class HintsBuffer {
    static final int ENTRY_OVERHEAD_SIZE = 12;
    private final ByteBuffer slab;
    private final AtomicLong position;
    private final ConcurrentMap<UUID, Queue<Integer>> offsets;
    private final OpOrder appendOrder;
    private final ConcurrentMap<UUID, Long> earliestHintByHost;

    private HintsBuffer(ByteBuffer slab) {
        this.slab = slab;
        this.position = new AtomicLong();
        this.offsets = new ConcurrentHashMap<UUID, Queue<Integer>>();
        this.appendOrder = new OpOrder();
        this.earliestHintByHost = new ConcurrentHashMap<UUID, Long>();
    }

    static HintsBuffer create(int slabSize) {
        return new HintsBuffer(ByteBuffer.allocateDirect(slabSize));
    }

    boolean isClosed() {
        return this.position.get() < 0L;
    }

    int capacity() {
        return this.slab.capacity();
    }

    int remaining() {
        long pos = this.position.get();
        return (int)(pos < 0L ? 0L : Math.max(0L, (long)this.capacity() - pos));
    }

    HintsBuffer recycle() {
        this.slab.clear();
        return new HintsBuffer(this.slab);
    }

    void free() {
        FileUtils.clean(this.slab);
    }

    void waitForModifications() {
        this.appendOrder.awaitNewBarrier();
    }

    Set<UUID> hostIds() {
        return this.offsets.keySet();
    }

    Iterator<ByteBuffer> consumingHintsIterator(UUID hostId) {
        final Queue bufferOffsets = (Queue)this.offsets.get(hostId);
        if (bufferOffsets == null) {
            return Collections.emptyIterator();
        }
        return new AbstractIterator<ByteBuffer>(){
            private final ByteBuffer flyweight;
            {
                this.flyweight = HintsBuffer.this.slab.duplicate();
            }

            @Override
            protected ByteBuffer computeNext() {
                Integer offset = (Integer)bufferOffsets.poll();
                if (offset == null) {
                    return (ByteBuffer)this.endOfData();
                }
                int totalSize = HintsBuffer.this.slab.getInt(offset) + 12;
                return this.flyweight.clear().position(offset).limit(offset + totalSize);
            }
        };
    }

    long getEarliestHintTime(UUID hostId) {
        return this.earliestHintByHost.getOrDefault(hostId, Clock.Global.currentTimeMillis());
    }

    void clearEarliestHintForHostId(UUID hostId) {
        this.earliestHintByHost.remove(hostId);
    }

    Allocation allocate(int hintSize) {
        int totalSize = hintSize + 12;
        if (totalSize > this.slab.capacity() / 2) {
            throw new IllegalArgumentException(String.format("Hint of %s bytes is too large - the maximum size is %s", hintSize, this.slab.capacity() / 2));
        }
        OpOrder.Group opGroup = this.appendOrder.start();
        try {
            return this.allocate(totalSize, opGroup);
        }
        catch (Throwable t) {
            opGroup.close();
            throw t;
        }
    }

    private Allocation allocate(int totalSize, OpOrder.Group opGroup) {
        int offset = this.allocateBytes(totalSize);
        if (offset < 0) {
            opGroup.close();
            return null;
        }
        return new Allocation(offset, totalSize, opGroup);
    }

    private int allocateBytes(int totalSize) {
        long prev = this.position.getAndAdd(totalSize);
        if (prev < 0L) {
            return -1;
        }
        if (prev + (long)totalSize > (long)this.slab.capacity()) {
            this.position.set(Long.MIN_VALUE);
            return -1;
        }
        return (int)prev;
    }

    private void put(UUID hostId, int offset) {
        Queue queue = (Queue)this.offsets.get(hostId);
        if (queue == null) {
            queue = this.offsets.computeIfAbsent(hostId, id -> new ConcurrentLinkedQueue());
        }
        queue.offer(offset);
    }

    final class Allocation
    implements AutoCloseable {
        private final Integer offset;
        private final int totalSize;
        private final OpOrder.Group opGroup;

        Allocation(int offset, int totalSize, OpOrder.Group opGroup) {
            this.offset = offset;
            this.totalSize = totalSize;
            this.opGroup = opGroup;
        }

        void write(Iterable<UUID> hostIds, Hint hint) {
            this.write(hint);
            long ts = Clock.Global.currentTimeMillis();
            for (UUID hostId : hostIds) {
                if (DatabaseDescriptor.hintWindowPersistentEnabled()) {
                    HintsBuffer.this.earliestHintByHost.putIfAbsent(hostId, ts);
                }
                HintsBuffer.this.put(hostId, this.offset);
            }
        }

        @Override
        public void close() {
            this.opGroup.close();
        }

        private void write(Hint hint) {
            ByteBuffer buffer = HintsBuffer.this.slab.duplicate().position(this.offset).limit(this.offset + this.totalSize);
            CRC32 crc = new CRC32();
            int hintSize = this.totalSize - 12;
            try (DataOutputBufferFixed dop = new DataOutputBufferFixed(buffer);){
                dop.writeInt(hintSize);
                FBUtilities.updateChecksumInt(crc, hintSize);
                dop.writeInt((int)crc.getValue());
                Hint.serializer.serialize(hint, (DataOutputPlus)dop, MessagingService.current_version);
                FBUtilities.updateChecksum(crc, buffer, buffer.position() - hintSize, hintSize);
                dop.writeInt((int)crc.getValue());
            }
            catch (IOException e) {
                throw new AssertionError();
            }
        }
    }
}

