/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.collect.ImmutableList;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.SerializationHeader;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.db.marshal.AbstractCompositeType;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.CollectionType;
import org.apache.cassandra.db.marshal.CompositeType;
import org.apache.cassandra.db.marshal.DynamicCompositeType;
import org.apache.cassandra.db.marshal.ListType;
import org.apache.cassandra.db.marshal.MapType;
import org.apache.cassandra.db.marshal.SetType;
import org.apache.cassandra.db.marshal.TupleType;
import org.apache.cassandra.db.marshal.UserType;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.metadata.MetadataComponent;
import org.apache.cassandra.io.sstable.metadata.MetadataType;
import org.apache.cassandra.schema.ColumnMetadata;
import org.apache.cassandra.schema.IndexMetadata;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.SchemaConstants;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.CassandraVersion;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class SSTableHeaderFix {
    private static final String SKIPAUTOMATICUDTFIX = "cassandra.skipautomaticudtfix";
    private static final boolean SKIP_AUTOMATIC_FIX_ON_UPGRADE = Boolean.getBoolean("cassandra.skipautomaticudtfix");
    private static final Logger logger = LoggerFactory.getLogger(SSTableHeaderFix.class);
    protected final Consumer<String> info;
    protected final Consumer<String> warn;
    protected final Consumer<String> error;
    protected final boolean dryRun;
    protected final Function<Descriptor, TableMetadata> schemaCallback;
    private final List<Descriptor> descriptors;
    private final List<Pair<Descriptor, Map<MetadataType, MetadataComponent>>> updates = new ArrayList<Pair<Descriptor, Map<MetadataType, MetadataComponent>>>();
    private boolean hasErrors;

    public static void fixNonFrozenUDTIfUpgradeFrom30() {
        String previousVersionString = FBUtilities.getPreviousReleaseVersionString();
        if (previousVersionString == null) {
            return;
        }
        CassandraVersion previousVersion = new CassandraVersion(previousVersionString);
        if (previousVersion.major != 3 || previousVersion.minor > 0) {
            return;
        }
        if (SKIP_AUTOMATIC_FIX_ON_UPGRADE) {
            logger.warn("Detected upgrade from {} to {}, but -D{}=true, NOT fixing UDT type references in sstable metadata serialization-headers", new Object[]{previousVersionString, FBUtilities.getReleaseVersionString(), SKIPAUTOMATICUDTFIX});
            return;
        }
        logger.info("Detected upgrade from {} to {}, fixing UDT type references in sstable metadata serialization-headers", (Object)previousVersionString, (Object)FBUtilities.getReleaseVersionString());
        SSTableHeaderFix instance = SSTableHeaderFix.builder().schemaCallback(() -> Schema.instance::getTableMetadata).build();
        instance.execute();
    }

    SSTableHeaderFix(Builder builder) {
        this.info = builder.info;
        this.warn = builder.warn;
        this.error = builder.error;
        this.dryRun = builder.dryRun;
        this.schemaCallback = (Function)builder.schemaCallback.get();
        this.descriptors = new ArrayList<Descriptor>(builder.descriptors);
        Objects.requireNonNull(this.info, "info is null");
        Objects.requireNonNull(this.warn, "warn is null");
        Objects.requireNonNull(this.error, "error is null");
        Objects.requireNonNull(this.schemaCallback, "schemaCallback is null");
    }

    public static Builder builder() {
        return new Builder();
    }

    public final void execute() {
        this.prepare();
        logger.debug("Processing {} sstables:{}", (Object)this.descriptors.size(), (Object)this.descriptors.stream().map(Descriptor::toString).collect(Collectors.joining("\n    ", "\n    ", "")));
        this.descriptors.forEach(this::processSSTable);
        if (this.updates.isEmpty()) {
            return;
        }
        if (this.hasErrors) {
            this.info.accept("Stopping due to previous errors. Either fix the errors or specify the ignore-errors option.");
            return;
        }
        if (this.dryRun) {
            this.info.accept("Not fixing identified and fixable serialization-header issues.");
            return;
        }
        this.info.accept("Writing new metadata files");
        this.updates.forEach(descAndMeta -> this.writeNewMetadata((Descriptor)descAndMeta.left, (Map)descAndMeta.right));
        this.info.accept("Finished writing new metadata files");
    }

    public boolean hasError() {
        return this.hasErrors;
    }

    public boolean hasChanges() {
        return !this.updates.isEmpty();
    }

    abstract void prepare();

    private void error(String format, Object ... args) {
        this.hasErrors = true;
        this.error.accept(String.format(format, args));
    }

    void processFileOrDirectory(Path path) {
        Stream.of(path).flatMap(SSTableHeaderFix::maybeExpandDirectory).filter(p -> {
            try {
                return ((Component)Descriptor.fromFilenameWithComponent((org.apache.cassandra.io.util.File)new org.apache.cassandra.io.util.File((File)p.toFile())).right).type == Component.Type.DATA;
            }
            catch (IllegalArgumentException t) {
                logger.info("Couldn't parse filename {}, ignoring", p);
                return false;
            }
        }).map(Path::toString).map(Descriptor::fromFilename).forEach(this.descriptors::add);
    }

    private static Stream<Path> maybeExpandDirectory(Path path) {
        if (Files.isRegularFile(path, new LinkOption[0])) {
            return Stream.of(path);
        }
        return LifecycleTransaction.getFiles(path, (file, fileType) -> fileType == Directories.FileType.FINAL, Directories.OnTxnErr.IGNORE).stream().map(org.apache.cassandra.io.util.File::toPath);
    }

    private void processSSTable(Descriptor desc) {
        if (desc.cfname.indexOf(46) != -1) {
            return;
        }
        TableMetadata tableMetadata = this.schemaCallback.apply(desc);
        if (tableMetadata == null) {
            this.error("Table %s.%s not found in the schema - NOT checking sstable %s", desc.ksname, desc.cfname, desc);
            return;
        }
        Set<Component> components = SSTable.discoverComponentsFor(desc);
        if (components.stream().noneMatch(c -> c.type == Component.Type.STATS)) {
            this.error("sstable %s has no -Statistics.db component.", desc);
            return;
        }
        Map<MetadataType, MetadataComponent> metadata = this.readSSTableMetadata(desc);
        if (metadata == null) {
            return;
        }
        MetadataComponent component = metadata.get((Object)MetadataType.HEADER);
        if (!(component instanceof SerializationHeader.Component)) {
            this.error("sstable %s: Expected %s, but got %s from metadata.get(MetadataType.HEADER)", desc, SerializationHeader.Component.class.getName(), component != null ? component.getClass().getName() : "'null'");
            return;
        }
        SerializationHeader.Component header = (SerializationHeader.Component)component;
        AbstractType<?> keyType = this.validatePartitionKey(desc, tableMetadata, header);
        List<AbstractType<?>> clusteringTypes = this.validateClusteringColumns(desc, tableMetadata, header);
        Map<ByteBuffer, AbstractType<?>> staticColumns = this.validateColumns(desc, tableMetadata, header.getStaticColumns(), ColumnMetadata.Kind.STATIC);
        Map<ByteBuffer, AbstractType<?>> regularColumns = this.validateColumns(desc, tableMetadata, header.getRegularColumns(), ColumnMetadata.Kind.REGULAR);
        SerializationHeader.Component newHeader = SerializationHeader.Component.buildComponentForTools(keyType, clusteringTypes, staticColumns, regularColumns, header.getEncodingStats());
        if (header.toString().equals(newHeader.toString())) {
            return;
        }
        LinkedHashMap<MetadataType, MetadataComponent> newMetadata = new LinkedHashMap<MetadataType, MetadataComponent>(metadata);
        newMetadata.put(MetadataType.HEADER, newHeader);
        this.updates.add(Pair.create(desc, newMetadata));
    }

    private AbstractType<?> validatePartitionKey(Descriptor desc, TableMetadata tableMetadata, SerializationHeader.Component header) {
        AbstractType<?> schemaKeyType;
        boolean schemaKeyComposite;
        boolean keyMismatch = false;
        CompositeType headerKeyType = header.getKeyType();
        boolean headerKeyComposite = headerKeyType instanceof CompositeType;
        if (headerKeyComposite != (schemaKeyComposite = (schemaKeyType = tableMetadata.partitionKeyType) instanceof CompositeType)) {
            keyMismatch = true;
        } else if (headerKeyComposite) {
            List<AbstractType<?>> headerKeyComponents = ((CompositeType)headerKeyType).types;
            List<AbstractType<?>> schemaKeyComponents = ((CompositeType)schemaKeyType).types;
            if (headerKeyComponents.size() != schemaKeyComponents.size()) {
                keyMismatch = true;
            } else {
                ArrayList newComponents = new ArrayList(schemaKeyComponents.size());
                for (int i = 0; i < schemaKeyComponents.size(); ++i) {
                    AbstractType<?> schemaKeyComponent;
                    AbstractType<?> headerKeyComponent = headerKeyComponents.get(i);
                    AbstractType<?> fixedType = this.fixType(desc, ((ColumnMetadata)tableMetadata.partitionKeyColumns().get((int)i)).name.bytes, headerKeyComponent, schemaKeyComponent = schemaKeyComponents.get(i), false);
                    if (fixedType == null) {
                        keyMismatch = true;
                    } else {
                        headerKeyComponent = fixedType;
                    }
                    newComponents.add(this.fixType(desc, ((ColumnMetadata)tableMetadata.partitionKeyColumns().get((int)i)).name.bytes, headerKeyComponent, schemaKeyComponent, false));
                }
                headerKeyType = CompositeType.getInstance(newComponents);
            }
        } else {
            AbstractType<?> fixedType = this.fixType(desc, ((ColumnMetadata)tableMetadata.partitionKeyColumns().get((int)0)).name.bytes, headerKeyType, schemaKeyType, false);
            if (fixedType == null) {
                keyMismatch = true;
            } else {
                headerKeyType = fixedType;
            }
        }
        if (keyMismatch) {
            this.error("sstable %s: Mismatch in partition key type between sstable serialization-header and schema (%s vs %s)", desc, headerKeyType.asCQL3Type(), schemaKeyType.asCQL3Type());
        }
        return headerKeyType;
    }

    private List<AbstractType<?>> validateClusteringColumns(Descriptor desc, TableMetadata tableMetadata, SerializationHeader.Component header) {
        List<AbstractType<?>> headerClusteringTypes = header.getClusteringTypes();
        ArrayList clusteringTypes = new ArrayList();
        boolean clusteringMismatch = false;
        ImmutableList<ColumnMetadata> schemaClustering = tableMetadata.clusteringColumns();
        if (schemaClustering.size() != headerClusteringTypes.size()) {
            clusteringMismatch = true;
            clusteringTypes.addAll(headerClusteringTypes);
        } else {
            for (int i = 0; i < headerClusteringTypes.size(); ++i) {
                AbstractType<?> headerType = headerClusteringTypes.get(i);
                ColumnMetadata column = (ColumnMetadata)schemaClustering.get(i);
                AbstractType schemaType = column.type;
                AbstractType<?> fixedType = this.fixType(desc, column.name.bytes, headerType, schemaType, false);
                if (fixedType == null) {
                    clusteringMismatch = true;
                } else {
                    headerType = fixedType;
                }
                clusteringTypes.add(headerType);
            }
        }
        if (clusteringMismatch) {
            this.error("sstable %s: mismatch in clustering columns between sstable serialization-header and schema (%s vs %s)", desc, headerClusteringTypes.stream().map(AbstractType::asCQL3Type).map(Object::toString).collect(Collectors.joining(",")), schemaClustering.stream().map(cd -> cd.type.asCQL3Type().toString()).collect(Collectors.joining(",")));
        }
        return clusteringTypes;
    }

    private Map<ByteBuffer, AbstractType<?>> validateColumns(Descriptor desc, TableMetadata tableMetadata, Map<ByteBuffer, AbstractType<?>> columns, ColumnMetadata.Kind kind) {
        LinkedHashMap target = new LinkedHashMap();
        for (Map.Entry<ByteBuffer, AbstractType<?>> nameAndType : columns.entrySet()) {
            AbstractType<?> type;
            ByteBuffer name = nameAndType.getKey();
            AbstractType<?> fixedType = this.validateColumn(desc, tableMetadata, kind, name, type = nameAndType.getValue());
            if (fixedType == null) {
                this.error("sstable %s: contains column '%s' of type '%s', which could not be validated", desc, type, SSTableHeaderFix.logColumnName(name));
                fixedType = type;
            }
            target.put(name, fixedType);
        }
        return target;
    }

    private AbstractType<?> validateColumn(Descriptor desc, TableMetadata tableMetadata, ColumnMetadata.Kind kind, ByteBuffer name, AbstractType<?> type) {
        ColumnMetadata cd = tableMetadata.getColumn(name);
        if (cd == null) {
            cd = tableMetadata.getDroppedColumn(name, kind == ColumnMetadata.Kind.STATIC);
            if (cd == null) {
                for (IndexMetadata indexMetadata : tableMetadata.indexes) {
                    String target = indexMetadata.options.get("target");
                    if (target == null || !ByteBufferUtil.bytes(target).equals(name)) continue;
                    this.warn.accept(String.format("sstable %s: contains column '%s', which is not a column in the table '%s.%s', but a target for that table's index '%s'", desc, SSTableHeaderFix.logColumnName(name), tableMetadata.keyspace, tableMetadata.name, indexMetadata.name));
                    return type;
                }
                this.warn.accept(String.format("sstable %s: contains column '%s', which is not present in the schema", desc, SSTableHeaderFix.logColumnName(name)));
            } else if (type instanceof UserType && cd.type instanceof TupleType) {
                return this.fixType(desc, name, type, cd.type, true);
            }
            return type;
        }
        if (cd.kind != kind) {
            this.error("sstable %s: contains column '%s' as a %s column, but is of kind %s in the schema", desc, SSTableHeaderFix.logColumnName(name), kind.name().toLowerCase(), cd.kind.name().toLowerCase());
        } else {
            type = this.fixType(desc, name, type, cd.type, false);
        }
        return type;
    }

    private AbstractType<?> fixType(Descriptor desc, ByteBuffer name, AbstractType<?> typeInHeader, AbstractType<?> typeInSchema, boolean droppedColumnMode) {
        AbstractType<?> fixedType = this.fixTypeInner(typeInHeader, typeInSchema, droppedColumnMode);
        if (fixedType != null) {
            if (fixedType != typeInHeader) {
                this.info.accept(String.format("sstable %s: Column '%s' needs to be updated from type '%s' to '%s'", desc, SSTableHeaderFix.logColumnName(name), typeInHeader.asCQL3Type(), fixedType.asCQL3Type()));
            }
            return fixedType;
        }
        this.error("sstable %s: contains column '%s' as type '%s', but schema mentions '%s'", desc, SSTableHeaderFix.logColumnName(name), typeInHeader.asCQL3Type(), typeInSchema.asCQL3Type());
        return typeInHeader;
    }

    private AbstractType<?> fixTypeInner(AbstractType<?> typeInHeader, AbstractType<?> typeInSchema, boolean droppedColumnMode) {
        if (this.typeEquals(typeInHeader, typeInSchema)) {
            return typeInHeader;
        }
        if (typeInHeader instanceof CollectionType) {
            return this.fixTypeInnerCollection(typeInHeader, typeInSchema, droppedColumnMode);
        }
        if (typeInHeader instanceof AbstractCompositeType) {
            return this.fixTypeInnerAbstractComposite(typeInHeader, typeInSchema, droppedColumnMode);
        }
        if (typeInHeader instanceof TupleType) {
            return this.fixTypeInnerAbstractTuple(typeInHeader, typeInSchema, droppedColumnMode);
        }
        if (typeInHeader.isCompatibleWith(typeInSchema)) {
            return typeInHeader;
        }
        return null;
    }

    private AbstractType<?> fixTypeInnerAbstractTuple(AbstractType<?> typeInHeader, AbstractType<?> typeInSchema, boolean droppedColumnMode) {
        if (droppedColumnMode && typeInHeader.getClass() == UserType.class && typeInSchema instanceof TupleType) {
            return this.fixTypeInnerUserTypeDropped((UserType)typeInHeader, (TupleType)typeInSchema);
        }
        if (typeInHeader.getClass() != typeInSchema.getClass()) {
            return null;
        }
        if (typeInHeader.getClass() == UserType.class) {
            return this.fixTypeInnerUserType((UserType)typeInHeader, (UserType)typeInSchema);
        }
        if (typeInHeader.getClass() == TupleType.class) {
            return this.fixTypeInnerTuple((TupleType)typeInHeader, (TupleType)typeInSchema, droppedColumnMode);
        }
        throw new IllegalArgumentException("Unknown tuple type class " + typeInHeader.getClass().getName());
    }

    private AbstractType<?> fixTypeInnerCollection(AbstractType<?> typeInHeader, AbstractType<?> typeInSchema, boolean droppedColumnMode) {
        if (typeInHeader.getClass() != typeInSchema.getClass()) {
            return null;
        }
        if (typeInHeader.getClass() == ListType.class) {
            return this.fixTypeInnerList((ListType)typeInHeader, (ListType)typeInSchema, droppedColumnMode);
        }
        if (typeInHeader.getClass() == SetType.class) {
            return this.fixTypeInnerSet((SetType)typeInHeader, (SetType)typeInSchema, droppedColumnMode);
        }
        if (typeInHeader.getClass() == MapType.class) {
            return this.fixTypeInnerMap((MapType)typeInHeader, (MapType)typeInSchema, droppedColumnMode);
        }
        throw new IllegalArgumentException("Unknown collection type class " + typeInHeader.getClass().getName());
    }

    private AbstractType<?> fixTypeInnerAbstractComposite(AbstractType<?> typeInHeader, AbstractType<?> typeInSchema, boolean droppedColumnMode) {
        if (typeInHeader.getClass() != typeInSchema.getClass()) {
            return null;
        }
        if (typeInHeader.getClass() == CompositeType.class) {
            return this.fixTypeInnerComposite((CompositeType)typeInHeader, (CompositeType)typeInSchema, droppedColumnMode);
        }
        if (typeInHeader.getClass() == DynamicCompositeType.class) {
            if (!typeInHeader.isCompatibleWith(typeInSchema)) {
                return null;
            }
            return typeInHeader;
        }
        throw new IllegalArgumentException("Unknown composite type class " + typeInHeader.getClass().getName());
    }

    private AbstractType<?> fixTypeInnerUserType(UserType cHeader, UserType cSchema) {
        if (!cHeader.keyspace.equals(cSchema.keyspace) || !cHeader.name.equals(cSchema.name)) {
            return null;
        }
        if (cHeader.isMultiCell() != cSchema.isMultiCell()) {
            if (cHeader.isMultiCell() && !cSchema.isMultiCell()) {
                return cSchema;
            }
            return null;
        }
        return cHeader;
    }

    private AbstractType<?> fixTypeInnerUserTypeDropped(UserType cHeader, TupleType cSchema) {
        if (cHeader.isMultiCell() && !cSchema.isMultiCell()) {
            return new UserType(cHeader.keyspace, cHeader.name, cHeader.fieldNames(), cHeader.fieldTypes(), cSchema.isMultiCell());
        }
        return cHeader;
    }

    private AbstractType<?> fixTypeInnerTuple(TupleType cHeader, TupleType cSchema, boolean droppedColumnMode) {
        if (cHeader.size() != cSchema.size()) {
            return null;
        }
        ArrayList cHeaderFixed = new ArrayList(cHeader.size());
        boolean anyChanged = false;
        for (int i = 0; i < cHeader.size(); ++i) {
            AbstractType<?> cHeaderComp = cHeader.type(i);
            AbstractType<?> cHeaderCompFixed = this.fixTypeInner(cHeaderComp, cSchema.type(i), droppedColumnMode);
            if (cHeaderCompFixed == null) {
                return null;
            }
            cHeaderFixed.add(cHeaderCompFixed);
            anyChanged |= cHeaderComp != cHeaderCompFixed;
        }
        if (anyChanged || cSchema.isMultiCell() != cHeader.isMultiCell()) {
            return new TupleType(cHeaderFixed);
        }
        return cHeader;
    }

    private AbstractType<?> fixTypeInnerComposite(CompositeType cHeader, CompositeType cSchema, boolean droppedColumnMode) {
        if (cHeader.types.size() != cSchema.types.size()) {
            return null;
        }
        ArrayList cHeaderFixed = new ArrayList(cHeader.types.size());
        boolean anyChanged = false;
        for (int i = 0; i < cHeader.types.size(); ++i) {
            AbstractType<?> cHeaderComp = cHeader.types.get(i);
            AbstractType<?> cHeaderCompFixed = this.fixTypeInner(cHeaderComp, cSchema.types.get(i), droppedColumnMode);
            if (cHeaderCompFixed == null) {
                return null;
            }
            cHeaderFixed.add(cHeaderCompFixed);
            anyChanged |= cHeaderComp != cHeaderCompFixed;
        }
        if (anyChanged) {
            return CompositeType.getInstance(cHeaderFixed);
        }
        return cHeader;
    }

    private AbstractType<?> fixTypeInnerList(ListType<?> cHeader, ListType<?> cSchema, boolean droppedColumnMode) {
        AbstractType<?> cHeaderElem = cHeader.getElementsType();
        AbstractType<?> cHeaderElemFixed = this.fixTypeInner(cHeaderElem, cSchema.getElementsType(), droppedColumnMode);
        if (cHeaderElemFixed == null) {
            return null;
        }
        if (cHeaderElem != cHeaderElemFixed) {
            return ListType.getInstance(cHeaderElemFixed, cHeader.isMultiCell());
        }
        return cHeader;
    }

    private AbstractType<?> fixTypeInnerSet(SetType<?> cHeader, SetType<?> cSchema, boolean droppedColumnMode) {
        AbstractType<?> cHeaderElem = cHeader.getElementsType();
        AbstractType<?> cHeaderElemFixed = this.fixTypeInner(cHeaderElem, cSchema.getElementsType(), droppedColumnMode);
        if (cHeaderElemFixed == null) {
            return null;
        }
        if (cHeaderElem != cHeaderElemFixed) {
            return SetType.getInstance(cHeaderElemFixed, cHeader.isMultiCell());
        }
        return cHeader;
    }

    private AbstractType<?> fixTypeInnerMap(MapType<?, ?> cHeader, MapType<?, ?> cSchema, boolean droppedColumnMode) {
        AbstractType<?> cHeaderKey = cHeader.getKeysType();
        AbstractType<?> cHeaderVal = cHeader.getValuesType();
        AbstractType<?> cHeaderKeyFixed = this.fixTypeInner(cHeaderKey, cSchema.getKeysType(), droppedColumnMode);
        AbstractType<?> cHeaderValFixed = this.fixTypeInner(cHeaderVal, cSchema.getValuesType(), droppedColumnMode);
        if (cHeaderKeyFixed == null || cHeaderValFixed == null) {
            return null;
        }
        if (cHeaderKey != cHeaderKeyFixed || cHeaderVal != cHeaderValFixed) {
            return MapType.getInstance(cHeaderKeyFixed, cHeaderValFixed, cHeader.isMultiCell());
        }
        return cHeader;
    }

    private boolean typeEquals(AbstractType<?> typeInHeader, AbstractType<?> typeInSchema) {
        return typeInHeader.equals(typeInSchema) || typeInHeader.toString().equals(typeInSchema.toString());
    }

    private static String logColumnName(ByteBuffer columnName) {
        try {
            return ByteBufferUtil.string(columnName);
        }
        catch (CharacterCodingException e) {
            return "?? " + e;
        }
    }

    private Map<MetadataType, MetadataComponent> readSSTableMetadata(Descriptor desc) {
        Map<MetadataType, MetadataComponent> metadata;
        try {
            metadata = desc.getMetadataSerializer().deserialize(desc, EnumSet.allOf(MetadataType.class));
        }
        catch (IOException e) {
            this.error("Failed to deserialize metadata for sstable %s: %s", desc, e.toString());
            return null;
        }
        return metadata;
    }

    private void writeNewMetadata(Descriptor desc, Map<MetadataType, MetadataComponent> newMetadata) {
        String file = desc.filenameFor(Component.STATS);
        this.info.accept(String.format("  Writing new metadata file %s", file));
        try {
            desc.getMetadataSerializer().rewriteSSTableMetadata(desc, newMetadata);
        }
        catch (IOException e) {
            this.error("Failed to write metadata component for %s: %s", file, e.toString());
            throw new RuntimeException(e);
        }
    }

    static class AutomaticHeaderFix
    extends SSTableHeaderFix {
        AutomaticHeaderFix(Builder builder) {
            super(builder);
        }

        @Override
        public void prepare() {
            this.info.accept("Scanning all data directories...");
            for (Directories.DataDirectory dataDirectory : Directories.dataDirectories) {
                this.scanDataDirectory(dataDirectory);
            }
            this.info.accept("Finished scanning all data directories...");
        }

        private void scanDataDirectory(Directories.DataDirectory dataDirectory) {
            this.info.accept(String.format("Scanning data directory %s", dataDirectory.location));
            org.apache.cassandra.io.util.File[] ksDirs = dataDirectory.location.tryList();
            if (ksDirs == null) {
                return;
            }
            for (org.apache.cassandra.io.util.File ksDir : ksDirs) {
                org.apache.cassandra.io.util.File[] tabDirs;
                String name;
                if (!ksDir.isDirectory() || !ksDir.isReadable() || SchemaConstants.isLocalSystemKeyspace(name = ksDir.name()) || SchemaConstants.isReplicatedSystemKeyspace(name) || (tabDirs = ksDir.tryList()) == null) continue;
                for (org.apache.cassandra.io.util.File tabDir : tabDirs) {
                    if (!tabDir.isDirectory() || !tabDir.isReadable()) continue;
                    this.processFileOrDirectory(tabDir.toPath());
                }
            }
        }
    }

    static class ManualHeaderFix
    extends SSTableHeaderFix {
        private final List<Path> paths;

        ManualHeaderFix(Builder builder) {
            super(builder);
            this.paths = builder.paths;
        }

        @Override
        public void prepare() {
            this.paths.forEach(this::processFileOrDirectory);
        }
    }

    public static class Builder {
        private final List<Path> paths = new ArrayList<Path>();
        private final List<Descriptor> descriptors = new ArrayList<Descriptor>();
        private Consumer<String> info = ln -> logger.info("{}", ln);
        private Consumer<String> warn = ln -> logger.warn("{}", ln);
        private Consumer<String> error = ln -> logger.error("{}", ln);
        private boolean dryRun;
        private Supplier<Function<Descriptor, TableMetadata>> schemaCallback = () -> null;

        private Builder() {
        }

        public Builder dryRun() {
            this.dryRun = true;
            return this;
        }

        public Builder info(Consumer<String> output) {
            this.info = output;
            return this;
        }

        public Builder warn(Consumer<String> warn) {
            this.warn = warn;
            return this;
        }

        public Builder error(Consumer<String> error) {
            this.error = error;
            return this;
        }

        public Builder withPath(Path path) {
            this.paths.add(path);
            return this;
        }

        public Builder withDescriptor(Descriptor descriptor) {
            this.descriptors.add(descriptor);
            return this;
        }

        public Builder schemaCallback(Supplier<Function<Descriptor, TableMetadata>> schemaCallback) {
            this.schemaCallback = schemaCallback;
            return this;
        }

        public SSTableHeaderFix build() {
            if (this.paths.isEmpty() && this.descriptors.isEmpty()) {
                return new AutomaticHeaderFix(this);
            }
            return new ManualHeaderFix(this);
        }

        public Builder logToList(List<String> output) {
            return this.info(ln -> output.add("INFO  " + ln)).warn(ln -> output.add("WARN  " + ln)).error(ln -> output.add("ERROR " + ln));
        }
    }
}

