/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.SnapshotCommand;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.utils.ExecutorUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiagnosticSnapshotService {
    private static final Logger logger = LoggerFactory.getLogger(DiagnosticSnapshotService.class);
    public static final DiagnosticSnapshotService instance = new DiagnosticSnapshotService(Executors.newSingleThreadExecutor(new NamedThreadFactory("DiagnosticSnapshot")));
    public static final String REPAIRED_DATA_MISMATCH_SNAPSHOT_PREFIX = "RepairedDataMismatch-";
    public static final String DUPLICATE_ROWS_DETECTED_SNAPSHOT_PREFIX = "DuplicateRows-";
    private final Executor executor;
    private static final long SNAPSHOT_INTERVAL_NANOS = TimeUnit.MINUTES.toNanos(1L);
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.BASIC_ISO_DATE;
    private final ConcurrentHashMap<TableId, AtomicLong> lastSnapshotTimes = new ConcurrentHashMap();

    private DiagnosticSnapshotService(Executor executor) {
        this.executor = executor;
    }

    public static void duplicateRows(TableMetadata metadata, Iterable<InetAddressAndPort> replicas) {
        instance.maybeTriggerSnapshot(metadata, DUPLICATE_ROWS_DETECTED_SNAPSHOT_PREFIX, replicas);
    }

    public static void repairedDataMismatch(TableMetadata metadata, Iterable<InetAddressAndPort> replicas) {
        instance.maybeTriggerSnapshot(metadata, REPAIRED_DATA_MISMATCH_SNAPSHOT_PREFIX, replicas);
    }

    public static boolean isDiagnosticSnapshotRequest(SnapshotCommand command) {
        return command.snapshot_name.startsWith(REPAIRED_DATA_MISMATCH_SNAPSHOT_PREFIX) || command.snapshot_name.startsWith(DUPLICATE_ROWS_DETECTED_SNAPSHOT_PREFIX);
    }

    public static void snapshot(SnapshotCommand command, InetAddressAndPort initiator) {
        Preconditions.checkArgument((boolean)DiagnosticSnapshotService.isDiagnosticSnapshotRequest(command));
        instance.maybeSnapshot(command, initiator);
    }

    public static String getSnapshotName(String prefix) {
        return String.format("%s%s", prefix, DATE_FORMAT.format(LocalDate.now()));
    }

    @VisibleForTesting
    public void shutdownAndWait(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
        ExecutorUtils.shutdownNowAndWait(timeout, unit, this.executor);
    }

    private void maybeTriggerSnapshot(TableMetadata metadata, String prefix, Iterable<InetAddressAndPort> endpoints) {
        long interval;
        AtomicLong cached;
        long last;
        long now = System.nanoTime();
        if (now - (last = (cached = this.lastSnapshotTimes.computeIfAbsent(metadata.id, u -> new AtomicLong(0L))).get()) > (interval = Long.getLong("cassandra.diagnostic_snapshot_interval_nanos", SNAPSHOT_INTERVAL_NANOS).longValue()) && cached.compareAndSet(last, now)) {
            Message<SnapshotCommand> msg = Message.out(Verb.SNAPSHOT_REQ, new SnapshotCommand(metadata.keyspace, metadata.name, DiagnosticSnapshotService.getSnapshotName(prefix), false));
            for (InetAddressAndPort replica : endpoints) {
                MessagingService.instance().send(msg, replica);
            }
        } else {
            logger.debug("Diagnostic snapshot request dropped due to throttling");
        }
    }

    private void maybeSnapshot(SnapshotCommand command, InetAddressAndPort initiator) {
        this.executor.execute(new DiagnosticSnapshotTask(command, initiator));
    }

    private static class DiagnosticSnapshotTask
    implements Runnable {
        final SnapshotCommand command;
        final InetAddressAndPort from;

        DiagnosticSnapshotTask(SnapshotCommand command, InetAddressAndPort from) {
            this.command = command;
            this.from = from;
        }

        @Override
        public void run() {
            try {
                Keyspace ks = Keyspace.open(this.command.keyspace);
                if (ks == null) {
                    logger.info("Snapshot request received from {} for {}.{} but keyspace not found", new Object[]{this.from, this.command.keyspace, this.command.column_family});
                    return;
                }
                ColumnFamilyStore cfs = ks.getColumnFamilyStore(this.command.column_family);
                if (cfs.snapshotExists(this.command.snapshot_name)) {
                    logger.info("Received diagnostic snapshot request from {} for {}.{}, but snapshot with tag {} already exists", new Object[]{this.from, this.command.keyspace, this.command.column_family, this.command.snapshot_name});
                    return;
                }
                logger.info("Creating snapshot requested by {} of {}.{} tag: {}", new Object[]{this.from, this.command.keyspace, this.command.column_family, this.command.snapshot_name});
                cfs.snapshot(this.command.snapshot_name);
            }
            catch (IllegalArgumentException e) {
                logger.warn("Snapshot request received from {} for {}.{} but CFS not found", new Object[]{this.from, this.command.keyspace, this.command.column_family});
            }
        }
    }
}

