/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.function.LongPredicate;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.sstable.IndexSummary;
import org.apache.cassandra.io.sstable.KeyIterator;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.metadata.MetadataType;
import org.apache.cassandra.io.sstable.metadata.ValidationMetadata;
import org.apache.cassandra.io.util.DataIntegrityMetadata;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.BloomFilter;
import org.apache.cassandra.utils.BloomFilterSerializer;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.OutputHandler;
import org.apache.cassandra.utils.UUIDGen;

public class Verifier
implements Closeable {
    private final ColumnFamilyStore cfs;
    private final SSTableReader sstable;
    private final CompactionController controller;
    private final ReadWriteLock fileAccessLock;
    private final RandomAccessReader dataFile;
    private final RandomAccessReader indexFile;
    private final VerifyInfo verifyInfo;
    private final RowIndexEntry.IndexSerializer rowIndexEntrySerializer;
    private final Options options;
    private final boolean isOffline;
    private final Function<String, ? extends Collection<Range<Token>>> tokenLookup;
    private int goodRows;
    private final OutputHandler outputHandler;
    private DataIntegrityMetadata.FileDigestValidator validator;

    public Verifier(ColumnFamilyStore cfs, SSTableReader sstable, boolean isOffline, Options options) {
        this(cfs, sstable, new OutputHandler.LogOutput(), isOffline, options);
    }

    public Verifier(ColumnFamilyStore cfs, SSTableReader sstable, OutputHandler outputHandler, boolean isOffline, Options options) {
        this.cfs = cfs;
        this.sstable = sstable;
        this.outputHandler = outputHandler;
        this.rowIndexEntrySerializer = sstable.descriptor.version.getSSTableFormat().getIndexSerializer(cfs.metadata(), sstable.descriptor.version, sstable.header);
        this.controller = new VerifyController(cfs);
        this.fileAccessLock = new ReentrantReadWriteLock();
        this.dataFile = isOffline ? sstable.openDataReader() : sstable.openDataReader(CompactionManager.instance.getRateLimiter());
        this.indexFile = RandomAccessReader.open(new File(sstable.descriptor.filenameFor(Component.PRIMARY_INDEX)));
        this.verifyInfo = new VerifyInfo(this.dataFile, sstable, this.fileAccessLock.readLock());
        this.options = options;
        this.isOffline = isOffline;
        this.tokenLookup = options.tokenLookup;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void verify() {
        DecoratedKey key;
        Object sstableMetadata;
        boolean extended = this.options.extendedVerification;
        long rowStart = 0L;
        this.outputHandler.output(String.format("Verifying %s (%s)", this.sstable, FBUtilities.prettyPrintMemory(this.dataFile.length())));
        if (this.options.checkVersion && !this.sstable.descriptor.version.isLatestVersion()) {
            String msg = String.format("%s is not the latest version, run upgradesstables", this.sstable);
            this.outputHandler.output(msg);
            throw new RuntimeException(msg);
        }
        this.outputHandler.output(String.format("Deserializing sstable metadata for %s ", this.sstable));
        try {
            EnumSet<MetadataType> types = EnumSet.of(MetadataType.VALIDATION, MetadataType.STATS, MetadataType.HEADER);
            sstableMetadata = this.sstable.descriptor.getMetadataSerializer().deserialize(this.sstable.descriptor, types);
            if (sstableMetadata.containsKey((Object)MetadataType.VALIDATION) && !((ValidationMetadata)sstableMetadata.get((Object)((Object)MetadataType.VALIDATION))).partitioner.equals(this.sstable.getPartitioner().getClass().getCanonicalName())) {
                throw new IOException("Partitioner does not match validation metadata");
            }
        }
        catch (Throwable t) {
            this.outputHandler.warn(t.getMessage());
            this.markAndThrow(false);
        }
        try {
            this.outputHandler.debug("Deserializing index for " + this.sstable);
            this.deserializeIndex(this.sstable);
        }
        catch (Throwable t) {
            this.outputHandler.warn(t.getMessage());
            this.markAndThrow();
        }
        try {
            this.outputHandler.debug("Deserializing index summary for " + this.sstable);
            this.deserializeIndexSummary(this.sstable);
        }
        catch (Throwable t) {
            this.outputHandler.output("Index summary is corrupt - if it is removed it will get rebuilt on startup " + this.sstable.descriptor.filenameFor(Component.SUMMARY));
            this.outputHandler.warn(t.getMessage());
            this.markAndThrow(false);
        }
        try {
            this.outputHandler.debug("Deserializing bloom filter for " + this.sstable);
            this.deserializeBloomFilter(this.sstable);
        }
        catch (Throwable t) {
            this.outputHandler.warn(t.getMessage());
            this.markAndThrow();
        }
        if (this.options.checkOwnsTokens && !this.isOffline) {
            this.outputHandler.debug("Checking that all tokens are owned by the current node");
            try {
                KeyIterator iter = new KeyIterator(this.sstable.descriptor, this.sstable.metadata());
                sstableMetadata = null;
                try {
                    List<Range<Token>> ownedRanges = Range.normalize(this.tokenLookup.apply(this.cfs.metadata.keyspace));
                    if (ownedRanges.isEmpty()) {
                        return;
                    }
                    RangeOwnHelper rangeOwnHelper = new RangeOwnHelper(ownedRanges);
                    while (iter.hasNext()) {
                        key = (DecoratedKey)iter.next();
                        rangeOwnHelper.validate(key);
                    }
                }
                catch (Throwable ownedRanges) {
                    sstableMetadata = ownedRanges;
                    throw ownedRanges;
                }
                finally {
                    if (iter != null) {
                        if (sstableMetadata != null) {
                            try {
                                iter.close();
                            }
                            catch (Throwable throwable) {
                                ((Throwable)sstableMetadata).addSuppressed(throwable);
                            }
                        } else {
                            iter.close();
                        }
                    }
                }
            }
            catch (Throwable t) {
                this.outputHandler.warn(t.getMessage());
                this.markAndThrow();
            }
        }
        if (this.options.quick) {
            return;
        }
        this.outputHandler.output(String.format("Checking computed hash of %s ", this.sstable));
        try {
            this.validator = null;
            if (new File(this.sstable.descriptor.filenameFor(Component.DIGEST)).exists()) {
                this.validator = DataIntegrityMetadata.fileDigestValidator(this.sstable.descriptor);
                this.validator.validate();
            } else {
                this.outputHandler.output("Data digest missing, assuming extended verification of disk values");
                extended = true;
            }
        }
        catch (IOException e) {
            this.outputHandler.warn(e.getMessage());
            this.markAndThrow();
        }
        finally {
            FileUtils.closeQuietly(this.validator);
        }
        if (!extended) {
            return;
        }
        this.outputHandler.output("Extended Verify requested, proceeding to inspect values");
        try {
            ByteBuffer nextIndexKey = ByteBufferUtil.readWithShortLength(this.indexFile);
            long firstRowPositionFromIndex = this.rowIndexEntrySerializer.deserializePositionAndSkip(this.indexFile);
            if (firstRowPositionFromIndex != 0L) {
                this.markAndThrow();
            }
            List<Range<Token>> ownedRanges = this.isOffline ? Collections.emptyList() : Range.normalize(this.tokenLookup.apply(this.cfs.metadata().keyspace));
            RangeOwnHelper rangeOwnHelper = new RangeOwnHelper(ownedRanges);
            DecoratedKey prevKey = null;
            while (!this.dataFile.isEOF()) {
                if (this.verifyInfo.isStopRequested()) {
                    throw new CompactionInterruptedException(this.verifyInfo.getCompactionInfo());
                }
                rowStart = this.dataFile.getFilePointer();
                this.outputHandler.debug("Reading row at " + rowStart);
                key = null;
                try {
                    key = this.sstable.decorateKey(ByteBufferUtil.readWithShortLength(this.dataFile));
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                }
                if (this.options.checkOwnsTokens && ownedRanges.size() > 0) {
                    try {
                        rangeOwnHelper.validate(key);
                    }
                    catch (Throwable t) {
                        this.outputHandler.warn(String.format("Key %s in sstable %s not owned by local ranges %s", key, this.sstable, ownedRanges), t);
                        this.markAndThrow();
                    }
                }
                ByteBuffer currentIndexKey = nextIndexKey;
                long nextRowPositionFromIndex = 0L;
                try {
                    nextIndexKey = this.indexFile.isEOF() ? null : ByteBufferUtil.readWithShortLength(this.indexFile);
                    nextRowPositionFromIndex = this.indexFile.isEOF() ? this.dataFile.length() : this.rowIndexEntrySerializer.deserializePositionAndSkip(this.indexFile);
                }
                catch (Throwable th) {
                    this.markAndThrow();
                }
                long dataStart = this.dataFile.getFilePointer();
                long dataStartFromIndex = currentIndexKey == null ? -1L : rowStart + 2L + (long)currentIndexKey.remaining();
                long dataSize = nextRowPositionFromIndex - dataStartFromIndex;
                String keyName = key == null ? "(unreadable key)" : ByteBufferUtil.bytesToHex(key.getKey());
                this.outputHandler.debug(String.format("row %s is %s", keyName, FBUtilities.prettyPrintMemory(dataSize)));
                assert (currentIndexKey != null || this.indexFile.isEOF());
                try {
                    if (key == null || dataSize > this.dataFile.length()) {
                        this.markAndThrow();
                    }
                    SSTableIdentityIterator iterator = SSTableIdentityIterator.create(this.sstable, this.dataFile, key);
                    Throwable throwable = null;
                    if (iterator != null) {
                        if (throwable != null) {
                            try {
                                iterator.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        } else {
                            iterator.close();
                        }
                    }
                    if (prevKey != null && prevKey.compareTo(key) > 0 || !key.getKey().equals(currentIndexKey) || dataStart != dataStartFromIndex) {
                        this.markAndThrow();
                    }
                    ++this.goodRows;
                    prevKey = key;
                    this.outputHandler.debug(String.format("Row %s at %s valid, moving to next row at %s ", this.goodRows, rowStart, nextRowPositionFromIndex));
                    this.dataFile.seek(nextRowPositionFromIndex);
                }
                catch (Throwable th) {
                    this.markAndThrow();
                }
            }
        }
        catch (Throwable t) {
            throw Throwables.propagate((Throwable)t);
        }
        finally {
            this.controller.close();
        }
        this.outputHandler.output("Verify of " + this.sstable + " succeeded. All " + this.goodRows + " rows read successfully");
    }

    private void deserializeIndex(SSTableReader sstable) throws IOException {
        try (RandomAccessReader primaryIndex = RandomAccessReader.open(new File(sstable.descriptor.filenameFor(Component.PRIMARY_INDEX)));){
            long indexSize = primaryIndex.length();
            while (primaryIndex.getFilePointer() != indexSize) {
                ByteBuffer key = ByteBufferUtil.readWithShortLength(primaryIndex);
                RowIndexEntry.Serializer.skip(primaryIndex, sstable.descriptor.version);
            }
        }
    }

    private void deserializeIndexSummary(SSTableReader sstable) throws IOException {
        File file = new File(sstable.descriptor.filenameFor(Component.SUMMARY));
        TableMetadata metadata = this.cfs.metadata();
        try (DataInputStream iStream = new DataInputStream(Files.newInputStream(file.toPath(), new OpenOption[0]));
             IndexSummary indexSummary = IndexSummary.serializer.deserialize(iStream, this.cfs.getPartitioner(), metadata.params.minIndexInterval, metadata.params.maxIndexInterval);){
            ByteBufferUtil.readWithLength(iStream);
            ByteBufferUtil.readWithLength(iStream);
        }
    }

    private void deserializeBloomFilter(SSTableReader sstable) throws IOException {
        try (DataInputStream stream = new DataInputStream(new BufferedInputStream(Files.newInputStream(Paths.get(sstable.descriptor.filenameFor(Component.FILTER), new String[0]), new OpenOption[0])));){
            BloomFilter bf = BloomFilterSerializer.deserialize(stream, sstable.descriptor.version.hasOldBfFormat());
            Throwable throwable = null;
            if (bf != null) {
                if (throwable != null) {
                    try {
                        bf.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                } else {
                    bf.close();
                }
            }
        }
    }

    @Override
    public void close() {
        this.fileAccessLock.writeLock().lock();
        try {
            FileUtils.closeQuietly(this.dataFile);
            FileUtils.closeQuietly(this.indexFile);
        }
        finally {
            this.fileAccessLock.writeLock().unlock();
        }
    }

    private void throwIfFatal(Throwable th) {
        if (th instanceof Error && !(th instanceof AssertionError) && !(th instanceof IOError)) {
            throw (Error)th;
        }
    }

    private void markAndThrow() {
        this.markAndThrow(true);
    }

    private void markAndThrow(boolean mutateRepaired) {
        if (mutateRepaired && this.options.mutateRepairStatus) {
            try {
                this.sstable.mutateRepairedAndReload(0L, this.sstable.getPendingRepair(), this.sstable.isTransient());
                this.cfs.getTracker().notifySSTableRepairedStatusChanged(Collections.singleton(this.sstable));
            }
            catch (IOException ioe) {
                this.outputHandler.output("Error mutating repairedAt for SSTable " + this.sstable.getFilename() + ", as part of markAndThrow");
            }
        }
        Exception e = new Exception(String.format("Invalid SSTable %s, please force %srepair", this.sstable.getFilename(), mutateRepaired && this.options.mutateRepairStatus ? "" : "a full "));
        if (this.options.invokeDiskFailurePolicy) {
            throw new CorruptSSTableException((Throwable)e, this.sstable.getFilename());
        }
        throw new RuntimeException(e);
    }

    public CompactionInfo.Holder getVerifyInfo() {
        return this.verifyInfo;
    }

    public static Options.Builder options() {
        return new Options.Builder();
    }

    public static class Options {
        public final boolean invokeDiskFailurePolicy;
        public final boolean extendedVerification;
        public final boolean checkVersion;
        public final boolean mutateRepairStatus;
        public final boolean checkOwnsTokens;
        public final boolean quick;
        public final Function<String, ? extends Collection<Range<Token>>> tokenLookup;

        private Options(boolean invokeDiskFailurePolicy, boolean extendedVerification, boolean checkVersion, boolean mutateRepairStatus, boolean checkOwnsTokens, boolean quick, Function<String, ? extends Collection<Range<Token>>> tokenLookup) {
            this.invokeDiskFailurePolicy = invokeDiskFailurePolicy;
            this.extendedVerification = extendedVerification;
            this.checkVersion = checkVersion;
            this.mutateRepairStatus = mutateRepairStatus;
            this.checkOwnsTokens = checkOwnsTokens;
            this.quick = quick;
            this.tokenLookup = tokenLookup;
        }

        public String toString() {
            return "Options{invokeDiskFailurePolicy=" + this.invokeDiskFailurePolicy + ", extendedVerification=" + this.extendedVerification + ", checkVersion=" + this.checkVersion + ", mutateRepairStatus=" + this.mutateRepairStatus + ", checkOwnsTokens=" + this.checkOwnsTokens + ", quick=" + this.quick + '}';
        }

        public static class Builder {
            private boolean invokeDiskFailurePolicy = false;
            private boolean extendedVerification = false;
            private boolean checkVersion = false;
            private boolean mutateRepairStatus = false;
            private boolean checkOwnsTokens = false;
            private boolean quick = false;
            private Function<String, ? extends Collection<Range<Token>>> tokenLookup = StorageService.instance::getLocalAndPendingRanges;

            public Builder invokeDiskFailurePolicy(boolean param) {
                this.invokeDiskFailurePolicy = param;
                return this;
            }

            public Builder extendedVerification(boolean param) {
                this.extendedVerification = param;
                return this;
            }

            public Builder checkVersion(boolean param) {
                this.checkVersion = param;
                return this;
            }

            public Builder mutateRepairStatus(boolean param) {
                this.mutateRepairStatus = param;
                return this;
            }

            public Builder checkOwnsTokens(boolean param) {
                this.checkOwnsTokens = param;
                return this;
            }

            public Builder quick(boolean param) {
                this.quick = param;
                return this;
            }

            public Builder tokenLookup(Function<String, ? extends Collection<Range<Token>>> tokenLookup) {
                this.tokenLookup = tokenLookup;
                return this;
            }

            public Options build() {
                return new Options(this.invokeDiskFailurePolicy, this.extendedVerification, this.checkVersion, this.mutateRepairStatus, this.checkOwnsTokens, this.quick, this.tokenLookup);
            }
        }
    }

    private static class VerifyController
    extends CompactionController {
        public VerifyController(ColumnFamilyStore cfs) {
            super(cfs, Integer.MAX_VALUE);
        }

        @Override
        public LongPredicate getPurgeEvaluator(DecoratedKey key) {
            return time -> false;
        }
    }

    private static class VerifyInfo
    extends CompactionInfo.Holder {
        private final RandomAccessReader dataFile;
        private final SSTableReader sstable;
        private final UUID verificationCompactionId;
        private final Lock fileReadLock;

        public VerifyInfo(RandomAccessReader dataFile, SSTableReader sstable, Lock fileReadLock) {
            this.dataFile = dataFile;
            this.sstable = sstable;
            this.fileReadLock = fileReadLock;
            this.verificationCompactionId = UUIDGen.getTimeUUID();
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            this.fileReadLock.lock();
            try {
                CompactionInfo compactionInfo = new CompactionInfo(this.sstable.metadata(), OperationType.VERIFY, this.dataFile.getFilePointer(), this.dataFile.length(), this.verificationCompactionId, (Collection<SSTableReader>)ImmutableSet.of((Object)this.sstable));
                return compactionInfo;
            }
            catch (Exception e) {
                throw new RuntimeException();
            }
            finally {
                this.fileReadLock.unlock();
            }
        }

        @Override
        public boolean isGlobal() {
            return false;
        }
    }

    @VisibleForTesting
    public static class RangeOwnHelper {
        private final List<Range<Token>> normalizedRanges;
        private int rangeIndex = 0;
        private DecoratedKey lastKey;

        public RangeOwnHelper(List<Range<Token>> normalizedRanges) {
            this.normalizedRanges = normalizedRanges;
            Range.assertNormalized(normalizedRanges);
        }

        public void validate(DecoratedKey key) {
            if (!this.check(key)) {
                throw new RuntimeException("Key " + key + " is not contained in the given ranges");
            }
        }

        public boolean check(DecoratedKey key) {
            assert (this.lastKey == null || key.compareTo(this.lastKey) > 0);
            this.lastKey = key;
            if (this.normalizedRanges.isEmpty()) {
                return true;
            }
            if (this.rangeIndex > this.normalizedRanges.size() - 1) {
                throw new IllegalStateException("RangeOwnHelper can only be used to find the first out-of-range-token");
            }
            while (!this.normalizedRanges.get(this.rangeIndex).contains(key.getToken())) {
                ++this.rangeIndex;
                if (this.rangeIndex <= this.normalizedRanges.size() - 1) continue;
                return false;
            }
            return true;
        }
    }
}

