/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;
import com.google.common.primitives.Ints;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LeveledGenerations {
    private static final Logger logger = LoggerFactory.getLogger(LeveledGenerations.class);
    private final boolean strictLCSChecksTest = Boolean.getBoolean("cassandra.test.strict_lcs_checks");
    static final int MAX_LEVEL_COUNT = (int)Math.log10(1.0E9);
    private final Map<SSTableReader, SSTableReader> allSSTables = new HashMap<SSTableReader, SSTableReader>();
    private final Set<SSTableReader> l0 = new HashSet<SSTableReader>();
    private static long lastOverlapCheck = System.nanoTime();
    private final TreeSet<SSTableReader>[] levels = new TreeSet[MAX_LEVEL_COUNT - 1];
    private static final Comparator<SSTableReader> nonL0Comparator = (o1, o2) -> {
        int cmp = SSTableReader.sstableComparator.compare((SSTableReader)o1, (SSTableReader)o2);
        if (cmp == 0) {
            cmp = Ints.compare((int)o1.descriptor.generation, (int)o2.descriptor.generation);
        }
        return cmp;
    };

    LeveledGenerations() {
        for (int i = 0; i < MAX_LEVEL_COUNT - 1; ++i) {
            this.levels[i] = new TreeSet<SSTableReader>(nonL0Comparator);
        }
    }

    Set<SSTableReader> get(int level) {
        if (level > this.levelCount() - 1 || level < 0) {
            throw new ArrayIndexOutOfBoundsException("Invalid generation " + level + " - maximum is " + (this.levelCount() - 1));
        }
        if (level == 0) {
            return this.l0;
        }
        return this.levels[level - 1];
    }

    int levelCount() {
        return this.levels.length + 1;
    }

    void addAll(Iterable<SSTableReader> readers) {
        this.logDistribution();
        for (SSTableReader sstable : readers) {
            assert (sstable.getSSTableLevel() < this.levelCount()) : "Invalid level " + sstable.getSSTableLevel() + " out of " + (this.levelCount() - 1);
            int existingLevel = this.getLevelIfExists(sstable);
            if (existingLevel != -1) {
                if (sstable.getSSTableLevel() != existingLevel) {
                    logger.error("SSTable {} on the wrong level in the manifest - {} instead of {} as recorded in the sstable metadata, removing from level {}", new Object[]{sstable, existingLevel, sstable.getSSTableLevel(), existingLevel});
                    if (this.strictLCSChecksTest) {
                        throw new AssertionError((Object)("SSTable not in matching level in manifest: " + sstable + ": " + existingLevel + " != " + sstable.getSSTableLevel()));
                    }
                } else {
                    logger.info("Manifest already contains {} in level {} - replacing instance", (Object)sstable, (Object)existingLevel);
                }
                this.get(existingLevel).remove(sstable);
                this.allSSTables.remove(sstable);
            }
            this.allSSTables.put(sstable, sstable);
            if (sstable.getSSTableLevel() == 0) {
                this.l0.add(sstable);
                continue;
            }
            TreeSet<SSTableReader> level = this.levels[sstable.getSSTableLevel() - 1];
            SSTableReader after = level.ceiling(sstable);
            SSTableReader before = level.floor(sstable);
            if (before != null && before.last.compareTo(sstable.first) >= 0 || after != null && after.first.compareTo(sstable.last) <= 0) {
                if (this.strictLCSChecksTest) {
                    throw new AssertionError((Object)("Got unexpected overlap in level " + sstable.getSSTableLevel()));
                }
                this.sendToL0(sstable);
                continue;
            }
            level.add(sstable);
        }
        this.maybeVerifyLevels();
    }

    private void sendToL0(SSTableReader sstable) {
        try {
            sstable.mutateLevelAndReload(0);
        }
        catch (IOException e) {
            logger.error("Failed mutating sstable metadata for {} - adding it to L0 to avoid overlap. Marking suspect", (Object)sstable, (Object)e);
            sstable.markSuspect();
        }
        this.l0.add(sstable);
    }

    private int getLevelIfExists(SSTableReader sstable) {
        for (int i = 0; i < this.levelCount(); ++i) {
            if (!this.get(i).contains(sstable)) continue;
            return i;
        }
        return -1;
    }

    int remove(Collection<SSTableReader> readers) {
        int minLevel = Integer.MAX_VALUE;
        for (SSTableReader sstable : readers) {
            int level = sstable.getSSTableLevel();
            minLevel = Math.min(minLevel, level);
            SSTableReader versionInManifest = this.allSSTables.get(sstable);
            if (versionInManifest == null) continue;
            this.get(level).remove(versionInManifest);
            this.allSSTables.remove(versionInManifest);
        }
        return minLevel;
    }

    int[] getAllLevelSize() {
        int[] counts = new int[this.levelCount()];
        for (int i = 0; i < this.levelCount(); ++i) {
            counts[i] = this.get(i).size();
        }
        return counts;
    }

    Set<SSTableReader> allSSTables() {
        ImmutableSet.Builder builder = ImmutableSet.builder();
        builder.addAll(this.l0);
        for (TreeSet<SSTableReader> sstables : this.levels) {
            builder.addAll(sstables);
        }
        return builder.build();
    }

    Iterator<SSTableReader> wrappingIterator(int lvl, SSTableReader lastCompactedSSTable) {
        assert (lvl > 0);
        TreeSet<SSTableReader> level = this.levels[lvl - 1];
        if (level.isEmpty()) {
            return Collections.emptyIterator();
        }
        if (lastCompactedSSTable == null) {
            return level.iterator();
        }
        PeekingIterator tail = Iterators.peekingIterator(level.tailSet(lastCompactedSSTable).iterator());
        SSTableReader pivot = null;
        while (tail.hasNext()) {
            SSTableReader potentialPivot = (SSTableReader)tail.peek();
            if (potentialPivot.first.compareTo(lastCompactedSSTable.last) > 0) {
                pivot = potentialPivot;
                break;
            }
            tail.next();
        }
        if (pivot == null) {
            return level.iterator();
        }
        return Iterators.concat((Iterator)tail, level.headSet(pivot, false).iterator());
    }

    void logDistribution() {
        if (logger.isTraceEnabled()) {
            for (int i = 0; i < this.levelCount(); ++i) {
                Set<SSTableReader> level = this.get(i);
                if (level.isEmpty()) continue;
                logger.trace("L{} contains {} SSTables ({}) in {}", new Object[]{i, level.size(), FBUtilities.prettyPrintMemory(SSTableReader.getTotalBytes(level)), this});
            }
        }
    }

    Set<SSTableReader>[] snapshot() {
        Set[] levelsCopy = new Set[this.levelCount()];
        for (int i = 0; i < this.levelCount(); ++i) {
            levelsCopy[i] = ImmutableSet.copyOf(this.get(i));
        }
        return levelsCopy;
    }

    private void maybeVerifyLevels() {
        if (!this.strictLCSChecksTest || System.nanoTime() - lastOverlapCheck <= TimeUnit.NANOSECONDS.convert(5L, TimeUnit.SECONDS)) {
            return;
        }
        logger.info("LCS verifying levels");
        lastOverlapCheck = System.nanoTime();
        for (int i = 1; i < this.levelCount(); ++i) {
            SSTableReader prev = null;
            for (SSTableReader sstable : this.get(i)) {
                assert (prev == null || prev.last.compareTo(sstable.first) < 0);
                prev = sstable;
                for (int j = 0; j < this.levelCount(); ++j) {
                    if (i != j) assert (!this.get(j).contains(sstable));
                }
            }
        }
    }

    void newLevel(SSTableReader sstable, int oldLevel) {
        SSTableReader versionInManifest = this.allSSTables.remove(sstable);
        boolean removed = false;
        if (versionInManifest != null) {
            removed = this.get(oldLevel).remove(versionInManifest);
        }
        if (!removed) {
            logger.warn("Could not remove " + sstable + " from " + oldLevel);
        }
        this.addAll(Collections.singleton(sstable));
    }
}

