/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.functions.Function;
import org.apache.cassandra.cql3.functions.FunctionName;
import org.apache.cassandra.cql3.functions.UDAggregate;
import org.apache.cassandra.cql3.functions.UDFunction;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.KeyspaceNotDefinedException;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.UserType;
import org.apache.cassandra.db.virtual.VirtualKeyspaceRegistry;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.exceptions.UnknownTableException;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.locator.LocalStrategy;
import org.apache.cassandra.schema.Functions;
import org.apache.cassandra.schema.KeyspaceMetadata;
import org.apache.cassandra.schema.Keyspaces;
import org.apache.cassandra.schema.SchemaChangeListener;
import org.apache.cassandra.schema.SchemaConstants;
import org.apache.cassandra.schema.SchemaDiagnostics;
import org.apache.cassandra.schema.SchemaKeyspace;
import org.apache.cassandra.schema.SchemaTransformation;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.schema.TableMetadataRef;
import org.apache.cassandra.schema.Tables;
import org.apache.cassandra.schema.Types;
import org.apache.cassandra.schema.ViewMetadata;
import org.apache.cassandra.schema.Views;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.Pair;
import org.cliffc.high_scale_lib.NonBlockingHashMap;

public final class Schema {
    public static final Schema instance = new Schema();
    private volatile Keyspaces keyspaces = Keyspaces.none();
    private final Map<TableId, TableMetadataRef> metadataRefs = new NonBlockingHashMap();
    private final Map<Pair<String, String>, TableMetadataRef> indexMetadataRefs = new NonBlockingHashMap();
    private final Map<String, Keyspace> keyspaceInstances = new NonBlockingHashMap();
    private volatile UUID version;
    private final List<SchemaChangeListener> changeListeners = new CopyOnWriteArrayList<SchemaChangeListener>();

    private Schema() {
        if (DatabaseDescriptor.isDaemonInitialized() || DatabaseDescriptor.isToolInitialized()) {
            this.load(SchemaKeyspace.metadata());
            this.load(SystemKeyspace.metadata());
        }
    }

    public void loadFromDisk() {
        this.loadFromDisk(true);
    }

    public void loadFromDisk(boolean updateVersion) {
        SchemaDiagnostics.schemataLoading(this);
        SchemaKeyspace.fetchNonSystemKeyspaces().forEach(this::load);
        if (updateVersion) {
            this.updateVersion();
        }
        SchemaDiagnostics.schemataLoaded(this);
    }

    public synchronized void load(KeyspaceMetadata ksm) {
        KeyspaceMetadata previous = this.keyspaces.getNullable(ksm.name);
        if (previous == null) {
            this.loadNew(ksm);
        } else {
            this.reload(previous, ksm);
        }
        this.keyspaces = this.keyspaces.withAddedOrUpdated(ksm);
    }

    private void loadNew(KeyspaceMetadata ksm) {
        ksm.tablesAndViews().forEach(metadata -> this.metadataRefs.put(metadata.id, new TableMetadataRef((TableMetadata)metadata)));
        ksm.tables.indexTables().forEach((name, metadata) -> this.indexMetadataRefs.put(Pair.create(ksm.name, name), new TableMetadataRef((TableMetadata)metadata)));
        SchemaDiagnostics.metadataInitialized(this, ksm);
    }

    private void reload(KeyspaceMetadata previous, KeyspaceMetadata updated) {
        Keyspace keyspace = this.getKeyspaceInstance(updated.name);
        if (null != keyspace) {
            keyspace.setMetadata(updated);
        }
        Tables.TablesDiff tablesDiff = Tables.diff(previous.tables, updated.tables);
        Views.ViewsDiff viewsDiff = Views.diff(previous.views, updated.views);
        MapDifference<String, TableMetadata> indexesDiff = previous.tables.indexesDiff(updated.tables);
        ((Tables)tablesDiff.dropped).forEach(table -> this.metadataRefs.remove(table.id));
        ((Views)viewsDiff.dropped).forEach(view -> this.metadataRefs.remove(view.metadata.id));
        indexesDiff.entriesOnlyOnLeft().values().forEach(indexTable -> this.indexMetadataRefs.remove(Pair.create(indexTable.keyspace, indexTable.indexName().get())));
        ((Tables)tablesDiff.created).forEach(table -> this.metadataRefs.put(table.id, new TableMetadataRef((TableMetadata)table)));
        ((Views)viewsDiff.created).forEach(view -> this.metadataRefs.put(view.metadata.id, new TableMetadataRef(view.metadata)));
        indexesDiff.entriesOnlyOnRight().values().forEach(indexTable -> this.indexMetadataRefs.put(Pair.create(indexTable.keyspace, indexTable.indexName().get()), new TableMetadataRef((TableMetadata)indexTable)));
        tablesDiff.altered.forEach(diff -> this.metadataRefs.get(((TableMetadata)diff.after).id).set((TableMetadata)diff.after));
        viewsDiff.altered.forEach(diff -> this.metadataRefs.get(((ViewMetadata)diff.after).metadata.id).set(((ViewMetadata)diff.after).metadata));
        indexesDiff.entriesDiffering().values().stream().map(MapDifference.ValueDifference::rightValue).forEach(indexTable -> this.indexMetadataRefs.get(Pair.create(indexTable.keyspace, indexTable.indexName().get())).set((TableMetadata)indexTable));
        SchemaDiagnostics.metadataReloaded(this, previous, updated, tablesDiff, viewsDiff, indexesDiff);
    }

    public void registerListener(SchemaChangeListener listener) {
        this.changeListeners.add(listener);
    }

    public void unregisterListener(SchemaChangeListener listener) {
        this.changeListeners.remove(listener);
    }

    public Keyspace getKeyspaceInstance(String keyspaceName) {
        return this.keyspaceInstances.get(keyspaceName);
    }

    public ColumnFamilyStore getColumnFamilyStoreInstance(TableId id) {
        TableMetadata metadata = this.getTableMetadata(id);
        if (metadata == null) {
            return null;
        }
        Keyspace instance = this.getKeyspaceInstance(metadata.keyspace);
        if (instance == null) {
            return null;
        }
        return instance.hasColumnFamilyStore(metadata.id) ? instance.getColumnFamilyStore(metadata.id) : null;
    }

    public void storeKeyspaceInstance(Keyspace keyspace) {
        if (this.keyspaceInstances.containsKey(keyspace.getName())) {
            throw new IllegalArgumentException(String.format("Keyspace %s was already initialized.", keyspace.getName()));
        }
        this.keyspaceInstances.put(keyspace.getName(), keyspace);
    }

    public Keyspace removeKeyspaceInstance(String keyspaceName) {
        return this.keyspaceInstances.remove(keyspaceName);
    }

    public Keyspaces snapshot() {
        return this.keyspaces;
    }

    synchronized void unload(KeyspaceMetadata ksm) {
        this.keyspaces = this.keyspaces.without(ksm.name);
        ksm.tablesAndViews().forEach(t -> this.metadataRefs.remove(t.id));
        ksm.tables.indexTables().keySet().forEach(name -> this.indexMetadataRefs.remove(Pair.create(ksm.name, name)));
        SchemaDiagnostics.metadataRemoved(this, ksm);
    }

    public int getNumberOfTables() {
        return this.keyspaces.stream().mapToInt(k -> Iterables.size(k.tablesAndViews())).sum();
    }

    public ViewMetadata getView(String keyspaceName, String viewName) {
        assert (keyspaceName != null);
        KeyspaceMetadata ksm = this.keyspaces.getNullable(keyspaceName);
        return ksm == null ? null : ksm.views.getNullable(viewName);
    }

    public KeyspaceMetadata getKeyspaceMetadata(String keyspaceName) {
        assert (keyspaceName != null);
        KeyspaceMetadata keyspace = this.keyspaces.getNullable(keyspaceName);
        return null != keyspace ? keyspace : VirtualKeyspaceRegistry.instance.getKeyspaceMetadataNullable(keyspaceName);
    }

    private Set<String> getNonSystemKeyspacesSet() {
        return Sets.difference(this.keyspaces.names(), SchemaConstants.LOCAL_SYSTEM_KEYSPACE_NAMES);
    }

    public ImmutableList<String> getNonSystemKeyspaces() {
        return ImmutableList.copyOf(this.getNonSystemKeyspacesSet());
    }

    public List<String> getNonLocalStrategyKeyspaces() {
        return this.keyspaces.stream().filter(keyspace -> keyspace.params.replication.klass != LocalStrategy.class).map(keyspace -> keyspace.name).collect(Collectors.toList());
    }

    public List<String> getUserKeyspaces() {
        return ImmutableList.copyOf((Collection)Sets.difference(this.getNonSystemKeyspacesSet(), SchemaConstants.REPLICATED_SYSTEM_KEYSPACE_NAMES));
    }

    public Iterable<TableMetadata> getTablesAndViews(String keyspaceName) {
        assert (keyspaceName != null);
        KeyspaceMetadata ksm = this.keyspaces.getNullable(keyspaceName);
        assert (ksm != null);
        return ksm.tablesAndViews();
    }

    public Set<String> getKeyspaces() {
        return this.keyspaces.names();
    }

    public TableMetadataRef getTableMetadataRef(String keyspace, String table) {
        TableMetadata tm = this.getTableMetadata(keyspace, table);
        return tm == null ? null : this.metadataRefs.get(tm.id);
    }

    public TableMetadataRef getIndexTableMetadataRef(String keyspace, String index) {
        return this.indexMetadataRefs.get(Pair.create(keyspace, index));
    }

    Map<Pair<String, String>, TableMetadataRef> getIndexTableMetadataRefs() {
        return this.indexMetadataRefs;
    }

    public TableMetadataRef getTableMetadataRef(TableId id) {
        return this.metadataRefs.get(id);
    }

    public TableMetadataRef getTableMetadataRef(Descriptor descriptor) {
        return this.getTableMetadataRef(descriptor.ksname, descriptor.cfname);
    }

    Map<TableId, TableMetadataRef> getTableMetadataRefs() {
        return this.metadataRefs;
    }

    public TableMetadata getTableMetadata(String keyspace, String table) {
        assert (keyspace != null);
        assert (table != null);
        KeyspaceMetadata ksm = this.getKeyspaceMetadata(keyspace);
        return ksm == null ? null : ksm.getTableOrViewNullable(table);
    }

    @Nullable
    public TableMetadata getTableMetadata(TableId id) {
        TableMetadata table = this.keyspaces.getTableOrViewNullable(id);
        return null != table ? table : VirtualKeyspaceRegistry.instance.getTableMetadataNullable(id);
    }

    public TableMetadata validateTable(String keyspaceName, String tableName) {
        if (tableName.isEmpty()) {
            throw new InvalidRequestException("non-empty table is required");
        }
        KeyspaceMetadata keyspace = this.getKeyspaceMetadata(keyspaceName);
        if (keyspace == null) {
            throw new KeyspaceNotDefinedException(String.format("keyspace %s does not exist", keyspaceName));
        }
        TableMetadata metadata = keyspace.getTableOrViewNullable(tableName);
        if (metadata == null) {
            throw new InvalidRequestException(String.format("table %s does not exist", tableName));
        }
        return metadata;
    }

    public TableMetadata getTableMetadata(Descriptor descriptor) {
        return this.getTableMetadata(descriptor.ksname, descriptor.cfname);
    }

    public TableMetadata getExistingTableMetadata(TableId id) throws UnknownTableException {
        TableMetadata metadata = this.getTableMetadata(id);
        if (metadata != null) {
            return metadata;
        }
        String message = String.format("Couldn't find table with id %s. If a table was just created, this is likely due to the schemanot being fully propagated.  Please wait for schema agreement on table creation.", id);
        throw new UnknownTableException(message, id);
    }

    public Collection<Function> getFunctions(FunctionName name) {
        if (!name.hasKeyspace()) {
            throw new IllegalArgumentException(String.format("Function name must be fully qualified: got %s", name));
        }
        KeyspaceMetadata ksm = this.getKeyspaceMetadata(name.keyspace);
        return ksm == null ? Collections.emptyList() : ksm.functions.get(name);
    }

    public Optional<Function> findFunction(FunctionName name, List<AbstractType<?>> argTypes) {
        if (!name.hasKeyspace()) {
            throw new IllegalArgumentException(String.format("Function name must be fully quallified: got %s", name));
        }
        KeyspaceMetadata ksm = this.getKeyspaceMetadata(name.keyspace);
        return ksm == null ? Optional.empty() : ksm.functions.find(name, argTypes);
    }

    public UUID getVersion() {
        return this.version;
    }

    public boolean isSameVersion(UUID schemaVersion) {
        return schemaVersion != null && schemaVersion.equals(this.version);
    }

    public boolean isEmpty() {
        return SchemaConstants.emptyVersion.equals(this.version);
    }

    public void updateVersion() {
        this.version = SchemaKeyspace.calculateSchemaDigest();
        SystemKeyspace.updateSchemaVersion(this.version);
        SchemaDiagnostics.versionUpdated(this);
    }

    public void updateVersionAndAnnounce() {
        this.updateVersion();
        this.passiveAnnounceVersion();
    }

    private void passiveAnnounceVersion() {
        Gossiper.instance.addLocalApplicationState(ApplicationState.SCHEMA, StorageService.instance.valueFactory.schema(this.version));
        SchemaDiagnostics.versionAnnounced(this);
    }

    public synchronized void clear() {
        this.getNonSystemKeyspaces().forEach(k -> this.unload(this.getKeyspaceMetadata((String)k)));
        this.updateVersionAndAnnounce();
        SchemaDiagnostics.schemataCleared(this);
    }

    public synchronized void reloadSchemaAndAnnounceVersion() {
        Keyspaces before = this.keyspaces.filter(k -> !SchemaConstants.isLocalSystemKeyspace(k.name));
        Keyspaces after = SchemaKeyspace.fetchNonSystemKeyspaces();
        this.merge(Keyspaces.diff(before, after));
        this.updateVersionAndAnnounce();
    }

    synchronized void mergeAndAnnounceVersion(Collection<Mutation> mutations) {
        this.merge(mutations);
        this.updateVersionAndAnnounce();
    }

    public synchronized TransformationResult transform(SchemaTransformation transformation, boolean locally, long now) {
        Keyspaces.KeyspacesDiff diff;
        try {
            Keyspaces before = this.keyspaces;
            Keyspaces after = transformation.apply(before);
            diff = Keyspaces.diff(before, after);
        }
        catch (RuntimeException e) {
            return new TransformationResult(e);
        }
        if (diff.isEmpty()) {
            return new TransformationResult(diff, Collections.emptyList());
        }
        Collection<Mutation> mutations = SchemaKeyspace.convertSchemaDiffToMutations(diff, now);
        SchemaKeyspace.applyChanges(mutations);
        this.merge(diff);
        this.updateVersion();
        if (!locally) {
            this.passiveAnnounceVersion();
        }
        return new TransformationResult(diff, mutations);
    }

    synchronized void merge(Collection<Mutation> mutations) {
        Set<String> affectedKeyspaces = SchemaKeyspace.affectedKeyspaces(mutations);
        Keyspaces before = this.keyspaces.filter(k -> affectedKeyspaces.contains(k.name));
        SchemaKeyspace.applyChanges(mutations);
        Keyspaces after = SchemaKeyspace.fetchKeyspaces(affectedKeyspaces);
        this.merge(Keyspaces.diff(before, after));
    }

    private void merge(Keyspaces.KeyspacesDiff diff) {
        diff.dropped.forEach(this::dropKeyspace);
        diff.created.forEach(this::createKeyspace);
        diff.altered.forEach(this::alterKeyspace);
    }

    private void alterKeyspace(KeyspaceMetadata.KeyspaceDiff delta) {
        SchemaDiagnostics.keyspaceAltering(this, delta);
        ((Views)delta.views.dropped).forEach(this::dropView);
        ((Tables)delta.tables.dropped).forEach(this::dropTable);
        this.load(delta.after);
        ((Tables)delta.tables.created).forEach(this::createTable);
        ((Views)delta.views.created).forEach(this::createView);
        delta.tables.altered.forEach(diff -> this.alterTable((TableMetadata)diff.after));
        delta.views.altered.forEach(diff -> this.alterView((ViewMetadata)diff.after));
        Keyspace.open((String)delta.after.name).viewManager.reload(true);
        ((Functions)delta.udas.dropped).forEach(uda -> this.notifyDropAggregate((UDAggregate)uda));
        ((Functions)delta.udfs.dropped).forEach(udf -> this.notifyDropFunction((UDFunction)udf));
        ((Views)delta.views.dropped).forEach(this::notifyDropView);
        ((Tables)delta.tables.dropped).forEach(this::notifyDropTable);
        ((Types)delta.types.dropped).forEach(this::notifyDropType);
        ((Types)delta.types.created).forEach(this::notifyCreateType);
        ((Tables)delta.tables.created).forEach(this::notifyCreateTable);
        ((Views)delta.views.created).forEach(this::notifyCreateView);
        ((Functions)delta.udfs.created).forEach(udf -> this.notifyCreateFunction((UDFunction)udf));
        ((Functions)delta.udas.created).forEach(uda -> this.notifyCreateAggregate((UDAggregate)uda));
        if (!delta.before.params.equals(delta.after.params)) {
            this.notifyAlterKeyspace(delta.before, delta.after);
        }
        delta.types.altered.forEach(diff -> this.notifyAlterType((UserType)diff.before, (UserType)diff.after));
        delta.tables.altered.forEach(diff -> this.notifyAlterTable((TableMetadata)diff.before, (TableMetadata)diff.after));
        delta.views.altered.forEach(diff -> this.notifyAlterView((ViewMetadata)diff.before, (ViewMetadata)diff.after));
        delta.udfs.altered.forEach(diff -> this.notifyAlterFunction((UDFunction)diff.before, (UDFunction)diff.after));
        delta.udas.altered.forEach(diff -> this.notifyAlterAggregate((UDAggregate)diff.before, (UDAggregate)diff.after));
        SchemaDiagnostics.keyspaceAltered(this, delta);
    }

    private void createKeyspace(KeyspaceMetadata keyspace) {
        SchemaDiagnostics.keyspaceCreating(this, keyspace);
        this.load(keyspace);
        Keyspace.open(keyspace.name);
        this.notifyCreateKeyspace(keyspace);
        keyspace.types.forEach(this::notifyCreateType);
        keyspace.tables.forEach(this::notifyCreateTable);
        keyspace.views.forEach(this::notifyCreateView);
        keyspace.functions.udfs().forEach(this::notifyCreateFunction);
        keyspace.functions.udas().forEach(this::notifyCreateAggregate);
        SchemaDiagnostics.keyspaceCreated(this, keyspace);
    }

    private void dropKeyspace(KeyspaceMetadata keyspace) {
        SchemaDiagnostics.keyspaceDroping(this, keyspace);
        keyspace.views.forEach(this::dropView);
        keyspace.tables.forEach(this::dropTable);
        Keyspace.clear(keyspace.name);
        this.unload(keyspace);
        Keyspace.writeOrder.awaitNewBarrier();
        keyspace.functions.udas().forEach(this::notifyDropAggregate);
        keyspace.functions.udfs().forEach(this::notifyDropFunction);
        keyspace.views.forEach(this::notifyDropView);
        keyspace.tables.forEach(this::notifyDropTable);
        keyspace.types.forEach(this::notifyDropType);
        this.notifyDropKeyspace(keyspace);
        SchemaDiagnostics.keyspaceDroped(this, keyspace);
    }

    private void dropView(ViewMetadata metadata) {
        Keyspace.open((String)metadata.keyspace()).viewManager.dropView(metadata.name());
        this.dropTable(metadata.metadata);
    }

    private void dropTable(TableMetadata metadata) {
        SchemaDiagnostics.tableDropping(this, metadata);
        ColumnFamilyStore cfs = Keyspace.open(metadata.keyspace).getColumnFamilyStore(metadata.name);
        assert (cfs != null);
        cfs.indexManager.markAllIndexesRemoved();
        CompactionManager.instance.interruptCompactionFor(Collections.singleton(metadata), sstable -> true, true);
        if (DatabaseDescriptor.isAutoSnapshot()) {
            cfs.snapshot(Keyspace.getTimestampedSnapshotNameWithPrefix(cfs.name, "dropped"));
        }
        CommitLog.instance.forceRecycleAllSegments(Collections.singleton(metadata.id));
        Keyspace.open(metadata.keyspace).dropCf(metadata.id);
        SchemaDiagnostics.tableDropped(this, metadata);
    }

    private void createTable(TableMetadata table) {
        SchemaDiagnostics.tableCreating(this, table);
        Keyspace.open(table.keyspace).initCf(this.metadataRefs.get(table.id), true);
        SchemaDiagnostics.tableCreated(this, table);
    }

    private void createView(ViewMetadata view) {
        Keyspace.open(view.keyspace()).initCf(this.metadataRefs.get(view.metadata.id), true);
    }

    private void alterTable(TableMetadata updated) {
        SchemaDiagnostics.tableAltering(this, updated);
        Keyspace.open(updated.keyspace).getColumnFamilyStore(updated.name).reload();
        SchemaDiagnostics.tableAltered(this, updated);
    }

    private void alterView(ViewMetadata updated) {
        Keyspace.open(updated.keyspace()).getColumnFamilyStore(updated.name()).reload();
    }

    private void notifyCreateKeyspace(KeyspaceMetadata ksm) {
        this.changeListeners.forEach(l -> l.onCreateKeyspace(ksm.name));
    }

    private void notifyCreateTable(TableMetadata metadata) {
        this.changeListeners.forEach(l -> l.onCreateTable(metadata.keyspace, metadata.name));
    }

    private void notifyCreateView(ViewMetadata view) {
        this.changeListeners.forEach(l -> l.onCreateView(view.keyspace(), view.name()));
    }

    private void notifyCreateType(UserType ut) {
        this.changeListeners.forEach(l -> l.onCreateType(ut.keyspace, ut.getNameAsString()));
    }

    private void notifyCreateFunction(UDFunction udf) {
        this.changeListeners.forEach(l -> l.onCreateFunction(udf.name().keyspace, udf.name().name, udf.argTypes()));
    }

    private void notifyCreateAggregate(UDAggregate udf) {
        this.changeListeners.forEach(l -> l.onCreateAggregate(udf.name().keyspace, udf.name().name, udf.argTypes()));
    }

    private void notifyAlterKeyspace(KeyspaceMetadata before, KeyspaceMetadata after) {
        this.changeListeners.forEach(l -> l.onAlterKeyspace(after.name));
    }

    private void notifyAlterTable(TableMetadata before, TableMetadata after) {
        boolean changeAffectedPreparedStatements = before.changeAffectsPreparedStatements(after);
        this.changeListeners.forEach(l -> l.onAlterTable(after.keyspace, after.name, changeAffectedPreparedStatements));
    }

    private void notifyAlterView(ViewMetadata before, ViewMetadata after) {
        boolean changeAffectedPreparedStatements = before.metadata.changeAffectsPreparedStatements(after.metadata);
        this.changeListeners.forEach(l -> l.onAlterView(after.keyspace(), after.name(), changeAffectedPreparedStatements));
    }

    private void notifyAlterType(UserType before, UserType after) {
        this.changeListeners.forEach(l -> l.onAlterType(after.keyspace, after.getNameAsString()));
    }

    private void notifyAlterFunction(UDFunction before, UDFunction after) {
        this.changeListeners.forEach(l -> l.onAlterFunction(after.name().keyspace, after.name().name, after.argTypes()));
    }

    private void notifyAlterAggregate(UDAggregate before, UDAggregate after) {
        this.changeListeners.forEach(l -> l.onAlterAggregate(after.name().keyspace, after.name().name, after.argTypes()));
    }

    private void notifyDropKeyspace(KeyspaceMetadata ksm) {
        this.changeListeners.forEach(l -> l.onDropKeyspace(ksm.name));
    }

    private void notifyDropTable(TableMetadata metadata) {
        this.changeListeners.forEach(l -> l.onDropTable(metadata.keyspace, metadata.name));
    }

    private void notifyDropView(ViewMetadata view) {
        this.changeListeners.forEach(l -> l.onDropView(view.keyspace(), view.name()));
    }

    private void notifyDropType(UserType ut) {
        this.changeListeners.forEach(l -> l.onDropType(ut.keyspace, ut.getNameAsString()));
    }

    private void notifyDropFunction(UDFunction udf) {
        this.changeListeners.forEach(l -> l.onDropFunction(udf.name().keyspace, udf.name().name, udf.argTypes()));
    }

    private void notifyDropAggregate(UDAggregate udf) {
        this.changeListeners.forEach(l -> l.onDropAggregate(udf.name().keyspace, udf.name().name, udf.argTypes()));
    }

    public static String schemaVersionToString(UUID version) {
        return version == null ? "unknown" : (SchemaConstants.emptyVersion.equals(version) ? "(empty)" : version.toString());
    }

    public static final class TransformationResult {
        public final boolean success;
        public final RuntimeException exception;
        public final Keyspaces.KeyspacesDiff diff;
        public final Collection<Mutation> mutations;

        private TransformationResult(boolean success, RuntimeException exception, Keyspaces.KeyspacesDiff diff, Collection<Mutation> mutations) {
            this.success = success;
            this.exception = exception;
            this.diff = diff;
            this.mutations = mutations;
        }

        TransformationResult(RuntimeException exception) {
            this(false, exception, null, null);
        }

        TransformationResult(Keyspaces.KeyspacesDiff diff, Collection<Mutation> mutations) {
            this(true, null, diff, mutations);
        }
    }
}

