/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import com.google.common.util.concurrent.RateLimiter;
import java.util.Objects;
import java.util.Optional;
import org.apache.cassandra.cache.ChunkCache;
import org.apache.cassandra.io.compress.BufferType;
import org.apache.cassandra.io.compress.CompressionMetadata;
import org.apache.cassandra.io.util.ChannelProxy;
import org.apache.cassandra.io.util.ChunkReader;
import org.apache.cassandra.io.util.CompressedChunkReader;
import org.apache.cassandra.io.util.DiskOptimizationStrategy;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.LimitingRebufferer;
import org.apache.cassandra.io.util.MmapRebufferer;
import org.apache.cassandra.io.util.MmappedRegions;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.io.util.Rebufferer;
import org.apache.cassandra.io.util.RebuffererFactory;
import org.apache.cassandra.io.util.SimpleChunkReader;
import org.apache.cassandra.utils.NativeLibrary;
import org.apache.cassandra.utils.Throwables;
import org.apache.cassandra.utils.concurrent.Ref;
import org.apache.cassandra.utils.concurrent.RefCounted;
import org.apache.cassandra.utils.concurrent.SharedCloseableImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileHandle
extends SharedCloseableImpl {
    private static final Logger logger = LoggerFactory.getLogger(FileHandle.class);
    public final ChannelProxy channel;
    public final long onDiskLength;
    private final RebuffererFactory rebuffererFactory;
    private final Optional<CompressionMetadata> compressionMetadata;

    private FileHandle(Cleanup cleanup, ChannelProxy channel, RebuffererFactory rebuffererFactory, CompressionMetadata compressionMetadata, long onDiskLength) {
        super(cleanup);
        this.rebuffererFactory = rebuffererFactory;
        this.channel = channel;
        this.compressionMetadata = Optional.ofNullable(compressionMetadata);
        this.onDiskLength = onDiskLength;
    }

    private FileHandle(FileHandle copy) {
        super(copy);
        this.channel = copy.channel;
        this.rebuffererFactory = copy.rebuffererFactory;
        this.compressionMetadata = copy.compressionMetadata;
        this.onDiskLength = copy.onDiskLength;
    }

    public String path() {
        return this.channel.filePath();
    }

    public long dataLength() {
        return this.compressionMetadata.map(c -> c.dataLength).orElseGet(this.rebuffererFactory::fileLength);
    }

    public RebuffererFactory rebuffererFactory() {
        return this.rebuffererFactory;
    }

    public Optional<CompressionMetadata> compressionMetadata() {
        return this.compressionMetadata;
    }

    @Override
    public void addTo(Ref.IdentityCollection identities) {
        super.addTo(identities);
        this.compressionMetadata.ifPresent(metadata -> metadata.addTo(identities));
    }

    @Override
    public FileHandle sharedCopy() {
        return new FileHandle(this);
    }

    public RandomAccessReader createReader() {
        return this.createReader(null);
    }

    public RandomAccessReader createReader(RateLimiter limiter) {
        return new RandomAccessReader(this.instantiateRebufferer(limiter));
    }

    public FileDataInput createReader(long position) {
        RandomAccessReader reader = this.createReader();
        reader.seek(position);
        return reader;
    }

    public void dropPageCache(long before) {
        long position = this.compressionMetadata.map(metadata -> {
            if (before >= metadata.dataLength) {
                return 0L;
            }
            return metadata.chunkFor((long)before).offset;
        }).orElse(before);
        NativeLibrary.trySkipCache(this.channel.getFileDescriptor(), 0L, position, this.path());
    }

    private Rebufferer instantiateRebufferer(RateLimiter limiter) {
        Rebufferer rebufferer = this.rebuffererFactory.instantiateRebufferer();
        if (limiter != null) {
            rebufferer = new LimitingRebufferer(rebufferer, limiter, 65536);
        }
        return rebufferer;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(path='" + this.path() + '\'' + ", length=" + this.rebuffererFactory.fileLength() + ')';
    }

    public static class Builder
    implements AutoCloseable {
        private final String path;
        private ChannelProxy channel;
        private CompressionMetadata compressionMetadata;
        private MmappedRegions regions;
        private ChunkCache chunkCache;
        private int bufferSize = 4096;
        private BufferType bufferType = BufferType.OFF_HEAP;
        private boolean mmapped = false;
        private boolean compressed = false;

        public Builder(String path) {
            this.path = path;
        }

        public Builder(ChannelProxy channel) {
            this.channel = channel;
            this.path = channel.filePath();
        }

        public Builder compressed(boolean compressed) {
            this.compressed = compressed;
            return this;
        }

        public Builder withChunkCache(ChunkCache chunkCache) {
            this.chunkCache = chunkCache;
            return this;
        }

        public Builder withCompressionMetadata(CompressionMetadata metadata) {
            this.compressed = Objects.nonNull(metadata);
            this.compressionMetadata = metadata;
            return this;
        }

        public Builder mmapped(boolean mmapped) {
            this.mmapped = mmapped;
            return this;
        }

        public Builder bufferSize(int bufferSize) {
            this.bufferSize = bufferSize;
            return this;
        }

        public Builder bufferType(BufferType bufferType) {
            this.bufferType = bufferType;
            return this;
        }

        public FileHandle complete() {
            return this.complete(-1L);
        }

        public FileHandle complete(long overrideLength) {
            boolean channelOpened = false;
            if (this.channel == null) {
                this.channel = new ChannelProxy(this.path);
                channelOpened = true;
            }
            ChannelProxy channelCopy = this.channel.sharedCopy();
            try {
                RebuffererFactory rebuffererFactory;
                long length;
                if (this.compressed && this.compressionMetadata == null) {
                    this.compressionMetadata = CompressionMetadata.create(channelCopy.filePath());
                }
                long l = overrideLength > 0L ? overrideLength : (length = this.compressed ? this.compressionMetadata.compressedFileLength : channelCopy.size());
                if (this.mmapped) {
                    if (this.compressed) {
                        this.regions = MmappedRegions.map(channelCopy, this.compressionMetadata);
                        rebuffererFactory = this.maybeCached(new CompressedChunkReader.Mmap(channelCopy, this.compressionMetadata, this.regions));
                    } else {
                        this.updateRegions(channelCopy, length);
                        rebuffererFactory = new MmapRebufferer(channelCopy, length, this.regions.sharedCopy());
                    }
                } else {
                    this.regions = null;
                    if (this.compressed) {
                        rebuffererFactory = this.maybeCached(new CompressedChunkReader.Standard(channelCopy, this.compressionMetadata));
                    } else {
                        int chunkSize = DiskOptimizationStrategy.roundForCaching(this.bufferSize, ChunkCache.roundUp);
                        rebuffererFactory = this.maybeCached(new SimpleChunkReader(channelCopy, length, this.bufferType, chunkSize));
                    }
                }
                Cleanup cleanup = new Cleanup(channelCopy, rebuffererFactory, this.compressionMetadata, this.chunkCache);
                return new FileHandle(cleanup, channelCopy, rebuffererFactory, this.compressionMetadata, length);
            }
            catch (Throwable t) {
                channelCopy.close();
                if (channelOpened) {
                    ChannelProxy c = this.channel;
                    this.channel = null;
                    throw Throwables.cleaned(c.close(t));
                }
                throw t;
            }
        }

        public Throwable close(Throwable accumulate) {
            if (!this.compressed && this.regions != null) {
                accumulate = this.regions.close(accumulate);
            }
            if (this.channel != null) {
                return this.channel.close(accumulate);
            }
            return accumulate;
        }

        @Override
        public void close() {
            Throwables.maybeFail(this.close(null));
        }

        private RebuffererFactory maybeCached(ChunkReader reader) {
            if (this.chunkCache != null && this.chunkCache.capacity() > 0L) {
                return this.chunkCache.wrap(reader);
            }
            return reader;
        }

        private void updateRegions(ChannelProxy channel, long length) {
            if (this.regions != null && !this.regions.isValid(channel)) {
                Throwable err = this.regions.close(null);
                if (err != null) {
                    logger.error("Failed to close mapped regions", err);
                }
                this.regions = null;
            }
            if (this.regions == null) {
                this.regions = MmappedRegions.map(channel, length);
            } else {
                this.regions.extend(length);
            }
        }
    }

    private static class Cleanup
    implements RefCounted.Tidy {
        final ChannelProxy channel;
        final RebuffererFactory rebufferer;
        final CompressionMetadata compressionMetadata;
        final Optional<ChunkCache> chunkCache;

        private Cleanup(ChannelProxy channel, RebuffererFactory rebufferer, CompressionMetadata compressionMetadata, ChunkCache chunkCache) {
            this.channel = channel;
            this.rebufferer = rebufferer;
            this.compressionMetadata = compressionMetadata;
            this.chunkCache = Optional.ofNullable(chunkCache);
        }

        @Override
        public String name() {
            return this.channel.filePath();
        }

        @Override
        public void tidy() {
            this.chunkCache.ifPresent(cache -> cache.invalidateFile(this.name()));
            try {
                if (this.compressionMetadata != null) {
                    this.compressionMetadata.close();
                }
            }
            finally {
                try {
                    this.channel.close();
                }
                finally {
                    this.rebufferer.close();
                }
            }
        }
    }
}

