/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.metrics;

import com.codahale.metrics.Clock;
import com.codahale.metrics.Reservoir;
import com.codahale.metrics.Snapshot;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Ints;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;
import org.apache.cassandra.utils.EstimatedHistogram;

public class DecayingEstimatedHistogramReservoir
implements Reservoir {
    public static final int DEFAULT_BUCKET_COUNT = 164;
    public static final boolean DEFAULT_ZERO_CONSIDERATION = false;
    public static final long[] DEFAULT_WITHOUT_ZERO_BUCKET_OFFSETS = EstimatedHistogram.newOffsets(164, false);
    public static final long[] DEFAULT_WITH_ZERO_BUCKET_OFFSETS = EstimatedHistogram.newOffsets(164, true);
    private final long[] bucketOffsets;
    private final LongAdder[] decayingBuckets;
    private final LongAdder[] buckets;
    public static final long HALF_TIME_IN_S = 60L;
    public static final double MEAN_LIFETIME_IN_S = 60.0 / Math.log(2.0);
    public static final long LANDMARK_RESET_INTERVAL_IN_MS = 1800000L;
    private final AtomicBoolean rescaling = new AtomicBoolean(false);
    private volatile long decayLandmark;
    private final Clock clock;

    public DecayingEstimatedHistogramReservoir() {
        this(false, 164, Clock.defaultClock());
    }

    public DecayingEstimatedHistogramReservoir(boolean considerZeroes) {
        this(considerZeroes, 164, Clock.defaultClock());
    }

    public DecayingEstimatedHistogramReservoir(boolean considerZeroes, int bucketCount) {
        this(considerZeroes, bucketCount, Clock.defaultClock());
    }

    @VisibleForTesting
    DecayingEstimatedHistogramReservoir(boolean considerZeroes, int bucketCount, Clock clock) {
        this.bucketOffsets = bucketCount == 164 ? (considerZeroes ? DEFAULT_WITH_ZERO_BUCKET_OFFSETS : DEFAULT_WITHOUT_ZERO_BUCKET_OFFSETS) : EstimatedHistogram.newOffsets(bucketCount, considerZeroes);
        this.decayingBuckets = new LongAdder[this.bucketOffsets.length + 1];
        this.buckets = new LongAdder[this.bucketOffsets.length + 1];
        for (int i = 0; i < this.buckets.length; ++i) {
            this.decayingBuckets[i] = new LongAdder();
            this.buckets[i] = new LongAdder();
        }
        this.clock = clock;
        this.decayLandmark = clock.getTime();
    }

    public void update(long value) {
        long now = this.clock.getTime();
        this.rescaleIfNeeded(now);
        int index = Arrays.binarySearch(this.bucketOffsets, value);
        if (index < 0) {
            index = -index - 1;
        }
        this.decayingBuckets[index].add(Math.round(this.forwardDecayWeight(now)));
        this.buckets[index].increment();
    }

    private double forwardDecayWeight(long now) {
        return Math.exp((double)(now - this.decayLandmark) / 1000.0 / MEAN_LIFETIME_IN_S);
    }

    public int size() {
        return this.decayingBuckets.length;
    }

    public Snapshot getSnapshot() {
        this.rescaleIfNeeded();
        return new EstimatedHistogramReservoirSnapshot(this);
    }

    @VisibleForTesting
    boolean isOverflowed() {
        return this.decayingBuckets[this.decayingBuckets.length - 1].sum() > 0L;
    }

    private void rescaleIfNeeded() {
        this.rescaleIfNeeded(this.clock.getTime());
    }

    private void rescaleIfNeeded(long now) {
        if (this.needRescale(now) && this.rescaling.compareAndSet(false, true)) {
            try {
                this.rescale(now);
            }
            finally {
                this.decayLandmark = now;
                this.rescaling.set(false);
            }
        }
    }

    private void rescale(long now) {
        double rescaleFactor = this.forwardDecayWeight(now);
        int bucketCount = this.decayingBuckets.length;
        for (int i = 0; i < bucketCount; ++i) {
            long storedValue = this.decayingBuckets[i].sumThenReset();
            storedValue = Math.round((double)storedValue / rescaleFactor);
            this.decayingBuckets[i].add(storedValue);
        }
    }

    private boolean needRescale(long now) {
        return now - this.decayLandmark > 1800000L;
    }

    @VisibleForTesting
    public void clear() {
        int bucketCount = this.decayingBuckets.length;
        for (int i = 0; i < bucketCount; ++i) {
            this.decayingBuckets[i].reset();
            this.buckets[i].reset();
        }
    }

    public void rebase(EstimatedHistogramReservoirSnapshot snapshot) {
        int i;
        if (this.decayingBuckets.length != snapshot.decayingBuckets.length) {
            throw new IllegalStateException("Unable to merge two DecayingEstimatedHistogramReservoirs with different bucket sizes");
        }
        for (i = 0; i < this.bucketOffsets.length; ++i) {
            if (this.bucketOffsets[i] == snapshot.bucketOffsets[i]) continue;
            throw new IllegalStateException("Merge is only supported with equal bucketOffsets");
        }
        this.decayLandmark = snapshot.snapshotLandmark;
        for (i = 0; i < this.decayingBuckets.length; ++i) {
            this.decayingBuckets[i].reset();
            this.buckets[i].reset();
            this.decayingBuckets[i].add(snapshot.decayingBuckets[i]);
            this.buckets[i].add(snapshot.values[i]);
        }
    }

    static class EstimatedHistogramReservoirSnapshot
    extends Snapshot {
        private final long[] decayingBuckets;
        private final long[] values;
        private long count;
        private long snapshotLandmark;
        private long[] bucketOffsets;
        private DecayingEstimatedHistogramReservoir reservoir;

        public EstimatedHistogramReservoirSnapshot(DecayingEstimatedHistogramReservoir reservoir) {
            int length = reservoir.decayingBuckets.length;
            double rescaleFactor = reservoir.forwardDecayWeight(reservoir.clock.getTime());
            this.decayingBuckets = new long[length];
            this.values = new long[length];
            this.snapshotLandmark = reservoir.decayLandmark;
            this.bucketOffsets = reservoir.bucketOffsets;
            for (int i = 0; i < length; ++i) {
                this.decayingBuckets[i] = Math.round((double)reservoir.decayingBuckets[i].sum() / rescaleFactor);
                this.values[i] = reservoir.buckets[i].sum();
            }
            this.count = this.count();
            this.reservoir = reservoir;
        }

        public double getValue(double quantile) {
            assert (quantile >= 0.0 && quantile <= 1.0);
            int lastBucket = this.decayingBuckets.length - 1;
            if (this.decayingBuckets[lastBucket] > 0L) {
                throw new IllegalStateException("Unable to compute when histogram overflowed");
            }
            long qcount = (long)Math.ceil((double)this.count() * quantile);
            if (qcount == 0L) {
                return 0.0;
            }
            long elements = 0L;
            for (int i = 0; i < lastBucket; ++i) {
                if ((elements += this.decayingBuckets[i]) < qcount) continue;
                return this.bucketOffsets[i];
            }
            return 0.0;
        }

        public long[] getValues() {
            return this.values;
        }

        public int size() {
            return Ints.saturatedCast((long)this.count);
        }

        @VisibleForTesting
        public long getSnapshotLandmark() {
            return this.snapshotLandmark;
        }

        private long count() {
            long sum = 0L;
            for (int i = 0; i < this.decayingBuckets.length; ++i) {
                sum += this.decayingBuckets[i];
            }
            return sum;
        }

        public long getMax() {
            int lastBucket = this.decayingBuckets.length - 1;
            if (this.decayingBuckets[lastBucket] > 0L) {
                return Long.MAX_VALUE;
            }
            for (int i = lastBucket - 1; i >= 0; --i) {
                if (this.decayingBuckets[i] <= 0L) continue;
                return this.bucketOffsets[i];
            }
            return 0L;
        }

        public double getMean() {
            int lastBucket = this.decayingBuckets.length - 1;
            if (this.decayingBuckets[lastBucket] > 0L) {
                throw new IllegalStateException("Unable to compute when histogram overflowed");
            }
            long elements = 0L;
            long sum = 0L;
            for (int i = 0; i < lastBucket; ++i) {
                long bCount = this.decayingBuckets[i];
                elements += bCount;
                sum += bCount * this.bucketOffsets[i];
            }
            return (double)sum / (double)elements;
        }

        public long getMin() {
            for (int i = 0; i < this.decayingBuckets.length; ++i) {
                if (this.decayingBuckets[i] <= 0L) continue;
                return i == 0 ? 0L : 1L + this.bucketOffsets[i - 1];
            }
            return 0L;
        }

        public double getStdDev() {
            int lastBucket = this.decayingBuckets.length - 1;
            if (this.decayingBuckets[lastBucket] > 0L) {
                throw new IllegalStateException("Unable to compute when histogram overflowed");
            }
            long count = this.count();
            if (count <= 1L) {
                return 0.0;
            }
            double mean = this.getMean();
            double sum = 0.0;
            for (int i = 0; i < lastBucket; ++i) {
                long value = this.bucketOffsets[i];
                double diff = (double)value - mean;
                sum += diff * diff * (double)this.decayingBuckets[i];
            }
            return Math.sqrt(sum / (double)(count - 1L));
        }

        public void dump(OutputStream output) {
            try (PrintWriter out = new PrintWriter(new OutputStreamWriter(output, StandardCharsets.UTF_8));){
                int length = this.decayingBuckets.length;
                for (int i = 0; i < length; ++i) {
                    out.printf("%d%n", this.decayingBuckets[i]);
                }
            }
        }

        public void add(Snapshot other) {
            int i;
            if (!(other instanceof EstimatedHistogramReservoirSnapshot)) {
                throw new IllegalStateException("Unable to add other types of Snapshot than another DecayingEstimatedHistogramReservoir");
            }
            EstimatedHistogramReservoirSnapshot snapshot = (EstimatedHistogramReservoirSnapshot)other;
            if (this.decayingBuckets.length != snapshot.decayingBuckets.length) {
                throw new IllegalStateException("Unable to merge two DecayingEstimatedHistogramReservoirs with different bucket sizes");
            }
            for (i = 0; i < this.bucketOffsets.length; ++i) {
                if (this.bucketOffsets[i] == snapshot.bucketOffsets[i]) continue;
                throw new IllegalStateException("Merge is only supported with equal bucketOffsets");
            }
            if (snapshot.snapshotLandmark < this.snapshotLandmark) {
                this.rescaleArray(snapshot.decayingBuckets, this.snapshotLandmark - snapshot.snapshotLandmark);
            } else if (snapshot.snapshotLandmark > this.snapshotLandmark) {
                this.rescaleArray(this.decayingBuckets, snapshot.snapshotLandmark - this.snapshotLandmark);
                this.snapshotLandmark = snapshot.snapshotLandmark;
            }
            for (i = 0; i < snapshot.decayingBuckets.length; ++i) {
                int n = i;
                this.decayingBuckets[n] = this.decayingBuckets[n] + snapshot.decayingBuckets[i];
                int n2 = i;
                this.values[n2] = this.values[n2] + snapshot.values[i];
            }
            this.count += snapshot.count;
        }

        private void rescaleArray(long[] decayingBuckets, long landMarkDifference) {
            double rescaleFactor = Math.exp((double)landMarkDifference / 1000.0 / MEAN_LIFETIME_IN_S);
            for (int i = 0; i < decayingBuckets.length; ++i) {
                decayingBuckets[i] = Math.round((double)decayingBuckets[i] / rescaleFactor);
            }
        }

        public void rebaseReservoir() {
            this.reservoir.rebase(this);
        }
    }
}

