/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.audit;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import org.apache.cassandra.audit.AuditLogEntry;
import org.apache.cassandra.audit.AuditLogEntryCategory;
import org.apache.cassandra.audit.AuditLogEntryType;
import org.apache.cassandra.audit.AuditLogFilter;
import org.apache.cassandra.audit.AuditLogOptions;
import org.apache.cassandra.audit.BinAuditLogger;
import org.apache.cassandra.audit.FullQueryLogger;
import org.apache.cassandra.audit.IAuditLogger;
import org.apache.cassandra.audit.NoOpAuditLogger;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.CQLStatement;
import org.apache.cassandra.cql3.QueryHandler;
import org.apache.cassandra.cql3.QueryOptions;
import org.apache.cassandra.cql3.statements.BatchStatement;
import org.apache.cassandra.exceptions.AuthenticationException;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.UnauthorizedException;
import org.apache.cassandra.service.QueryState;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuditLogManager {
    private static final Logger logger = LoggerFactory.getLogger(AuditLogManager.class);
    private static final AuditLogManager instance = new AuditLogManager();
    private final FullQueryLogger fullQueryLogger;
    private final ImmutableSet<AuditLogEntryCategory> fqlIncludeFilter = ImmutableSet.of((Object)((Object)AuditLogEntryCategory.OTHER), (Object)((Object)AuditLogEntryCategory.QUERY), (Object)((Object)AuditLogEntryCategory.DCL), (Object)((Object)AuditLogEntryCategory.DML), (Object)((Object)AuditLogEntryCategory.DDL));
    private volatile IAuditLogger auditLogger;
    private volatile AuditLogFilter filter;
    private volatile boolean isAuditLogEnabled;

    private AuditLogManager() {
        this.fullQueryLogger = new FullQueryLogger();
        if (DatabaseDescriptor.getAuditLoggingOptions().enabled) {
            logger.info("Audit logging is enabled.");
            this.auditLogger = this.getAuditLogger(DatabaseDescriptor.getAuditLoggingOptions().logger);
            this.isAuditLogEnabled = true;
        } else {
            logger.debug("Audit logging is disabled.");
            this.isAuditLogEnabled = false;
            this.auditLogger = new NoOpAuditLogger();
        }
        this.filter = AuditLogFilter.create(DatabaseDescriptor.getAuditLoggingOptions());
    }

    public static AuditLogManager getInstance() {
        return instance;
    }

    private IAuditLogger getAuditLogger(String loggerClassName) throws ConfigurationException {
        if (loggerClassName != null) {
            return FBUtilities.newAuditLogger(loggerClassName);
        }
        return FBUtilities.newAuditLogger(BinAuditLogger.class.getName());
    }

    @VisibleForTesting
    public IAuditLogger getLogger() {
        return this.auditLogger;
    }

    public boolean isAuditingEnabled() {
        return this.isAuditLogEnabled;
    }

    public boolean isLoggingEnabled() {
        return this.isAuditingEnabled() || this.isFQLEnabled();
    }

    private boolean isFQLEnabled() {
        return this.fullQueryLogger.enabled();
    }

    private void logAuditLoggerEntry(AuditLogEntry logEntry) {
        if (!this.filter.isFiltered(logEntry)) {
            this.auditLogger.log(logEntry);
        }
    }

    public void log(AuditLogEntry logEntry) {
        if (logEntry == null) {
            return;
        }
        if (this.isAuditingEnabled()) {
            this.logAuditLoggerEntry(logEntry);
        }
        if (this.isFQLEnabled() && this.fqlIncludeFilter.contains((Object)logEntry.getType().getCategory())) {
            this.fullQueryLogger.log(logEntry);
        }
    }

    public void log(AuditLogEntry logEntry, Exception e) {
        if (logEntry != null && this.isAuditingEnabled()) {
            AuditLogEntry.Builder builder = new AuditLogEntry.Builder(logEntry);
            if (e instanceof UnauthorizedException) {
                builder.setType(AuditLogEntryType.UNAUTHORIZED_ATTEMPT);
            } else if (e instanceof AuthenticationException) {
                builder.setType(AuditLogEntryType.LOGIN_ERROR);
            } else {
                builder.setType(AuditLogEntryType.REQUEST_FAILURE);
            }
            builder.appendToOperation(e.getMessage());
            this.log(builder.build());
        }
    }

    public void logBatch(BatchStatement.Type type, List<Object> queryOrIdList, List<List<ByteBuffer>> values, List<QueryHandler.Prepared> prepared, QueryOptions options, QueryState state, long queryStartTimeMillis) {
        if (this.isAuditingEnabled()) {
            List<AuditLogEntry> entries = AuditLogManager.buildEntriesForBatch(queryOrIdList, prepared, state, options, queryStartTimeMillis);
            for (AuditLogEntry auditLogEntry : entries) {
                this.logAuditLoggerEntry(auditLogEntry);
            }
        }
        if (this.isFQLEnabled()) {
            ArrayList<String> queryStrings = new ArrayList<String>(queryOrIdList.size());
            for (QueryHandler.Prepared prepStatment : prepared) {
                queryStrings.add(prepStatment.rawCQLStatement);
            }
            this.fullQueryLogger.logBatch(type, queryStrings, values, options, state, queryStartTimeMillis);
        }
    }

    private static List<AuditLogEntry> buildEntriesForBatch(List<Object> queryOrIdList, List<QueryHandler.Prepared> prepared, QueryState state, QueryOptions options, long queryStartTimeMillis) {
        ArrayList<AuditLogEntry> auditLogEntries = new ArrayList<AuditLogEntry>(queryOrIdList.size() + 1);
        UUID batchId = UUID.randomUUID();
        String queryString = String.format("BatchId:[%s] - BATCH of [%d] statements", batchId, queryOrIdList.size());
        AuditLogEntry entry = new AuditLogEntry.Builder(state).setOperation(queryString).setOptions(options).setTimestamp(queryStartTimeMillis).setBatch(batchId).setType(AuditLogEntryType.BATCH).build();
        auditLogEntries.add(entry);
        for (int i = 0; i < queryOrIdList.size(); ++i) {
            CQLStatement statement = prepared.get((int)i).statement;
            entry = new AuditLogEntry.Builder(state).setType(statement.getAuditLogContext().auditLogEntryType).setOperation(prepared.get((int)i).rawCQLStatement).setTimestamp(queryStartTimeMillis).setScope(statement).setKeyspace(state, statement).setOptions(options).setBatch(batchId).build();
            auditLogEntries.add(entry);
        }
        return auditLogEntries;
    }

    public synchronized void disableAuditLog() {
        if (this.isAuditLogEnabled) {
            this.isAuditLogEnabled = false;
            IAuditLogger oldLogger = this.auditLogger;
            this.auditLogger = new NoOpAuditLogger();
            oldLogger.stop();
        }
    }

    public synchronized void enableAuditLog(AuditLogOptions auditLogOptions) throws ConfigurationException {
        if (this.isFQLEnabled() && this.fullQueryLogger.path().toString().equals(auditLogOptions.audit_logs_dir)) {
            throw new IllegalArgumentException(String.format("audit log path (%s) cannot be the same as the running full query logger (%s)", auditLogOptions.audit_logs_dir, this.fullQueryLogger.path()));
        }
        this.filter = AuditLogFilter.create(auditLogOptions);
        IAuditLogger oldLogger = this.auditLogger;
        if (oldLogger.getClass().getSimpleName().equals(auditLogOptions.logger)) {
            return;
        }
        this.auditLogger = this.getAuditLogger(auditLogOptions.logger);
        this.isAuditLogEnabled = true;
        oldLogger.stop();
    }

    public void configureFQL(Path path, String rollCycle, boolean blocking, int maxQueueWeight, long maxLogSize, String archiveCommand, int maxArchiveRetries) {
        if (path.equals(this.auditLogger.path())) {
            throw new IllegalArgumentException(String.format("fullquerylogger path (%s) cannot be the same as the running audit logger (%s)", path, this.auditLogger.path()));
        }
        this.fullQueryLogger.configure(path, rollCycle, blocking, maxQueueWeight, maxLogSize, archiveCommand, maxArchiveRetries);
    }

    public void resetFQL(String fullQueryLogPath) {
        this.fullQueryLogger.reset(fullQueryLogPath);
    }

    public void disableFQL() {
        this.fullQueryLogger.stop();
    }

    FullQueryLogger getFullQueryLogger() {
        return this.fullQueryLogger;
    }
}

