/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.zip.CRC32;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.AbstractCommitLogService;
import org.apache.cassandra.db.commitlog.BatchCommitLogService;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogMBean;
import org.apache.cassandra.db.commitlog.CommitLogReplayer;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.CommitLogSegmentManager;
import org.apache.cassandra.db.commitlog.PeriodicCommitLogService;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.DataOutputBufferFixed;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.metrics.CommitLogMetrics;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.MBeanWrapper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLog
implements CommitLogMBean {
    private static final Logger logger = LoggerFactory.getLogger(CommitLog.class);
    public static final CommitLog instance = CommitLog.construct();
    private final long MAX_MUTATION_SIZE = DatabaseDescriptor.getMaxMutationSize();
    public final CommitLogSegmentManager allocator;
    public final CommitLogArchiver archiver;
    final CommitLogMetrics metrics;
    final AbstractCommitLogService executor;
    volatile Configuration configuration;
    public final String location;

    private static CommitLog construct() {
        CommitLog log = new CommitLog(DatabaseDescriptor.getCommitLogLocation(), CommitLogArchiver.construct());
        MBeanWrapper.instance.registerMBean((Object)log, "org.apache.cassandra.db:type=Commitlog");
        return log.start();
    }

    @VisibleForTesting
    CommitLog(String location, CommitLogArchiver archiver) {
        this.location = location;
        this.configuration = new Configuration(DatabaseDescriptor.getCommitLogCompression());
        DatabaseDescriptor.createAllDirectories();
        this.archiver = archiver;
        this.metrics = new CommitLogMetrics();
        this.executor = DatabaseDescriptor.getCommitLogSync() == Config.CommitLogSync.batch ? new BatchCommitLogService(this) : new PeriodicCommitLogService(this);
        this.allocator = new CommitLogSegmentManager(this);
        this.metrics.attach(this.executor, this.allocator);
    }

    CommitLog start() {
        this.executor.start();
        this.allocator.start();
        return this;
    }

    public int recover() throws IOException {
        if (this.allocator.createReserveSegments) {
            return 0;
        }
        FilenameFilter unmanagedFilesFilter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return CommitLogDescriptor.isValid(name) && CommitLogSegment.shouldReplay(name);
            }
        };
        for (File file : new File(DatabaseDescriptor.getCommitLogLocation()).listFiles(unmanagedFilesFilter)) {
            this.archiver.maybeArchive(file.getPath(), file.getName());
            this.archiver.maybeWaitForArchiving(file.getName());
        }
        assert (this.archiver.archivePending.isEmpty()) : "Not all commit log archive tasks were completed before restore";
        this.archiver.maybeRestoreArchive();
        Object[] files = new File(DatabaseDescriptor.getCommitLogLocation()).listFiles(unmanagedFilesFilter);
        int replayed = 0;
        this.allocator.enableReserveSegmentCreation();
        if (files.length == 0) {
            logger.info("No commitlog files found; skipping replay");
        } else {
            Arrays.sort(files, new CommitLogSegment.CommitLogSegmentFileComparator());
            logger.info("Replaying {}", (Object)StringUtils.join((Object[])files, (String)", "));
            replayed = this.recover((File[])files);
            logger.info("Log replay complete, {} replayed mutations", (Object)replayed);
            for (Object f : files) {
                this.allocator.recycleSegment((File)f);
            }
        }
        return replayed;
    }

    public int recover(File ... clogs) throws IOException {
        CommitLogReplayer recovery = CommitLogReplayer.construct(this);
        recovery.recover(clogs);
        return recovery.blockForWrites();
    }

    @Override
    public void recover(String path) throws IOException {
        CommitLogReplayer recovery = CommitLogReplayer.construct(this);
        recovery.recover(new File(path), false);
        recovery.blockForWrites();
    }

    public ReplayPosition getContext() {
        return this.allocator.allocatingFrom().getContext();
    }

    public void forceRecycleAllSegments(Iterable<UUID> droppedCfs) {
        this.allocator.forceRecycleAll(droppedCfs);
    }

    public void forceRecycleAllSegments() {
        this.allocator.forceRecycleAll(Collections.emptyList());
    }

    public void sync(boolean syncAllSegments, boolean flush) {
        CommitLogSegment current = this.allocator.allocatingFrom();
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            if (!syncAllSegments && segment.id > current.id) {
                return;
            }
            segment.sync(flush);
        }
    }

    public void requestExtraSync() {
        this.executor.requestExtraSync();
    }

    public ReplayPosition add(Mutation mutation) {
        assert (mutation != null);
        int size = (int)Mutation.serializer.serializedSize(mutation, MessagingService.current_version);
        int totalSize = size + 12;
        if ((long)totalSize > this.MAX_MUTATION_SIZE) {
            throw new IllegalArgumentException(String.format("Mutation of %s bytes is too large for the maximum size of %s", totalSize, this.MAX_MUTATION_SIZE));
        }
        CommitLogSegment.Allocation alloc = this.allocator.allocate(mutation, totalSize);
        CRC32 checksum = new CRC32();
        ByteBuffer buffer = alloc.getBuffer();
        try (DataOutputBufferFixed dos = new DataOutputBufferFixed(buffer);){
            dos.writeInt(size);
            FBUtilities.updateChecksumInt(checksum, size);
            buffer.putInt((int)checksum.getValue());
            Mutation.serializer.serialize(mutation, (DataOutputPlus)dos, MessagingService.current_version);
            FBUtilities.updateChecksum(checksum, buffer, buffer.position() - size, size);
            buffer.putInt((int)checksum.getValue());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, alloc.getSegment().getPath());
        }
        finally {
            alloc.markWritten();
        }
        this.executor.finishWriteFor(alloc);
        return alloc.getReplayPosition();
    }

    public void discardCompletedSegments(UUID cfId, ReplayPosition lowerBound, ReplayPosition upperBound) {
        logger.trace("discard completed log segments for {}-{}, table {}", new Object[]{lowerBound, upperBound, cfId});
        Iterator<CommitLogSegment> iter = this.allocator.getActiveSegments().iterator();
        while (iter.hasNext()) {
            CommitLogSegment segment = iter.next();
            segment.markClean(cfId, lowerBound, upperBound);
            if (segment.isUnused()) {
                logger.trace("Commit log segment {} is unused", (Object)segment);
                this.allocator.recycleSegment(segment);
            } else if (logger.isTraceEnabled()) {
                logger.trace("Not safe to delete{} commit log segment {}; dirty is {}", new Object[]{iter.hasNext() ? "" : " active", segment, segment.dirtyString()});
            }
            if (!segment.contains(upperBound)) continue;
            break;
        }
    }

    @Override
    public String getArchiveCommand() {
        return this.archiver.archiveCommand;
    }

    @Override
    public String getRestoreCommand() {
        return this.archiver.restoreCommand;
    }

    @Override
    public String getRestoreDirectories() {
        return this.archiver.restoreDirectories;
    }

    @Override
    public long getRestorePointInTime() {
        return this.archiver.restorePointInTime;
    }

    @Override
    public String getRestorePrecision() {
        return this.archiver.precision.toString();
    }

    @Override
    public List<String> getActiveSegmentNames() {
        ArrayList<String> segmentNames = new ArrayList<String>();
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            segmentNames.add(segment.getName());
        }
        return segmentNames;
    }

    @Override
    public List<String> getArchivingSegmentNames() {
        return new ArrayList<String>(this.archiver.archivePending.keySet());
    }

    @Override
    public long getActiveContentSize() {
        long size = 0L;
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            size += segment.contentSize();
        }
        return size;
    }

    @Override
    public long getActiveOnDiskSize() {
        return this.allocator.onDiskSize();
    }

    @Override
    public Map<String, Double> getActiveSegmentCompressionRatios() {
        TreeMap<String, Double> segmentRatios = new TreeMap<String, Double>();
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            segmentRatios.put(segment.getName(), 1.0 * (double)segment.onDiskSize() / (double)segment.contentSize());
        }
        return segmentRatios;
    }

    public void shutdownBlocking() throws InterruptedException {
        this.executor.shutdown();
        this.executor.awaitTermination();
        this.allocator.shutdown();
        this.allocator.awaitTermination();
    }

    public int resetUnsafe(boolean deleteSegments) throws IOException {
        this.stopUnsafe(deleteSegments);
        this.resetConfiguration();
        return this.restartUnsafe();
    }

    public void stopUnsafe(boolean deleteSegments) {
        this.executor.shutdown();
        try {
            this.executor.awaitTermination();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        this.allocator.stopUnsafe(deleteSegments);
        CommitLogSegment.resetReplayLimit();
    }

    public void resetConfiguration() {
        this.configuration = new Configuration(DatabaseDescriptor.getCommitLogCompression());
    }

    public int restartUnsafe() throws IOException {
        this.allocator.start();
        this.executor.restartUnsafe();
        try {
            return this.recover();
        }
        catch (FSWriteError e) {
            return 0;
        }
    }

    public int activeSegments() {
        return this.allocator.getActiveSegments().size();
    }

    @VisibleForTesting
    public static boolean handleCommitError(String message, Throwable t) {
        JVMStabilityInspector.inspectCommitLogThrowable(t);
        switch (DatabaseDescriptor.getCommitFailurePolicy()) {
            case die: 
            case stop: {
                StorageService.instance.stopTransports();
            }
            case stop_commit: {
                logger.error(String.format("%s. Commit disk failure policy is %s; terminating thread", new Object[]{message, DatabaseDescriptor.getCommitFailurePolicy()}), t);
                return false;
            }
            case ignore: {
                logger.error(message, t);
                return true;
            }
        }
        throw new AssertionError((Object)DatabaseDescriptor.getCommitFailurePolicy());
    }

    public static final class Configuration {
        private final ParameterizedClass compressorClass;
        private final ICompressor compressor;

        public Configuration(ParameterizedClass compressorClass) {
            this.compressorClass = compressorClass;
            this.compressor = compressorClass != null ? CompressionParams.createCompressor(compressorClass) : null;
        }

        public boolean useCompression() {
            return this.compressor != null;
        }

        public ICompressor getCompressor() {
            return this.compressor;
        }

        public ParameterizedClass getCompressorClass() {
            return this.compressorClass;
        }

        public String getCompressorName() {
            return this.useCompression() ? this.compressor.getClass().getSimpleName() : "none";
        }
    }
}

