/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.lifecycle;

import com.google.common.collect.Iterables;
import java.io.File;
import java.io.Serializable;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.LogRecord;
import org.apache.cassandra.db.lifecycle.LogTransaction;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.format.big.BigFormat;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.Throwables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class LogFile {
    private static final Logger logger = LoggerFactory.getLogger(LogFile.class);
    static String EXT = ".log";
    static char SEP = (char)95;
    static Pattern FILE_REGEX = Pattern.compile(String.format("^(.{2})_txn_(.*)_(.*)%s$", EXT));
    final File file;
    final Set<LogRecord> records = new LinkedHashSet<LogRecord>();
    final OperationType opType;
    final UUID id;
    final File folder;
    final int folderDescriptor;

    static LogFile make(File logFile, int folderDescriptor) {
        Matcher matcher = FILE_REGEX.matcher(logFile.getName());
        assert (matcher.matches() && matcher.groupCount() == 3);
        OperationType operationType = OperationType.fromFileName(matcher.group(2));
        UUID id = UUID.fromString(matcher.group(3));
        return new LogFile(operationType, logFile.getParentFile(), folderDescriptor, id);
    }

    void sync() {
        if (this.folderDescriptor > 0) {
            CLibrary.trySync(this.folderDescriptor);
        }
    }

    OperationType getType() {
        return this.opType;
    }

    UUID getId() {
        return this.id;
    }

    Throwable removeUnfinishedLeftovers(Throwable accumulate) {
        try {
            this.deleteRecords(this.committed() ? LogRecord.Type.REMOVE : LogRecord.Type.ADD);
            this.sync();
            Files.delete(this.file.toPath());
        }
        catch (Throwable t) {
            accumulate = Throwables.merge(accumulate, t);
        }
        return accumulate;
    }

    static boolean isLogFile(File file) {
        return FILE_REGEX.matcher(file.getName()).matches();
    }

    LogFile(OperationType opType, File folder, int folderDescriptor, UUID id) {
        this.opType = opType;
        this.id = id;
        this.folder = folder;
        this.file = new File(LogFile.getFileName(folder, opType, id));
        this.folderDescriptor = folderDescriptor;
    }

    public void readRecords() {
        assert (this.records.isEmpty());
        FileUtils.readLines(this.file).stream().map(LogRecord::make).forEach(this.records::add);
    }

    public boolean verify() {
        Optional<LogRecord> firstInvalid = this.records.stream().filter(this::isInvalid).findFirst();
        if (!firstInvalid.isPresent()) {
            return true;
        }
        LogRecord failedOn = firstInvalid.get();
        if (this.getLastRecord() != failedOn) {
            LogFile.logError(failedOn);
            return false;
        }
        if (this.records.stream().filter(r -> r != failedOn).filter(LogFile::isInvalidWithCorruptedLastRecord).map(LogFile::logError).findFirst().isPresent()) {
            LogFile.logError(failedOn);
            return false;
        }
        logger.warn(String.format("Last record of transaction %s is corrupt or incomplete [%s], but all previous records match state on disk; continuing", this.id, failedOn.error));
        return true;
    }

    static LogRecord logError(LogRecord record) {
        logger.error("{}", (Object)record.error);
        return record;
    }

    boolean isInvalid(LogRecord record) {
        if (!record.isValid()) {
            return true;
        }
        if (record.type == LogRecord.Type.UNKNOWN) {
            record.error(String.format("Could not parse record [%s]", record));
            return true;
        }
        if (record.checksum != record.computeChecksum()) {
            record.error(String.format("Invalid checksum for sstable [%s], record [%s]: [%d] should have been [%d]", record.relativeFilePath, record, record.checksum, record.computeChecksum()));
            return true;
        }
        if (record.type != LogRecord.Type.REMOVE) {
            return false;
        }
        List<File> files = record.getExistingFiles(this.folder);
        record.onDiskRecord = LogRecord.make(record.type, files, 0, record.relativeFilePath);
        if (record.updateTime != record.onDiskRecord.updateTime && record.onDiskRecord.numFiles > 0) {
            record.error(String.format("Unexpected files detected for sstable [%s], record [%s]: last update time [%tT] should have been [%tT]", record.relativeFilePath, record, record.onDiskRecord.updateTime, record.updateTime));
            return true;
        }
        return false;
    }

    static boolean isInvalidWithCorruptedLastRecord(LogRecord record) {
        if (record.type == LogRecord.Type.REMOVE && record.onDiskRecord.numFiles < record.numFiles) {
            record.error(String.format("Incomplete fileset detected for sstable [%s], record [%s]: number of files [%d] should have been [%d]. Treating as unrecoverable due to corruption of the final record.", record.relativeFilePath, record.raw, record.onDiskRecord.numFiles, record.numFiles));
            return true;
        }
        return false;
    }

    public void commit() {
        assert (!this.completed()) : "Already completed!";
        this.addRecord(LogRecord.makeCommit(System.currentTimeMillis()));
    }

    public void abort() {
        assert (!this.completed()) : "Already completed!";
        this.addRecord(LogRecord.makeAbort(System.currentTimeMillis()));
    }

    private boolean isLastRecordValidWithType(LogRecord.Type type) {
        LogRecord lastRecord = this.getLastRecord();
        return lastRecord != null && lastRecord.type == type && !this.isInvalid(lastRecord);
    }

    public boolean committed() {
        return this.isLastRecordValidWithType(LogRecord.Type.COMMIT);
    }

    public boolean aborted() {
        return this.isLastRecordValidWithType(LogRecord.Type.ABORT);
    }

    public boolean completed() {
        return this.committed() || this.aborted();
    }

    public void add(LogRecord.Type type, SSTable table) {
        if (!this.addRecord(this.makeRecord(type, table))) {
            throw new IllegalStateException();
        }
    }

    private LogRecord makeRecord(LogRecord.Type type, SSTable table) {
        assert (type == LogRecord.Type.ADD || type == LogRecord.Type.REMOVE);
        return LogRecord.make(type, this.folder, table);
    }

    private boolean addRecord(LogRecord record) {
        if (!this.records.add(record)) {
            return false;
        }
        FileUtils.append(this.file, record.toString());
        this.sync();
        return true;
    }

    public void remove(LogRecord.Type type, SSTable table) {
        LogRecord record = this.makeRecord(type, table);
        assert (this.records.contains(record)) : String.format("[%s] is not tracked by %s", record, this.file);
        this.records.remove(record);
        this.deleteRecord(record);
    }

    public boolean contains(LogRecord.Type type, SSTable table) {
        return this.records.contains(this.makeRecord(type, table));
    }

    public void deleteRecords(LogRecord.Type type) {
        assert (this.file.exists()) : String.format("Expected %s to exists", this.file);
        this.records.stream().filter(type::matches).forEach(this::deleteRecord);
        this.records.clear();
    }

    private void deleteRecord(LogRecord record) {
        List<File> files = record.getExistingFiles(this.folder);
        files.sort((f1, f2) -> Long.compare(f1.lastModified(), f2.lastModified()));
        files.forEach(LogTransaction::delete);
    }

    public Map<LogRecord, Set<File>> getFilesOfType(NavigableSet<File> files, LogRecord.Type type) {
        HashMap<LogRecord, Set<File>> ret = new HashMap<LogRecord, Set<File>>();
        this.records.stream().filter(type::matches).filter(LogRecord::isValid).forEach(r -> ret.put((LogRecord)r, this.getRecordFiles(files, (LogRecord)r)));
        return ret;
    }

    public LogRecord getLastRecord() {
        return (LogRecord)Iterables.getLast(this.records, null);
    }

    private Set<File> getRecordFiles(NavigableSet<File> files, LogRecord record) {
        File file;
        HashSet<File> ret = new HashSet<File>();
        Iterator iterator = files.tailSet(new File(this.folder, record.relativeFilePath)).iterator();
        while (iterator.hasNext() && (file = (File)iterator.next()).getName().startsWith(record.relativeFilePath)) {
            ret.add(file);
        }
        return ret;
    }

    public void delete() {
        LogTransaction.delete(this.file);
    }

    public boolean exists() {
        return this.file.exists();
    }

    public String toString() {
        return FileUtils.getRelativePath(this.folder.getPath(), this.file.getPath());
    }

    static String getFileName(File folder, OperationType opType, UUID id) {
        String fileName = StringUtils.join((Object[])new Object[]{BigFormat.latestVersion, Character.valueOf(SEP), "txn", Character.valueOf(SEP), opType.fileName, Character.valueOf(SEP), id.toString(), EXT});
        return StringUtils.join((Object[])new Serializable[]{folder, File.separator, fileName});
    }
}

