/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.lifecycle;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Runnables;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.SecureDirectoryStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.Tracker;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.format.big.BigFormat;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.Throwables;
import org.apache.cassandra.utils.UUIDGen;
import org.apache.cassandra.utils.concurrent.Ref;
import org.apache.cassandra.utils.concurrent.RefCounted;
import org.apache.cassandra.utils.concurrent.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransactionLog
extends Transactional.AbstractTransactional
implements Transactional {
    private static final Logger logger = LoggerFactory.getLogger(TransactionLog.class);
    private static final NoSpamLogger noSpamLogger = NoSpamLogger.getLogger(logger, 1L, TimeUnit.HOURS);
    private final Tracker tracker;
    private final TransactionData data;
    private final Ref<TransactionLog> selfRef;
    private static final Queue<Runnable> failedDeletions = new ConcurrentLinkedQueue<Runnable>();

    TransactionLog(OperationType opType, CFMetaData metadata) {
        this(opType, metadata, null);
    }

    TransactionLog(OperationType opType, CFMetaData metadata, Tracker tracker) {
        this(opType, new Directories(metadata), tracker);
    }

    TransactionLog(OperationType opType, Directories directories, Tracker tracker) {
        this(opType, directories.getDirectoryForNewSSTables(), tracker);
    }

    TransactionLog(OperationType opType, File folder, Tracker tracker) {
        this.tracker = tracker;
        this.data = new TransactionData(opType, folder, UUIDGen.getTimeUUID());
        this.selfRef = new Ref<TransactionLog>(this, new TransactionTidier(this.data));
        if (logger.isDebugEnabled()) {
            logger.debug("Created transaction logs with id {}", (Object)this.data.id);
        }
    }

    void trackNew(SSTable table) {
        if (!this.data.file.add(RecordType.ADD, table)) {
            throw new IllegalStateException(table + " is already tracked as new");
        }
    }

    void untrackNew(SSTable table) {
        this.data.file.remove(RecordType.ADD, table);
    }

    SSTableTidier obsoleted(SSTableReader reader) {
        if (this.data.file.contains(RecordType.ADD, reader)) {
            if (this.data.file.contains(RecordType.REMOVE, reader)) {
                throw new IllegalArgumentException();
            }
            return new SSTableTidier(reader, true, this);
        }
        if (!this.data.file.add(RecordType.REMOVE, reader)) {
            throw new IllegalStateException();
        }
        if (this.tracker != null) {
            this.tracker.notifyDeleting(reader);
        }
        return new SSTableTidier(reader, false, this);
    }

    OperationType getType() {
        return this.data.getType();
    }

    UUID getId() {
        return this.data.getId();
    }

    @VisibleForTesting
    String getDataFolder() {
        return this.data.getFolder();
    }

    @VisibleForTesting
    TransactionData getData() {
        return this.data;
    }

    private static void delete(File file) {
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Deleting {}", (Object)file);
            }
            Files.delete(file.toPath());
        }
        catch (NoSuchFileException e) {
            logger.error("Unable to delete {} as it does not exist", (Object)file);
        }
        catch (IOException e) {
            logger.error("Unable to delete {}", (Object)file, (Object)e);
            throw new RuntimeException(e);
        }
    }

    public static void rescheduleFailedDeletions() {
        Runnable task;
        while (null != (task = failedDeletions.poll())) {
            ScheduledExecutors.nonPeriodicTasks.submit(task);
        }
    }

    public static void waitForDeletions() {
        FBUtilities.waitOnFuture(ScheduledExecutors.nonPeriodicTasks.schedule(Runnables.doNothing(), 0L, TimeUnit.MILLISECONDS));
    }

    @VisibleForTesting
    Throwable complete(Throwable accumulate) {
        try {
            accumulate = this.selfRef.ensureReleased(accumulate);
            return accumulate;
        }
        catch (Throwable t) {
            logger.error("Failed to complete file transaction {}", (Object)this.getId(), (Object)t);
            return Throwables.merge(accumulate, t);
        }
    }

    @Override
    protected Throwable doCommit(Throwable accumulate) {
        this.data.file.commit();
        return this.complete(accumulate);
    }

    @Override
    protected Throwable doAbort(Throwable accumulate) {
        this.data.file.abort();
        return this.complete(accumulate);
    }

    @Override
    protected void doPrepare() {
    }

    static void removeUnfinishedLeftovers(CFMetaData metadata) {
        Throwable accumulate = null;
        for (File dir : new Directories(metadata).getCFDirectories()) {
            File[] logs;
            for (File log : logs = dir.listFiles((dir1, name) -> TransactionData.isLogFile(name))) {
                try (TransactionData data = TransactionData.make(log);){
                    accumulate = data.readLogFile(accumulate);
                    if (accumulate == null) {
                        accumulate = data.removeUnfinishedLeftovers(accumulate);
                        continue;
                    }
                    logger.error("Possible disk corruption: failed to read transaction log {}", (Object)log, (Object)accumulate);
                }
            }
        }
        if (accumulate != null) {
            logger.error("Failed to remove unfinished transaction leftovers", accumulate);
        }
    }

    @VisibleForTesting
    static Set<File> getTemporaryFiles(CFMetaData metadata, File folder) {
        HashSet<File> ret = new HashSet<File>();
        List<File> directories = new Directories(metadata).getCFDirectories();
        directories.add(folder);
        for (File dir : directories) {
            ret.addAll(new FileLister(dir.toPath(), (file, type) -> type != Directories.FileType.FINAL, Directories.OnTxnErr.IGNORE).list());
        }
        return ret;
    }

    static final class FileLister {
        private static final int MAX_ATTEMPTS = 5;
        private static final int REATTEMPT_DELAY_MILLIS = 5;
        private final Path folder;
        private final BiFunction<File, Directories.FileType, Boolean> filter;
        private final Directories.OnTxnErr onTxnErr;
        private int attempts;

        public FileLister(Path folder, BiFunction<File, Directories.FileType, Boolean> filter, Directories.OnTxnErr onTxnErr) {
            this.folder = folder;
            this.filter = filter;
            this.onTxnErr = onTxnErr;
            this.attempts = 0;
        }

        public List<File> list() {
            while (true) {
                try {
                    return this.attemptList();
                }
                catch (Throwable t) {
                    if (this.attempts >= 5) {
                        throw new RuntimeException(String.format("Failed to list files in %s after multiple attempts, giving up", this.folder), t);
                    }
                    logger.warn("Failed to list files in {} : {}", (Object)this.folder, (Object)t.getMessage());
                    try {
                        Thread.sleep(5L);
                    }
                    catch (InterruptedException e) {
                        logger.error("Interrupted whilst waiting to reattempt listing files in {}, giving up", (Object)this.folder, (Object)e);
                        throw new RuntimeException(String.format("Failed to list files in %s due to interruption, giving up", this.folder), t);
                    }
                }
            }
        }

        List<File> attemptList() throws IOException {
            ++this.attempts;
            HashMap files = new HashMap();
            try (DirectoryStream<Path> in = Files.newDirectoryStream(this.folder);){
                if (!(in instanceof SecureDirectoryStream)) {
                    noSpamLogger.warn("This platform does not support atomic directory streams (SecureDirectoryStream); race conditions when loading sstable files could occurr", new Object[0]);
                }
                in.forEach(path -> {
                    File file = path.toFile();
                    if (file.isDirectory()) {
                        return;
                    }
                    if (TransactionData.isLogFile(file.getName())) {
                        Set<File> tmpFiles = this.getTemporaryFiles(file);
                        if (tmpFiles != null) {
                            tmpFiles.stream().forEach(f -> files.put(f, Directories.FileType.TEMPORARY));
                            files.put(file, Directories.FileType.TXN_LOG);
                        }
                    } else {
                        files.putIfAbsent(file, Directories.FileType.FINAL);
                    }
                });
            }
            return files.entrySet().stream().filter(e -> this.filter.apply((File)e.getKey(), (Directories.FileType)((Object)((Object)e.getValue())))).map(Map.Entry::getKey).collect(Collectors.toList());
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        Set<File> getTemporaryFiles(File file) {
            try (TransactionData txn = TransactionData.make(file);){
                Throwables.maybeFail(txn.readLogFile(null));
                Set<File> set = txn.getTemporaryFiles();
                return set;
            }
            catch (Throwable t) {
                if (this.attempts < 5) throw new RuntimeException(t);
                if (this.onTxnErr == Directories.OnTxnErr.THROW) {
                    throw new RuntimeException(t);
                }
                logger.error("Failed to read temporary files of txn log {}", (Object)file, (Object)t);
                return null;
            }
        }
    }

    public static class SSTableTidier
    implements Runnable {
        private final Descriptor desc;
        private final long sizeOnDisk;
        private final Tracker tracker;
        private final boolean wasNew;
        private final Ref<TransactionLog> parentRef;

        public SSTableTidier(SSTableReader referent, boolean wasNew, TransactionLog parent) {
            this.desc = referent.descriptor;
            this.sizeOnDisk = referent.bytesOnDisk();
            this.tracker = parent.tracker;
            this.wasNew = wasNew;
            this.parentRef = parent.selfRef.tryRef();
        }

        @Override
        public void run() {
            SystemKeyspace.clearSSTableReadMeter(this.desc.ksname, this.desc.cfname, this.desc.generation);
            try {
                File datafile = new File(this.desc.filenameFor(Component.DATA));
                TransactionLog.delete(datafile);
                SSTable.delete(this.desc, SSTable.discoverComponentsFor(this.desc));
            }
            catch (Throwable t) {
                logger.error("Failed deletion for {}, we'll retry after GC and on server restart", (Object)this.desc);
                failedDeletions.add(this);
                return;
            }
            if (this.tracker != null && this.tracker.cfstore != null && !this.wasNew) {
                this.tracker.cfstore.metric.totalDiskSpaceUsed.dec(this.sizeOnDisk);
            }
            this.parentRef.release();
        }

        public void abort() {
            this.parentRef.release();
        }
    }

    static class Obsoletion {
        final SSTableReader reader;
        final SSTableTidier tidier;

        public Obsoletion(SSTableReader reader, SSTableTidier tidier) {
            this.reader = reader;
            this.tidier = tidier;
        }
    }

    private static class TransactionTidier
    implements RefCounted.Tidy,
    Runnable {
        private final TransactionData data;

        public TransactionTidier(TransactionData data) {
            this.data = data;
        }

        @Override
        public void tidy() throws Exception {
            this.run();
        }

        @Override
        public String name() {
            return this.data.toString();
        }

        @Override
        public void run() {
            if (logger.isDebugEnabled()) {
                logger.debug("Removing files for transaction {}", (Object)this.name());
            }
            assert (this.data.completed()) : "Expected a completed transaction: " + this.data;
            Throwable err = this.data.removeUnfinishedLeftovers(null);
            if (err != null) {
                logger.info("Failed deleting files for transaction {}, we'll retry after GC and on on server restart", (Object)this.name(), (Object)err);
                failedDeletions.add(this);
            } else {
                if (logger.isDebugEnabled()) {
                    logger.debug("Closing file transaction {}", (Object)this.name());
                }
                this.data.close();
            }
        }
    }

    static final class TransactionData
    implements AutoCloseable {
        private final OperationType opType;
        private final UUID id;
        private final File folder;
        private final TransactionFile file;
        private int folderDescriptor;

        static TransactionData make(File logFile) {
            Matcher matcher = TransactionFile.FILE_REGEX.matcher(logFile.getName());
            assert (matcher.matches() && matcher.groupCount() == 3);
            OperationType operationType = OperationType.fromFileName(matcher.group(2));
            UUID id = UUID.fromString(matcher.group(3));
            return new TransactionData(operationType, logFile.getParentFile(), id);
        }

        TransactionData(OperationType opType, File folder, UUID id) {
            this.opType = opType;
            this.id = id;
            this.folder = folder;
            this.file = new TransactionFile(this);
            this.folderDescriptor = CLibrary.tryOpenDirectory(folder.getPath());
        }

        public Throwable readLogFile(Throwable accumulate) {
            try {
                this.file.readRecords();
            }
            catch (Throwable t) {
                accumulate = Throwables.merge(accumulate, t);
            }
            return accumulate;
        }

        @Override
        public void close() {
            if (this.folderDescriptor > 0) {
                CLibrary.tryCloseFD(this.folderDescriptor);
                this.folderDescriptor = -1;
            }
        }

        void sync() {
            if (this.folderDescriptor > 0) {
                CLibrary.trySync(this.folderDescriptor);
            }
        }

        OperationType getType() {
            return this.opType;
        }

        UUID getId() {
            return this.id;
        }

        boolean completed() {
            return this.file.completed();
        }

        Throwable removeUnfinishedLeftovers(Throwable accumulate) {
            try {
                if (this.file.committed()) {
                    this.file.deleteRecords(RecordType.REMOVE);
                } else {
                    this.file.deleteRecords(RecordType.ADD);
                }
                this.sync();
                this.file.delete();
            }
            catch (Throwable t) {
                accumulate = Throwables.merge(accumulate, t);
            }
            return accumulate;
        }

        Set<File> getTemporaryFiles() {
            this.sync();
            if (!this.file.exists()) {
                return Collections.emptySet();
            }
            if (this.file.committed()) {
                return this.file.getTrackedFiles(RecordType.REMOVE);
            }
            return this.file.getTrackedFiles(RecordType.ADD);
        }

        String getFileName() {
            String fileName = StringUtils.join((Object[])new Object[]{BigFormat.latestVersion, Character.valueOf(TransactionFile.SEP), "txn", Character.valueOf(TransactionFile.SEP), this.opType.fileName, Character.valueOf(TransactionFile.SEP), this.id.toString(), TransactionFile.EXT});
            return StringUtils.join((Object[])new Serializable[]{this.folder, File.separator, fileName});
        }

        String getFolder() {
            return this.folder.getPath();
        }

        static boolean isLogFile(String name) {
            return TransactionFile.FILE_REGEX.matcher(name).matches();
        }

        @VisibleForTesting
        TransactionFile getLogFile() {
            return this.file;
        }

        public String toString() {
            return String.format("[%s]", this.file.toString());
        }
    }

    static final class TransactionFile {
        static String EXT = ".log";
        static char SEP = (char)95;
        static String FILE_REGEX_STR = String.format("^(.{2})_txn_(.*)_(.*)%s$", EXT);
        static Pattern FILE_REGEX = Pattern.compile(FILE_REGEX_STR);
        static String LINE_REGEX_STR = "^(.*)\\[(\\d*)\\]$";
        static Pattern LINE_REGEX = Pattern.compile(LINE_REGEX_STR);
        public final File file;
        public final TransactionData parent;
        public final Set<Record> records = new HashSet<Record>();
        public final Checksum checksum = new CRC32();

        public TransactionFile(TransactionData parent) {
            this.file = new File(parent.getFileName());
            this.parent = parent;
        }

        public void readRecords() {
            this.records.clear();
            this.checksum.reset();
            Iterator<String> it = FileUtils.readLines(this.file).iterator();
            while (it.hasNext()) {
                this.records.add(this.readRecord(it.next(), !it.hasNext()));
            }
            for (Record record : this.records) {
                if (record.verify(this.parent.getFolder(), false)) continue;
                throw new CorruptTransactionLogException(String.format("Failed to verify transaction %s record [%s]: possible disk corruption, aborting", this.parent.getId(), record), this);
            }
        }

        private Record readRecord(String line, boolean isLast) {
            Matcher matcher = LINE_REGEX.matcher(line);
            if (!matcher.matches() || matcher.groupCount() != 2) {
                this.handleReadRecordError(String.format("cannot parse line \"%s\"", line), isLast);
                return Record.make(line, isLast);
            }
            byte[] bytes = matcher.group(1).getBytes(FileUtils.CHARSET);
            this.checksum.update(bytes, 0, bytes.length);
            if (this.checksum.getValue() != Long.valueOf(matcher.group(2)).longValue()) {
                this.handleReadRecordError(String.format("invalid line checksum %s for \"%s\"", matcher.group(2), line), isLast);
            }
            try {
                return Record.make(matcher.group(1), isLast);
            }
            catch (Throwable t) {
                throw new CorruptTransactionLogException(String.format("Cannot make record \"%s\": %s", line, t.getMessage()), this);
            }
        }

        private void handleReadRecordError(String message, boolean isLast) {
            if (isLast) {
                for (Record record : this.records) {
                    if (record.verify(this.parent.getFolder(), true)) continue;
                    throw new CorruptTransactionLogException(String.format("Last record of transaction %s is corrupt [%s] and at least one previous record does not match state on disk, possible disk corruption, aborting", this.parent.getId(), message), this);
                }
            } else {
                throw new CorruptTransactionLogException(String.format("Non-last record of transaction %s is corrupt [%s], possible disk corruption, aborting", this.parent.getId(), message), this);
            }
            logger.warn(String.format("Last record of transaction %s is corrupt or incomplete [%s], but all previous records match state on disk; continuing", this.parent.getId(), message));
        }

        public void commit() {
            assert (!this.completed()) : "Already completed!";
            this.addRecord(Record.makeCommit(System.currentTimeMillis()));
        }

        public void abort() {
            assert (!this.completed()) : "Already completed!";
            this.addRecord(Record.makeAbort(System.currentTimeMillis()));
        }

        public boolean committed() {
            return this.records.contains(Record.makeCommit(0L));
        }

        public boolean aborted() {
            return this.records.contains(Record.makeAbort(0L));
        }

        public boolean completed() {
            return this.committed() || this.aborted();
        }

        public boolean add(RecordType type, SSTable table) {
            Record record = this.makeRecord(type, table);
            if (this.records.contains(record)) {
                return false;
            }
            this.addRecord(record);
            return true;
        }

        private Record makeRecord(RecordType type, SSTable table) {
            String relativePath = FileUtils.getRelativePath(this.parent.getFolder(), table.descriptor.baseFilename());
            if (type == RecordType.ADD) {
                return Record.makeNew(relativePath);
            }
            if (type == RecordType.REMOVE) {
                return Record.makeOld(this.parent.getFolder(), relativePath);
            }
            throw new AssertionError((Object)("Invalid record type " + (Object)((Object)type)));
        }

        private void addRecord(Record record) {
            byte[] bytes = record.getBytes();
            this.checksum.update(bytes, 0, bytes.length);
            this.records.add(record);
            FileUtils.append(this.file, String.format("%s[%d]", record, this.checksum.getValue()));
            this.parent.sync();
        }

        public void remove(RecordType type, SSTable table) {
            Record record = this.makeRecord(type, table);
            assert (this.records.contains(record)) : String.format("[%s] is not tracked by %s", record, this.file);
            this.records.remove(record);
            this.deleteRecord(record);
        }

        public boolean contains(RecordType type, SSTable table) {
            return this.records.contains(this.makeRecord(type, table));
        }

        public void deleteRecords(RecordType type) {
            assert (this.file.exists()) : String.format("Expected %s to exists", this.file);
            this.records.stream().filter(r -> r.type == type).forEach(this::deleteRecord);
            this.records.clear();
        }

        private void deleteRecord(Record record) {
            List<File> files = record.getTrackedFiles(this.parent.getFolder());
            if (files.isEmpty()) {
                return;
            }
            files.sort((f1, f2) -> Long.compare(f1.lastModified(), f2.lastModified()));
            files.forEach(x$0 -> TransactionLog.delete(x$0));
        }

        public Set<File> getTrackedFiles(RecordType type) {
            return this.records.stream().filter(r -> r.type == type).map(r -> r.getTrackedFiles(this.parent.getFolder())).flatMap(Collection::stream).collect(Collectors.toSet());
        }

        public void delete() {
            TransactionLog.delete(this.file);
        }

        public boolean exists() {
            return this.file.exists();
        }

        public String toString() {
            return FileUtils.getRelativePath(this.parent.getFolder(), FileUtils.getCanonicalPath(this.file));
        }
    }

    static final class Record {
        public final RecordType type;
        public final String relativeFilePath;
        public final long updateTime;
        public final int numFiles;
        public final String record;
        static String REGEX_STR = "^(add|remove|commit|abort):\\[([^,]*),?([^,]*),?([^,]*)\\]$";
        static Pattern REGEX = Pattern.compile(REGEX_STR, 2);

        public static Record make(String record, boolean isLast) {
            try {
                Matcher matcher = REGEX.matcher(record);
                if (!matcher.matches() || matcher.groupCount() != 4) {
                    throw new IllegalStateException(String.format("Invalid record \"%s\"", record));
                }
                RecordType type = RecordType.fromPrefix(matcher.group(1));
                return new Record(type, matcher.group(2), Long.valueOf(matcher.group(3)), Integer.valueOf(matcher.group(4)), record);
            }
            catch (Throwable t) {
                RecordType recordType;
                if (!isLast) {
                    throw t;
                }
                int pos = record.indexOf(58);
                if (pos <= 0) {
                    throw t;
                }
                try {
                    recordType = RecordType.fromPrefix(record.substring(0, pos));
                }
                catch (Throwable ignore) {
                    throw t;
                }
                return new Record(recordType, "", 0L, 0, record);
            }
        }

        public static Record makeCommit(long updateTime) {
            return new Record(RecordType.COMMIT, "", updateTime, 0, "");
        }

        public static Record makeAbort(long updateTime) {
            return new Record(RecordType.ABORT, "", updateTime, 0, "");
        }

        public static Record makeNew(String relativeFilePath) {
            return new Record(RecordType.ADD, relativeFilePath, 0L, 0, "");
        }

        public static Record makeOld(String parentFolder, String relativeFilePath) {
            return Record.makeOld(Record.getTrackedFiles(parentFolder, relativeFilePath), relativeFilePath);
        }

        public static Record makeOld(List<File> files, String relativeFilePath) {
            long lastModified = files.stream().mapToLong(File::lastModified).reduce(0L, Long::max);
            return new Record(RecordType.REMOVE, relativeFilePath, lastModified, files.size(), "");
        }

        private Record(RecordType type, String relativeFilePath, long updateTime, int numFiles, String record) {
            this.type = type;
            this.relativeFilePath = Record.hasFilePath(type) ? relativeFilePath : "";
            this.updateTime = type == RecordType.REMOVE ? updateTime : 0L;
            this.numFiles = type == RecordType.REMOVE ? numFiles : 0;
            this.record = record.isEmpty() ? this.format() : record;
        }

        private static boolean hasFilePath(RecordType type) {
            return type == RecordType.ADD || type == RecordType.REMOVE;
        }

        private String format() {
            return String.format("%s:[%s,%d,%d]", this.type.toString(), this.relativeFilePath, this.updateTime, this.numFiles);
        }

        public byte[] getBytes() {
            return this.record.getBytes(FileUtils.CHARSET);
        }

        public boolean verify(String parentFolder, boolean lastRecordIsCorrupt) {
            if (this.type != RecordType.REMOVE) {
                return true;
            }
            List<File> files = this.getTrackedFiles(parentFolder);
            Record currentRecord = Record.makeOld(files, this.relativeFilePath);
            if (this.updateTime != currentRecord.updateTime) {
                logger.error("Possible disk corruption detected for sstable [{}], record [{}]: last update time [{}] should have been [{}]", new Object[]{this.relativeFilePath, this.record, new Date(currentRecord.updateTime), new Date(this.updateTime)});
                return false;
            }
            if (lastRecordIsCorrupt && currentRecord.numFiles < this.numFiles) {
                logger.error("Possible disk corruption detected for sstable [{}], record [{}]: number of files [{}] should have been [{}]", new Object[]{this.relativeFilePath, this.record, currentRecord.numFiles, this.numFiles});
                return false;
            }
            return true;
        }

        public List<File> getTrackedFiles(String parentFolder) {
            if (!Record.hasFilePath(this.type)) {
                return Collections.emptyList();
            }
            return Record.getTrackedFiles(parentFolder, this.relativeFilePath);
        }

        public static List<File> getTrackedFiles(String parentFolder, String relativeFilePath) {
            return Arrays.asList(new File(parentFolder).listFiles((dir, name) -> name.startsWith(relativeFilePath)));
        }

        public int hashCode() {
            return Objects.hash(new Object[]{this.type, this.relativeFilePath});
        }

        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Record other = (Record)obj;
            return this.type.equals((Object)other.type) && this.relativeFilePath.equals(other.relativeFilePath);
        }

        public String toString() {
            return this.record;
        }
    }

    public static enum RecordType {
        ADD,
        REMOVE,
        COMMIT,
        ABORT;


        public static RecordType fromPrefix(String prefix) {
            return RecordType.valueOf(prefix.toUpperCase());
        }
    }

    public static final class CorruptTransactionLogException
    extends RuntimeException {
        public final TransactionFile file;

        public CorruptTransactionLogException(String message, TransactionFile file) {
            super(message);
            this.file = file;
        }
    }
}

