/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.google.common.util.concurrent.Futures;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.apache.cassandra.cache.AutoSavingCache;
import org.apache.cassandra.cache.CacheProvider;
import org.apache.cassandra.cache.ConcurrentLinkedHashCache;
import org.apache.cassandra.cache.CounterCacheKey;
import org.apache.cassandra.cache.ICache;
import org.apache.cassandra.cache.IRowCacheEntry;
import org.apache.cassandra.cache.KeyCacheKey;
import org.apache.cassandra.cache.RowCacheKey;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Cell;
import org.apache.cassandra.db.ClockAndCount;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.composites.CellName;
import org.apache.cassandra.db.context.CounterContext;
import org.apache.cassandra.db.filter.QueryFilter;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.service.CacheServiceMBean;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.MBeanWrapper;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CacheService
implements CacheServiceMBean {
    private static final Logger logger = LoggerFactory.getLogger(CacheService.class);
    public static final String MBEAN_NAME = "org.apache.cassandra.db:type=Caches";
    public static final CacheService instance = new CacheService();
    public final AutoSavingCache<KeyCacheKey, RowIndexEntry> keyCache;
    public final AutoSavingCache<RowCacheKey, IRowCacheEntry> rowCache;
    public final AutoSavingCache<CounterCacheKey, ClockAndCount> counterCache;

    private CacheService() {
        MBeanWrapper.instance.registerMBean((Object)this, MBEAN_NAME);
        this.keyCache = this.initKeyCache();
        this.rowCache = this.initRowCache();
        this.counterCache = this.initCounterCache();
    }

    private AutoSavingCache<KeyCacheKey, RowIndexEntry> initKeyCache() {
        logger.info("Initializing key cache with capacity of {} MBs.", (Object)DatabaseDescriptor.getKeyCacheSizeInMB());
        long keyCacheInMemoryCapacity = DatabaseDescriptor.getKeyCacheSizeInMB() * 1024L * 1024L;
        ConcurrentLinkedHashCache kc = ConcurrentLinkedHashCache.create(keyCacheInMemoryCapacity);
        AutoSavingCache<KeyCacheKey, RowIndexEntry> keyCache = new AutoSavingCache<KeyCacheKey, RowIndexEntry>(kc, CacheType.KEY_CACHE, new KeyCacheSerializer());
        int keyCacheKeysToSave = DatabaseDescriptor.getKeyCacheKeysToSave();
        keyCache.scheduleSaving(DatabaseDescriptor.getKeyCacheSavePeriod(), keyCacheKeysToSave);
        return keyCache;
    }

    private AutoSavingCache<RowCacheKey, IRowCacheEntry> initRowCache() {
        CacheProvider cacheProvider;
        logger.info("Initializing row cache with capacity of {} MBs", (Object)DatabaseDescriptor.getRowCacheSizeInMB());
        String cacheProviderClassName = DatabaseDescriptor.getRowCacheSizeInMB() > 0L ? DatabaseDescriptor.getRowCacheClassName() : "org.apache.cassandra.cache.NopCacheProvider";
        try {
            Class<?> cacheProviderClass = Class.forName(cacheProviderClassName);
            cacheProvider = (CacheProvider)cacheProviderClass.newInstance();
        }
        catch (Exception e) {
            throw new RuntimeException("Cannot find configured row cache provider class " + DatabaseDescriptor.getRowCacheClassName());
        }
        ICache rc = cacheProvider.create();
        AutoSavingCache<RowCacheKey, IRowCacheEntry> rowCache = new AutoSavingCache<RowCacheKey, IRowCacheEntry>(rc, CacheType.ROW_CACHE, new RowCacheSerializer());
        int rowCacheKeysToSave = DatabaseDescriptor.getRowCacheKeysToSave();
        rowCache.scheduleSaving(DatabaseDescriptor.getRowCacheSavePeriod(), rowCacheKeysToSave);
        return rowCache;
    }

    private AutoSavingCache<CounterCacheKey, ClockAndCount> initCounterCache() {
        logger.info("Initializing counter cache with capacity of {} MBs", (Object)DatabaseDescriptor.getCounterCacheSizeInMB());
        long capacity = DatabaseDescriptor.getCounterCacheSizeInMB() * 1024L * 1024L;
        AutoSavingCache<CounterCacheKey, ClockAndCount> cache = new AutoSavingCache<CounterCacheKey, ClockAndCount>(ConcurrentLinkedHashCache.create(capacity), CacheType.COUNTER_CACHE, new CounterCacheSerializer());
        int keysToSave = DatabaseDescriptor.getCounterCacheKeysToSave();
        logger.info("Scheduling counter cache save to every {} seconds (going to save {} keys).", (Object)DatabaseDescriptor.getCounterCacheSavePeriod(), keysToSave == Integer.MAX_VALUE ? "all" : Integer.valueOf(keysToSave));
        cache.scheduleSaving(DatabaseDescriptor.getCounterCacheSavePeriod(), keysToSave);
        return cache;
    }

    @Override
    public int getRowCacheSavePeriodInSeconds() {
        return DatabaseDescriptor.getRowCacheSavePeriod();
    }

    @Override
    public void setRowCacheSavePeriodInSeconds(int seconds) {
        if (seconds < 0) {
            throw new RuntimeException("RowCacheSavePeriodInSeconds must be non-negative.");
        }
        DatabaseDescriptor.setRowCacheSavePeriod(seconds);
        this.rowCache.scheduleSaving(seconds, DatabaseDescriptor.getRowCacheKeysToSave());
    }

    @Override
    public int getKeyCacheSavePeriodInSeconds() {
        return DatabaseDescriptor.getKeyCacheSavePeriod();
    }

    @Override
    public void setKeyCacheSavePeriodInSeconds(int seconds) {
        if (seconds < 0) {
            throw new RuntimeException("KeyCacheSavePeriodInSeconds must be non-negative.");
        }
        DatabaseDescriptor.setKeyCacheSavePeriod(seconds);
        this.keyCache.scheduleSaving(seconds, DatabaseDescriptor.getKeyCacheKeysToSave());
    }

    @Override
    public int getCounterCacheSavePeriodInSeconds() {
        return DatabaseDescriptor.getCounterCacheSavePeriod();
    }

    @Override
    public void setCounterCacheSavePeriodInSeconds(int seconds) {
        if (seconds < 0) {
            throw new RuntimeException("CounterCacheSavePeriodInSeconds must be non-negative.");
        }
        DatabaseDescriptor.setCounterCacheSavePeriod(seconds);
        this.counterCache.scheduleSaving(seconds, DatabaseDescriptor.getCounterCacheKeysToSave());
    }

    @Override
    public int getRowCacheKeysToSave() {
        return DatabaseDescriptor.getRowCacheKeysToSave();
    }

    @Override
    public void setRowCacheKeysToSave(int count) {
        if (count < 0) {
            throw new RuntimeException("RowCacheKeysToSave must be non-negative.");
        }
        DatabaseDescriptor.setRowCacheKeysToSave(count);
        this.rowCache.scheduleSaving(this.getRowCacheSavePeriodInSeconds(), count);
    }

    @Override
    public int getKeyCacheKeysToSave() {
        return DatabaseDescriptor.getKeyCacheKeysToSave();
    }

    @Override
    public void setKeyCacheKeysToSave(int count) {
        if (count < 0) {
            throw new RuntimeException("KeyCacheKeysToSave must be non-negative.");
        }
        DatabaseDescriptor.setKeyCacheKeysToSave(count);
        this.keyCache.scheduleSaving(this.getKeyCacheSavePeriodInSeconds(), count);
    }

    @Override
    public int getCounterCacheKeysToSave() {
        return DatabaseDescriptor.getCounterCacheKeysToSave();
    }

    @Override
    public void setCounterCacheKeysToSave(int count) {
        if (count < 0) {
            throw new RuntimeException("CounterCacheKeysToSave must be non-negative.");
        }
        DatabaseDescriptor.setCounterCacheKeysToSave(count);
        this.counterCache.scheduleSaving(this.getCounterCacheSavePeriodInSeconds(), count);
    }

    @Override
    public void invalidateKeyCache() {
        this.keyCache.clear();
    }

    public void invalidateKeyCacheForCf(Pair<String, String> ksAndCFName) {
        Iterator keyCacheIterator = this.keyCache.keyIterator();
        while (keyCacheIterator.hasNext()) {
            KeyCacheKey key = (KeyCacheKey)keyCacheIterator.next();
            if (!key.ksAndCFName.equals(ksAndCFName)) continue;
            keyCacheIterator.remove();
        }
    }

    @Override
    public void invalidateRowCache() {
        this.rowCache.clear();
    }

    public void invalidateRowCacheForCf(Pair<String, String> ksAndCFName) {
        Iterator rowCacheIterator = this.rowCache.keyIterator();
        while (rowCacheIterator.hasNext()) {
            RowCacheKey rowCacheKey = (RowCacheKey)rowCacheIterator.next();
            if (!rowCacheKey.ksAndCFName.equals(ksAndCFName)) continue;
            rowCacheIterator.remove();
        }
    }

    public void invalidateCounterCacheForCf(Pair<String, String> ksAndCFName) {
        Iterator counterCacheIterator = this.counterCache.keyIterator();
        while (counterCacheIterator.hasNext()) {
            CounterCacheKey counterCacheKey = (CounterCacheKey)counterCacheIterator.next();
            if (!counterCacheKey.ksAndCFName.equals(ksAndCFName)) continue;
            counterCacheIterator.remove();
        }
    }

    @Override
    public void invalidateCounterCache() {
        this.counterCache.clear();
    }

    @Override
    public void setRowCacheCapacityInMB(long capacity) {
        if (capacity < 0L) {
            throw new RuntimeException("capacity should not be negative.");
        }
        this.rowCache.setCapacity(capacity * 1024L * 1024L);
    }

    @Override
    public void setKeyCacheCapacityInMB(long capacity) {
        if (capacity < 0L) {
            throw new RuntimeException("capacity should not be negative.");
        }
        this.keyCache.setCapacity(capacity * 1024L * 1024L);
    }

    @Override
    public void setCounterCacheCapacityInMB(long capacity) {
        if (capacity < 0L) {
            throw new RuntimeException("capacity should not be negative.");
        }
        this.counterCache.setCapacity(capacity * 1024L * 1024L);
    }

    @Override
    public void saveCaches() throws ExecutionException, InterruptedException {
        ArrayList futures = new ArrayList(3);
        logger.debug("submitting cache saves");
        futures.add(this.keyCache.submitWrite(DatabaseDescriptor.getKeyCacheKeysToSave()));
        futures.add(this.rowCache.submitWrite(DatabaseDescriptor.getRowCacheKeysToSave()));
        futures.add(this.counterCache.submitWrite(DatabaseDescriptor.getCounterCacheKeysToSave()));
        FBUtilities.waitOnFutures(futures);
        logger.debug("cache saves completed");
    }

    public static class KeyCacheSerializer
    implements AutoSavingCache.CacheSerializer<KeyCacheKey, RowIndexEntry> {
        @Override
        public void serialize(KeyCacheKey key, DataOutputPlus out, ColumnFamilyStore cfs) throws IOException {
            RowIndexEntry entry = (RowIndexEntry)CacheService.instance.keyCache.getInternal(key);
            if (entry == null) {
                return;
            }
            out.write(cfs.metadata.ksAndCFBytes);
            ByteBufferUtil.writeWithLength(key.key, (DataOutput)out);
            out.writeInt(key.desc.generation);
            out.writeBoolean(true);
            key.desc.getFormat().getIndexSerializer(cfs.metadata).serialize(entry, out);
        }

        @Override
        public Future<Pair<KeyCacheKey, RowIndexEntry>> deserialize(DataInputStream input, ColumnFamilyStore cfs) throws IOException {
            int keyLength = input.readInt();
            if (keyLength > 65535) {
                throw new IOException(String.format("Corrupted key cache. Key length of %d is longer than maximum of %d", keyLength, 65535));
            }
            ByteBuffer key = ByteBufferUtil.read(input, keyLength);
            int generation = input.readInt();
            input.readBoolean();
            SSTableReader reader = null;
            if (cfs == null || !cfs.isKeyCacheEnabled() || (reader = this.findDesc(generation, cfs.getSSTables())) == null) {
                RowIndexEntry.Serializer.skip(input);
                return null;
            }
            RowIndexEntry<?> entry = reader.descriptor.getFormat().getIndexSerializer(reader.metadata).deserialize(input, reader.descriptor.version);
            return Futures.immediateFuture(Pair.create(new KeyCacheKey(cfs.metadata.ksAndCFName, reader.descriptor, key), entry));
        }

        private SSTableReader findDesc(int generation, Collection<SSTableReader> collection) {
            for (SSTableReader sstable : collection) {
                if (sstable.descriptor.generation != generation) continue;
                return sstable;
            }
            return null;
        }
    }

    public static class RowCacheSerializer
    implements AutoSavingCache.CacheSerializer<RowCacheKey, IRowCacheEntry> {
        @Override
        public void serialize(RowCacheKey key, DataOutputPlus out, ColumnFamilyStore cfs) throws IOException {
            assert (!cfs.isIndex());
            out.write(cfs.metadata.ksAndCFBytes);
            ByteBufferUtil.writeWithLength(key.key, (DataOutput)out);
        }

        @Override
        public Future<Pair<RowCacheKey, IRowCacheEntry>> deserialize(DataInputStream in, final ColumnFamilyStore cfs) throws IOException {
            final ByteBuffer buffer = ByteBufferUtil.readWithLength(in);
            if (cfs == null || !cfs.isRowCacheEnabled()) {
                return null;
            }
            assert (!cfs.isIndex());
            return StageManager.getStage(Stage.READ).submit(new Callable<Pair<RowCacheKey, IRowCacheEntry>>(){

                @Override
                public Pair<RowCacheKey, IRowCacheEntry> call() throws Exception {
                    DecoratedKey key = cfs.partitioner.decorateKey(buffer);
                    QueryFilter cacheFilter = new QueryFilter(key, cfs.getColumnFamilyName(), cfs.readFilterForCache(), Integer.MIN_VALUE);
                    ColumnFamily data = cfs.getTopLevelColumns(cacheFilter, Integer.MIN_VALUE);
                    return Pair.create(new RowCacheKey(cfs.metadata.ksAndCFName, key), data);
                }
            });
        }
    }

    public static class CounterCacheSerializer
    implements AutoSavingCache.CacheSerializer<CounterCacheKey, ClockAndCount> {
        @Override
        public void serialize(CounterCacheKey key, DataOutputPlus out, ColumnFamilyStore cfs) throws IOException {
            assert (cfs.metadata.isCounter());
            out.write(cfs.metadata.ksAndCFBytes);
            ByteBufferUtil.writeWithLength(key.partitionKey, (DataOutput)out);
            ByteBufferUtil.writeWithLength(key.cellName, (DataOutput)out);
        }

        @Override
        public Future<Pair<CounterCacheKey, ClockAndCount>> deserialize(DataInputStream in, final ColumnFamilyStore cfs) throws IOException {
            final ByteBuffer partitionKey = ByteBufferUtil.readWithLength(in);
            ByteBuffer cellNameBuffer = ByteBufferUtil.readWithLength(in);
            if (cfs == null || !cfs.metadata.isCounter() || !cfs.isCounterCacheEnabled()) {
                return null;
            }
            assert (cfs.metadata.isCounter());
            final CellName cellName = cfs.metadata.comparator.cellFromByteBuffer(cellNameBuffer);
            return StageManager.getStage(Stage.READ).submit(new Callable<Pair<CounterCacheKey, ClockAndCount>>(){

                @Override
                public Pair<CounterCacheKey, ClockAndCount> call() throws Exception {
                    DecoratedKey key = cfs.partitioner.decorateKey(partitionKey);
                    QueryFilter filter = QueryFilter.getNamesFilter(key, cfs.metadata.cfName, FBUtilities.singleton(cellName, cfs.metadata.comparator), Long.MIN_VALUE);
                    ColumnFamily cf = cfs.getTopLevelColumns(filter, Integer.MIN_VALUE);
                    if (cf == null) {
                        return null;
                    }
                    Cell cell = cf.getColumn(cellName);
                    if (cell == null || !cell.isLive(Long.MIN_VALUE)) {
                        return null;
                    }
                    ClockAndCount clockAndCount = CounterContext.instance().getLocalClockAndCount(cell.value());
                    return Pair.create(CounterCacheKey.create(cfs.metadata.ksAndCFName, partitionKey, cellName), clockAndCount);
                }
            });
        }
    }

    public static enum CacheType {
        KEY_CACHE("KeyCache"),
        ROW_CACHE("RowCache"),
        COUNTER_CACHE("CounterCache");

        private final String name;

        private CacheType(String typeName) {
            this.name = typeName;
        }

        public String toString() {
            return this.name;
        }
    }
}

