/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.github.tjake.ICRC32;
import com.google.common.base.Predicate;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.Ordering;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.Cell;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ColumnSerializer;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.UnknownColumnFamilyException;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.compress.CompressionParameters;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.AbstractDataInput;
import org.apache.cassandra.io.util.ByteBufferDataInput;
import org.apache.cassandra.io.util.FastByteArrayInputStream;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.CRC32Factory;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.WrappedRunnable;
import org.apache.commons.lang3.StringUtils;
import org.cliffc.high_scale_lib.NonBlockingHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogReplayer {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogReplayer.class);
    private static final int MAX_OUTSTANDING_REPLAY_COUNT = Integer.getInteger("cassandra.commitlog_max_outstanding_replay_count", 1024);
    private static final int LEGACY_END_OF_SEGMENT_MARKER = 0;
    private final Set<Keyspace> keyspacesRecovered = new NonBlockingHashSet();
    private final List<Future<?>> futures = new ArrayList();
    private final Map<UUID, AtomicInteger> invalidMutations;
    private final AtomicInteger replayedCount;
    private final Map<UUID, ReplayPosition> cfPositions;
    private final ReplayPosition globalPosition;
    private final ICRC32 checksum;
    private byte[] buffer = new byte[4096];
    private byte[] uncompressedBuffer = new byte[4096];
    private final ReplayFilter replayFilter;

    CommitLogReplayer(ReplayPosition globalPosition, Map<UUID, ReplayPosition> cfPositions, ReplayFilter replayFilter) {
        this.invalidMutations = new HashMap<UUID, AtomicInteger>();
        this.replayedCount = new AtomicInteger();
        this.checksum = CRC32Factory.instance.create();
        this.cfPositions = cfPositions;
        this.globalPosition = globalPosition;
        this.replayFilter = replayFilter;
    }

    public static CommitLogReplayer create() {
        HashMap<UUID, ReplayPosition> cfPositions = new HashMap<UUID, ReplayPosition>();
        Ordering replayPositionOrdering = Ordering.from(ReplayPosition.comparator);
        ReplayFilter replayFilter = ReplayFilter.create();
        for (ColumnFamilyStore cfs : ColumnFamilyStore.all()) {
            ReplayPosition rp = ReplayPosition.getReplayPosition(cfs.getSSTables());
            ReplayPosition truncatedAt = SystemKeyspace.getTruncatedPosition(cfs.metadata.cfId);
            if (truncatedAt != null) {
                long restoreTime = CommitLog.instance.archiver.restorePointInTime;
                long truncatedTime = SystemKeyspace.getTruncatedAt(cfs.metadata.cfId);
                if (truncatedTime > restoreTime) {
                    if (replayFilter.includes(cfs.metadata)) {
                        logger.info("Restore point in time is before latest truncation of table {}.{}. Clearing truncation record.", (Object)cfs.metadata.ksName, (Object)cfs.metadata.cfName);
                        SystemKeyspace.removeTruncationRecord(cfs.metadata.cfId);
                    }
                } else {
                    rp = (ReplayPosition)replayPositionOrdering.max(Arrays.asList(rp, truncatedAt));
                }
            }
            cfPositions.put(cfs.metadata.cfId, rp);
        }
        ReplayPosition globalPosition = (ReplayPosition)replayPositionOrdering.min(cfPositions.values());
        logger.debug("Global replay position is {} from columnfamilies {}", (Object)globalPosition, (Object)FBUtilities.toString(cfPositions));
        return new CommitLogReplayer(globalPosition, cfPositions, replayFilter);
    }

    public void recover(File[] clogs) throws IOException {
        for (File file : clogs) {
            this.recover(file);
        }
    }

    public int blockForWrites() {
        for (Map.Entry<UUID, AtomicInteger> entry : this.invalidMutations.entrySet()) {
            logger.info(String.format("Skipped %d mutations from unknown (probably removed) CF with id %s", entry.getValue().intValue(), entry.getKey()));
        }
        FBUtilities.waitOnFutures(this.futures);
        logger.debug("Finished waiting on mutations from recovery");
        this.futures.clear();
        for (Keyspace keyspace : this.keyspacesRecovered) {
            this.futures.addAll(keyspace.flush());
        }
        FBUtilities.waitOnFutures(this.futures);
        return this.replayedCount.get();
    }

    private int readSyncMarker(CommitLogDescriptor descriptor, int offset, RandomAccessReader reader) throws IOException {
        if ((long)offset > reader.length() - 8L) {
            return -1;
        }
        reader.seek(offset);
        ICRC32 crc = CRC32Factory.instance.create();
        crc.updateInt((int)(descriptor.id & 0xFFFFFFFFL));
        crc.updateInt((int)(descriptor.id >>> 32));
        crc.updateInt((int)reader.getPosition());
        int end = reader.readInt();
        long filecrc = (long)reader.readInt() & 0xFFFFFFFFL;
        if (crc.getValue() != filecrc) {
            if (end != 0 || filecrc != 0L) {
                logger.warn("Encountered bad header at position {} of commit log {}, with invalid CRC. The end of segment marker should be zero.", (Object)offset, (Object)reader.getPath());
            }
            return -1;
        }
        if (end < offset || (long)end > reader.length()) {
            logger.warn("Encountered bad header at position {} of commit log {}, with bad position but valid CRC", (Object)offset, (Object)reader.getPath());
            return -1;
        }
        return end;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void recover(File file) throws IOException {
        CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(file.getName());
        RandomAccessReader reader = RandomAccessReader.open(new File(file.getAbsolutePath()));
        try {
            int end;
            if (desc.version < 4) {
                if (this.logAndCheckIfShouldSkip(file, desc)) {
                    return;
                }
                if (this.globalPosition.segment == desc.id) {
                    reader.seek(this.globalPosition.position);
                }
                this.replaySyncSection(reader, -1, desc);
                return;
            }
            long segmentId = desc.id;
            try {
                desc = CommitLogDescriptor.readHeader(reader);
            }
            catch (IOException e) {
                desc = null;
            }
            if (desc == null) {
                logger.warn("Could not read commit log descriptor in file {}", (Object)file);
                return;
            }
            assert (segmentId == desc.id);
            if (this.logAndCheckIfShouldSkip(file, desc)) {
                return;
            }
            ICompressor compressor = null;
            if (desc.compression != null) {
                try {
                    compressor = CompressionParameters.createCompressor(desc.compression);
                }
                catch (ConfigurationException e) {
                    logger.warn("Unknown compression: {}", (Object)e.getMessage());
                    FileUtils.closeQuietly(reader);
                    logger.info("Finished reading {}", (Object)file);
                    return;
                }
            }
            assert (reader.length() <= Integer.MAX_VALUE);
            int replayEnd = end = (int)reader.getFilePointer();
            while ((end = this.readSyncMarker(desc, end, reader)) >= 0) {
                int replayPos = replayEnd + 8;
                if (logger.isDebugEnabled()) {
                    logger.trace("Replaying {} between {} and {}", new Object[]{file, reader.getFilePointer(), end});
                }
                if (compressor != null) {
                    int uncompressedLength = reader.readInt();
                    replayEnd = replayPos + uncompressedLength;
                } else {
                    replayEnd = end;
                }
                if (segmentId == this.globalPosition.segment && replayEnd < this.globalPosition.position) continue;
                AbstractDataInput sectionReader = reader;
                if (compressor != null) {
                    try {
                        int start = (int)reader.getFilePointer();
                        int compressedLength = end - start;
                        if (logger.isDebugEnabled()) {
                            logger.trace("Decompressing {} between replay positions {} and {}", new Object[]{file, replayPos, replayEnd});
                        }
                        if (compressedLength > this.buffer.length) {
                            this.buffer = new byte[(int)(1.2 * (double)compressedLength)];
                        }
                        reader.readFully(this.buffer, 0, compressedLength);
                        int uncompressedLength = replayEnd - replayPos;
                        if (uncompressedLength > this.uncompressedBuffer.length) {
                            this.uncompressedBuffer = new byte[(int)(1.2 * (double)uncompressedLength)];
                        }
                        compressedLength = compressor.uncompress(this.buffer, 0, compressedLength, this.uncompressedBuffer, 0);
                        sectionReader = new ByteBufferDataInput(ByteBuffer.wrap(this.uncompressedBuffer), reader.getPath(), replayPos, 0);
                    }
                    catch (IOException e) {
                        logger.error("Unexpected exception decompressing section {}", (Throwable)e);
                        continue;
                    }
                }
                if (this.replaySyncSection((FileDataInput)((Object)sectionReader), replayEnd, desc)) continue;
                break;
            }
        }
        finally {
            FileUtils.closeQuietly(reader);
            logger.info("Finished reading {}", (Object)file);
        }
    }

    public boolean logAndCheckIfShouldSkip(File file, CommitLogDescriptor desc) {
        logger.info("Replaying {} (CL version {}, messaging version {}, compression {})", new Object[]{file.getPath(), desc.version, desc.getMessagingVersion(), desc.compression});
        if (this.globalPosition.segment > desc.id) {
            logger.debug("skipping replay of fully-flushed {}", (Object)file);
            return true;
        }
        return false;
    }

    private boolean replaySyncSection(FileDataInput reader, int end, CommitLogDescriptor desc) throws IOException {
        while (reader.getFilePointer() < (long)end && !reader.isEOF()) {
            long claimedCRC32;
            int serializedSize;
            if (logger.isDebugEnabled()) {
                logger.trace("Reading mutation at {}", (Object)reader.getFilePointer());
            }
            try {
                serializedSize = reader.readInt();
                if (serializedSize == 0) {
                    logger.debug("Encountered end of segment marker at {}", (Object)reader.getFilePointer());
                    return false;
                }
                if (serializedSize < 10) {
                    return false;
                }
                long claimedSizeChecksum = desc.version < 4 ? reader.readLong() : (long)reader.readInt() & 0xFFFFFFFFL;
                this.checksum.reset();
                if (desc.version < 3) {
                    this.checksum.update(serializedSize);
                } else {
                    this.checksum.updateInt(serializedSize);
                }
                if (this.checksum.getValue() != claimedSizeChecksum) {
                    return false;
                }
                if (serializedSize > this.buffer.length) {
                    this.buffer = new byte[(int)(1.2 * (double)serializedSize)];
                }
                reader.readFully(this.buffer, 0, serializedSize);
                claimedCRC32 = desc.version < 4 ? reader.readLong() : (long)reader.readInt() & 0xFFFFFFFFL;
            }
            catch (EOFException eof) {
                return false;
            }
            this.checksum.update(this.buffer, 0, serializedSize);
            if (claimedCRC32 != this.checksum.getValue()) continue;
            this.replayMutation(this.buffer, serializedSize, reader.getFilePointer(), desc);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void replayMutation(byte[] inputBuffer, int size, final long entryLocation, final CommitLogDescriptor desc) throws IOException {
        Mutation mutation;
        FastByteArrayInputStream bufIn = new FastByteArrayInputStream(inputBuffer, 0, size);
        try {
            mutation = Mutation.serializer.deserialize(new DataInputStream(bufIn), desc.getMessagingVersion(), ColumnSerializer.Flag.LOCAL);
            for (ColumnFamily cf : mutation.getColumnFamilies()) {
                for (Cell cell : cf) {
                    cf.getComparator().validate(cell.name());
                }
            }
        }
        catch (UnknownColumnFamilyException ex) {
            if (ex.cfId == null) {
                return;
            }
            AtomicInteger i = this.invalidMutations.get(ex.cfId);
            if (i == null) {
                i = new AtomicInteger(1);
                this.invalidMutations.put(ex.cfId, i);
            } else {
                i.incrementAndGet();
            }
            return;
        }
        catch (Throwable t) {
            JVMStabilityInspector.inspectThrowable(t);
            File f = File.createTempFile("mutation", "dat");
            try (DataOutputStream out = new DataOutputStream(new FileOutputStream(f));){
                out.write(inputBuffer, 0, size);
            }
            String st = String.format("Unexpected error deserializing mutation; saved to %s and ignored.  This may be caused by replaying a mutation against a table with the same name but incompatible schema.  Exception follows: ", f.getAbsolutePath());
            logger.error(st, t);
            return;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("replaying mutation for {}.{}: {}", new Object[]{mutation.getKeyspaceName(), ByteBufferUtil.bytesToHex(mutation.key()), "{" + StringUtils.join(mutation.getColumnFamilies().iterator(), (String)", ") + "}"});
        }
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws IOException {
                if (Schema.instance.getKSMetaData(mutation.getKeyspaceName()) == null) {
                    return;
                }
                if (CommitLogReplayer.this.pointInTimeExceeded(mutation)) {
                    return;
                }
                Keyspace keyspace = Keyspace.open(mutation.getKeyspaceName());
                Mutation newMutation = null;
                for (ColumnFamily columnFamily : CommitLogReplayer.this.replayFilter.filter(mutation)) {
                    if (Schema.instance.getCF(columnFamily.id()) == null) continue;
                    ReplayPosition rp = (ReplayPosition)CommitLogReplayer.this.cfPositions.get(columnFamily.id());
                    if (desc.id <= rp.segment && (desc.id != rp.segment || entryLocation <= (long)rp.position)) continue;
                    if (newMutation == null) {
                        newMutation = new Mutation(mutation.getKeyspaceName(), mutation.key());
                    }
                    newMutation.add(columnFamily);
                    CommitLogReplayer.this.replayedCount.incrementAndGet();
                }
                if (newMutation != null) {
                    assert (!newMutation.isEmpty());
                    Keyspace.open(newMutation.getKeyspaceName()).apply(newMutation, false);
                    CommitLogReplayer.this.keyspacesRecovered.add(keyspace);
                }
            }
        };
        this.futures.add(StageManager.getStage(Stage.MUTATION).submit(runnable));
        if (this.futures.size() > MAX_OUTSTANDING_REPLAY_COUNT) {
            FBUtilities.waitOnFutures(this.futures);
            this.futures.clear();
        }
    }

    protected boolean pointInTimeExceeded(Mutation fm) {
        long restoreTarget = CommitLog.instance.archiver.restorePointInTime;
        for (ColumnFamily families : fm.getColumnFamilies()) {
            if (CommitLog.instance.archiver.precision.toMillis(families.maxTimestamp()) <= restoreTarget) continue;
            return true;
        }
        return false;
    }

    private static class CustomReplayFilter
    extends ReplayFilter {
        private Multimap<String, String> toReplay;

        public CustomReplayFilter(Multimap<String, String> toReplay) {
            this.toReplay = toReplay;
        }

        @Override
        public Iterable<ColumnFamily> filter(Mutation mutation) {
            final Collection cfNames = this.toReplay.get((Object)mutation.getKeyspaceName());
            if (cfNames == null) {
                return Collections.emptySet();
            }
            return Iterables.filter(mutation.getColumnFamilies(), (Predicate)new Predicate<ColumnFamily>(){

                public boolean apply(ColumnFamily cf) {
                    return cfNames.contains(cf.metadata().cfName);
                }
            });
        }

        @Override
        public boolean includes(CFMetaData metadata) {
            return this.toReplay.containsEntry((Object)metadata.ksName, (Object)metadata.cfName);
        }
    }

    private static class AlwaysReplayFilter
    extends ReplayFilter {
        private AlwaysReplayFilter() {
        }

        @Override
        public Iterable<ColumnFamily> filter(Mutation mutation) {
            return mutation.getColumnFamilies();
        }

        @Override
        public boolean includes(CFMetaData metadata) {
            return true;
        }
    }

    static abstract class ReplayFilter {
        ReplayFilter() {
        }

        public abstract Iterable<ColumnFamily> filter(Mutation var1);

        public abstract boolean includes(CFMetaData var1);

        public static ReplayFilter create() {
            if (System.getProperty("cassandra.replayList") == null) {
                return new AlwaysReplayFilter();
            }
            HashMultimap toReplay = HashMultimap.create();
            for (String rawPair : System.getProperty("cassandra.replayList").split(",")) {
                String[] pair = rawPair.trim().split("\\.");
                if (pair.length != 2) {
                    throw new IllegalArgumentException("Each table to be replayed must be fully qualified with keyspace name, e.g., 'system.peers'");
                }
                Keyspace ks = Schema.instance.getKeyspaceInstance(pair[0]);
                if (ks == null) {
                    throw new IllegalArgumentException("Unknown keyspace " + pair[0]);
                }
                ColumnFamilyStore cfs = ks.getColumnFamilyStore(pair[1]);
                if (cfs == null) {
                    throw new IllegalArgumentException(String.format("Unknown table %s.%s", pair[0], pair[1]));
                }
                toReplay.put((Object)pair[0], (Object)pair[1]);
            }
            return new CustomReplayFilter((Multimap<String, String>)toReplay);
        }
    }
}

