/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import java.io.DataInput;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.config.UTMetaData;
import org.apache.cassandra.db.DefsTables;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.db.marshal.UserType;
import org.apache.cassandra.exceptions.AlreadyExistsException;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.gms.VersionedValue;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.IMigrationListener;
import org.apache.cassandra.service.MigrationTask;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationManager {
    private static final Logger logger = LoggerFactory.getLogger(MigrationManager.class);
    public static final MigrationManager instance = new MigrationManager();
    private static final RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
    public static final int MIGRATION_DELAY_IN_MS = 60000;
    private final List<IMigrationListener> listeners = new CopyOnWriteArrayList<IMigrationListener>();

    private MigrationManager() {
    }

    public void register(IMigrationListener listener) {
        this.listeners.add(listener);
    }

    public void unregister(IMigrationListener listener) {
        this.listeners.remove(listener);
    }

    public void scheduleSchemaPull(InetAddress endpoint, EndpointState state) {
        VersionedValue value = state.getApplicationState(ApplicationState.SCHEMA);
        if (!endpoint.equals(FBUtilities.getBroadcastAddress()) && value != null) {
            MigrationManager.maybeScheduleSchemaPull(UUID.fromString(value.value), endpoint);
        }
    }

    private static void maybeScheduleSchemaPull(UUID theirVersion, final InetAddress endpoint) {
        if (Schema.instance.getVersion() != null && Schema.instance.getVersion().equals(theirVersion) || !MigrationManager.shouldPullSchemaFrom(endpoint)) {
            logger.debug("Not pulling schema because versions match or shouldPullSchemaFrom returned false");
            return;
        }
        if (Schema.emptyVersion.equals(Schema.instance.getVersion()) || runtimeMXBean.getUptime() < 60000L) {
            logger.debug("Submitting migration task for {}", (Object)endpoint);
            MigrationManager.submitMigrationTask(endpoint);
        } else {
            Runnable runnable = new Runnable(){

                @Override
                public void run() {
                    EndpointState epState = Gossiper.instance.getEndpointStateForEndpoint(endpoint);
                    if (epState == null) {
                        logger.debug("epState vanished for {}, not submitting migration task", (Object)endpoint);
                        return;
                    }
                    VersionedValue value = epState.getApplicationState(ApplicationState.SCHEMA);
                    UUID currentVersion = UUID.fromString(value.value);
                    if (Schema.instance.getVersion().equals(currentVersion)) {
                        logger.debug("not submitting migration task for {} because our versions match", (Object)endpoint);
                        return;
                    }
                    logger.debug("submitting migration task for {}", (Object)endpoint);
                    MigrationManager.submitMigrationTask(endpoint);
                }
            };
            StorageService.optionalTasks.schedule(runnable, 60000L, TimeUnit.MILLISECONDS);
        }
    }

    private static Future<?> submitMigrationTask(InetAddress endpoint) {
        return StageManager.getStage(Stage.MIGRATION).submit(new MigrationTask(endpoint));
    }

    private static boolean shouldPullSchemaFrom(InetAddress endpoint) {
        return MessagingService.instance().knowsVersion(endpoint) && MessagingService.instance().getRawVersion(endpoint) == 8 && !Gossiper.instance.isFatClient(endpoint);
    }

    public static boolean isReadyForBootstrap() {
        return ((ThreadPoolExecutor)((Object)StageManager.getStage(Stage.MIGRATION))).getActiveCount() == 0;
    }

    public void notifyCreateKeyspace(KSMetaData ksm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onCreateKeyspace(ksm.name);
        }
    }

    public void notifyCreateColumnFamily(CFMetaData cfm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onCreateColumnFamily(cfm.ksName, cfm.cfName);
        }
    }

    public void notifyCreateUserType(UserType ut) {
        for (IMigrationListener listener : this.listeners) {
            listener.onCreateUserType(ut.keyspace, ut.getNameAsString());
        }
    }

    public void notifyUpdateKeyspace(KSMetaData ksm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onUpdateKeyspace(ksm.name);
        }
    }

    public void notifyUpdateColumnFamily(CFMetaData cfm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onUpdateColumnFamily(cfm.ksName, cfm.cfName);
        }
    }

    public void notifyUpdateUserType(UserType ut) {
        for (IMigrationListener listener : this.listeners) {
            listener.onUpdateUserType(ut.keyspace, ut.getNameAsString());
        }
    }

    public void notifyDropKeyspace(KSMetaData ksm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onDropKeyspace(ksm.name);
        }
    }

    public void notifyDropColumnFamily(CFMetaData cfm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onDropColumnFamily(cfm.ksName, cfm.cfName);
        }
    }

    public void notifyDropUserType(UserType ut) {
        for (IMigrationListener listener : this.listeners) {
            listener.onDropUserType(ut.keyspace, ut.getNameAsString());
        }
    }

    public static void announceNewKeyspace(KSMetaData ksm) throws ConfigurationException {
        MigrationManager.announceNewKeyspace(ksm, FBUtilities.timestampMicros());
    }

    public static void announceNewKeyspace(KSMetaData ksm, long timestamp) throws ConfigurationException {
        ksm.validate();
        if (Schema.instance.getKSMetaData(ksm.name) != null) {
            throw new AlreadyExistsException(ksm.name);
        }
        logger.info(String.format("Create new Keyspace: %s", ksm));
        MigrationManager.announce(ksm.toSchema(timestamp));
    }

    public static void announceNewColumnFamily(CFMetaData cfm) throws ConfigurationException {
        cfm.validate();
        KSMetaData ksm = Schema.instance.getKSMetaData(cfm.ksName);
        if (ksm == null) {
            throw new ConfigurationException(String.format("Cannot add column family '%s' to non existing keyspace '%s'.", cfm.cfName, cfm.ksName));
        }
        if (ksm.cfMetaData().containsKey(cfm.cfName)) {
            throw new AlreadyExistsException(cfm.ksName, cfm.cfName);
        }
        logger.info(String.format("Create new ColumnFamily: %s", cfm));
        MigrationManager.announce(MigrationManager.addSerializedKeyspace(cfm.toSchema(FBUtilities.timestampMicros()), cfm.ksName));
    }

    public static void announceNewType(UserType newType) {
        MigrationManager.announce(UTMetaData.toSchema(newType, FBUtilities.timestampMicros()));
    }

    public static void announceKeyspaceUpdate(KSMetaData ksm) throws ConfigurationException {
        ksm.validate();
        KSMetaData oldKsm = Schema.instance.getKSMetaData(ksm.name);
        if (oldKsm == null) {
            throw new ConfigurationException(String.format("Cannot update non existing keyspace '%s'.", ksm.name));
        }
        logger.info(String.format("Update Keyspace '%s' From %s To %s", ksm.name, oldKsm, ksm));
        MigrationManager.announce(oldKsm.toSchemaUpdate(ksm, FBUtilities.timestampMicros()));
    }

    public static void announceColumnFamilyUpdate(CFMetaData cfm, boolean fromThrift) throws ConfigurationException {
        cfm.validate();
        CFMetaData oldCfm = Schema.instance.getCFMetaData(cfm.ksName, cfm.cfName);
        if (oldCfm == null) {
            throw new ConfigurationException(String.format("Cannot update non existing column family '%s' in keyspace '%s'.", cfm.cfName, cfm.ksName));
        }
        oldCfm.validateCompatility(cfm);
        logger.info(String.format("Update ColumnFamily '%s/%s' From %s To %s", cfm.ksName, cfm.cfName, oldCfm, cfm));
        MigrationManager.announce(MigrationManager.addSerializedKeyspace(oldCfm.toSchemaUpdate(cfm, FBUtilities.timestampMicros(), fromThrift), cfm.ksName));
    }

    public static void announceTypeUpdate(UserType updatedType) {
        MigrationManager.announceNewType(updatedType);
    }

    public static void announceKeyspaceDrop(String ksName) throws ConfigurationException {
        KSMetaData oldKsm = Schema.instance.getKSMetaData(ksName);
        if (oldKsm == null) {
            throw new ConfigurationException(String.format("Cannot drop non existing keyspace '%s'.", ksName));
        }
        logger.info(String.format("Drop Keyspace '%s'", oldKsm.name));
        MigrationManager.announce(oldKsm.dropFromSchema(FBUtilities.timestampMicros()));
    }

    public static void announceColumnFamilyDrop(String ksName, String cfName) throws ConfigurationException {
        CFMetaData oldCfm = Schema.instance.getCFMetaData(ksName, cfName);
        if (oldCfm == null) {
            throw new ConfigurationException(String.format("Cannot drop non existing column family '%s' in keyspace '%s'.", cfName, ksName));
        }
        logger.info(String.format("Drop ColumnFamily '%s/%s'", oldCfm.ksName, oldCfm.cfName));
        MigrationManager.announce(MigrationManager.addSerializedKeyspace(oldCfm.dropFromSchema(FBUtilities.timestampMicros()), ksName));
    }

    private static Mutation addSerializedKeyspace(Mutation migration, String ksName) {
        migration.add(SystemKeyspace.readSchemaRow((String)ksName).cf);
        return migration;
    }

    public static void announceTypeDrop(UserType droppedType) {
        MigrationManager.announce(UTMetaData.dropFromSchema(droppedType, FBUtilities.timestampMicros()));
    }

    private static void announce(Mutation schema) {
        FBUtilities.waitOnFuture(MigrationManager.announce(Collections.singletonList(schema)));
    }

    private static void pushSchemaMutation(InetAddress endpoint, Collection<Mutation> schema) {
        MessageOut<Collection<Mutation>> msg = new MessageOut<Collection<Mutation>>(MessagingService.Verb.DEFINITIONS_UPDATE, schema, MigrationsSerializer.instance);
        MessagingService.instance().sendOneWay(msg, endpoint);
    }

    private static Future<?> announce(final Collection<Mutation> schema) {
        Future<?> f = StageManager.getStage(Stage.MIGRATION).submit(new WrappedRunnable(){

            @Override
            protected void runMayThrow() throws IOException, ConfigurationException {
                DefsTables.mergeSchema(schema);
            }
        });
        for (InetAddress endpoint : Gossiper.instance.getLiveMembers()) {
            if (endpoint.equals(FBUtilities.getBroadcastAddress()) || !MessagingService.instance().knowsVersion(endpoint) || MessagingService.instance().getRawVersion(endpoint) != 8) continue;
            MigrationManager.pushSchemaMutation(endpoint, schema);
        }
        return f;
    }

    public static void passiveAnnounce(UUID version) {
        Gossiper.instance.addLocalApplicationState(ApplicationState.SCHEMA, StorageService.instance.valueFactory.schema(version));
        logger.debug("Gossiping my schema version {}", (Object)version);
    }

    public static void resetLocalSchema() throws IOException {
        logger.info("Starting local schema reset...");
        logger.debug("Truncating schema tables...");
        for (String cf : SystemKeyspace.allSchemaCfs) {
            SystemKeyspace.schemaCFS(cf).truncateBlocking();
        }
        logger.debug("Clearing local schema keyspace definitions...");
        Schema.instance.clear();
        Set<InetAddress> liveEndpoints = Gossiper.instance.getLiveMembers();
        liveEndpoints.remove(FBUtilities.getBroadcastAddress());
        for (InetAddress node : liveEndpoints) {
            if (!MigrationManager.shouldPullSchemaFrom(node)) continue;
            logger.debug("Requesting schema from {}", (Object)node);
            FBUtilities.waitOnFuture(MigrationManager.submitMigrationTask(node));
            break;
        }
        logger.info("Local schema reset is complete.");
    }

    public static class MigrationsSerializer
    implements IVersionedSerializer<Collection<Mutation>> {
        public static MigrationsSerializer instance = new MigrationsSerializer();

        @Override
        public void serialize(Collection<Mutation> schema, DataOutputPlus out, int version) throws IOException {
            out.writeInt(schema.size());
            for (Mutation mutation : schema) {
                Mutation.serializer.serialize(mutation, out, version);
            }
        }

        @Override
        public Collection<Mutation> deserialize(DataInput in, int version) throws IOException {
            int count = in.readInt();
            ArrayList<Mutation> schema = new ArrayList<Mutation>(count);
            for (int i = 0; i < count; ++i) {
                schema.add(Mutation.serializer.deserialize(in, version));
            }
            return schema;
        }

        @Override
        public long serializedSize(Collection<Mutation> schema, int version) {
            int size = TypeSizes.NATIVE.sizeof(schema.size());
            for (Mutation mutation : schema) {
                size = (int)((long)size + Mutation.serializer.serializedSize(mutation, version));
            }
            return size;
        }
    }
}

