/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.collect.ImmutableMap;
import java.io.File;
import java.io.FileFilter;
import java.io.IOError;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.BlacklistedDirectories;
import org.apache.cassandra.io.FSError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Directories {
    private static final Logger logger = LoggerFactory.getLogger(Directories.class);
    public static final String BACKUPS_SUBDIR = "backups";
    public static final String SNAPSHOT_SUBDIR = "snapshots";
    public static final String SECONDARY_INDEX_NAME_SEPARATOR = ".";
    public static final DataDirectory[] dataFileLocations;
    private final String keyspacename;
    private final String cfname;
    private final File[] sstableDirectories;

    public static boolean hasFullPermissions(File dir, String dataDir) {
        if (!dir.isDirectory()) {
            logger.error("Not a directory {}", (Object)dataDir);
            return false;
        }
        if (!FileAction.hasPrivilege(dir, FileAction.X)) {
            logger.error("Doesn't have execute permissions for {} directory", (Object)dataDir);
            return false;
        }
        if (!FileAction.hasPrivilege(dir, FileAction.R)) {
            logger.error("Doesn't have read permissions for {} directory", (Object)dataDir);
            return false;
        }
        if (dir.exists() && !FileAction.hasPrivilege(dir, FileAction.W)) {
            logger.error("Doesn't have write permissions for {} directory", (Object)dataDir);
            return false;
        }
        return true;
    }

    public static Directories create(String keyspacename, String cfname) {
        int idx = cfname.indexOf(SECONDARY_INDEX_NAME_SEPARATOR);
        if (idx > 0) {
            return new Directories(keyspacename, cfname, cfname.substring(0, idx));
        }
        return new Directories(keyspacename, cfname, cfname);
    }

    private Directories(String keyspacename, String cfname, String directoryName) {
        this.keyspacename = keyspacename;
        this.cfname = cfname;
        this.sstableDirectories = new File[dataFileLocations.length];
        for (int i = 0; i < dataFileLocations.length; ++i) {
            this.sstableDirectories[i] = new File(Directories.dataFileLocations[i].location, Directories.join(keyspacename, directoryName));
        }
        if (!StorageService.instance.isClientMode()) {
            for (File dir : this.sstableDirectories) {
                try {
                    FileUtils.createDirectory(dir);
                }
                catch (FSError e) {
                    logger.error("Failed to create {} directory", (Object)dir);
                    FileUtils.handleFSError(e);
                }
            }
        }
    }

    public File getLocationForDisk(DataDirectory dataDirectory) {
        if (dataDirectory != null) {
            for (File dir : this.sstableDirectories) {
                if (!dir.getAbsolutePath().startsWith(dataDirectory.location.getAbsolutePath())) continue;
                return dir;
            }
        }
        return null;
    }

    public Descriptor find(String filename) {
        for (File dir : this.sstableDirectories) {
            if (!new File(dir, filename).exists()) continue;
            return (Descriptor)Descriptor.fromFilename((File)dir, (String)filename).left;
        }
        return null;
    }

    public File getDirectoryForNewSSTables() {
        return this.getWriteableLocationAsFile(-1L);
    }

    public File getWriteableLocationAsFile(long writeSize) {
        return this.getLocationForDisk(this.getWriteableLocation(writeSize));
    }

    public DataDirectory getWriteableLocation(long writeSize) {
        ArrayList<DataDirectoryCandidate> candidates = new ArrayList<DataDirectoryCandidate>();
        long totalAvailable = 0L;
        boolean tooBig = false;
        for (DataDirectory dataDir : dataFileLocations) {
            if (BlacklistedDirectories.isUnwritable(this.getLocationForDisk(dataDir))) {
                logger.debug("removing blacklisted candidate {}", (Object)dataDir.location);
                continue;
            }
            DataDirectoryCandidate candidate = new DataDirectoryCandidate(dataDir);
            if (candidate.availableSpace < writeSize) {
                logger.debug("removing candidate {}, usable={}, requested={}", new Object[]{candidate.dataDirectory.location, candidate.availableSpace, writeSize});
                tooBig = true;
                continue;
            }
            candidates.add(candidate);
            totalAvailable += candidate.availableSpace;
        }
        if (candidates.isEmpty()) {
            if (tooBig) {
                return null;
            }
            throw new IOError(new IOException("All configured data directories have been blacklisted as unwritable for erroring out"));
        }
        if (candidates.size() == 1) {
            return ((DataDirectoryCandidate)candidates.get((int)0)).dataDirectory;
        }
        Directories.sortWriteableCandidates(candidates, totalAvailable);
        return Directories.pickWriteableDirectory(candidates);
    }

    static DataDirectory pickWriteableDirectory(List<DataDirectoryCandidate> candidates) {
        double rnd = ThreadLocalRandom.current().nextDouble();
        for (DataDirectoryCandidate candidate : candidates) {
            if (!((rnd -= candidate.perc) <= 0.0)) continue;
            return candidate.dataDirectory;
        }
        return candidates.get((int)0).dataDirectory;
    }

    static void sortWriteableCandidates(List<DataDirectoryCandidate> candidates, long totalAvailable) {
        for (DataDirectoryCandidate candidate : candidates) {
            candidate.calcFreePerc(totalAvailable);
        }
        Collections.sort(candidates);
    }

    public boolean hasAvailableDiskSpace(long estimatedSSTables, long expectedTotalWriteSize) {
        long writeSize = expectedTotalWriteSize / estimatedSSTables;
        long totalAvailable = 0L;
        for (DataDirectory dataDir : dataFileLocations) {
            if (BlacklistedDirectories.isUnwritable(this.getLocationForDisk(dataDir))) continue;
            DataDirectoryCandidate candidate = new DataDirectoryCandidate(dataDir);
            if (candidate.availableSpace < writeSize) continue;
            totalAvailable += candidate.availableSpace;
        }
        return totalAvailable > expectedTotalWriteSize;
    }

    public static File getSnapshotDirectory(Descriptor desc, String snapshotName) {
        return Directories.getOrCreate(desc.directory, SNAPSHOT_SUBDIR, snapshotName);
    }

    public static File getBackupsDirectory(Descriptor desc) {
        return Directories.getOrCreate(desc.directory, BACKUPS_SUBDIR);
    }

    public SSTableLister sstableLister() {
        return new SSTableLister();
    }

    @Deprecated
    public File tryGetLeveledManifest() {
        for (File dir : this.sstableDirectories) {
            File manifestFile = new File(dir, this.cfname + ".json");
            if (!manifestFile.exists()) continue;
            logger.debug("Found manifest at {}", (Object)manifestFile);
            return manifestFile;
        }
        logger.debug("No level manifest found");
        return null;
    }

    @Deprecated
    public void snapshotLeveledManifest(String snapshotName) {
        File manifest = this.tryGetLeveledManifest();
        if (manifest != null) {
            File snapshotDirectory = Directories.getOrCreate(manifest.getParentFile(), SNAPSHOT_SUBDIR, snapshotName);
            File target = new File(snapshotDirectory, manifest.getName());
            FileUtils.createHardLink(manifest, target);
        }
    }

    public boolean snapshotExists(String snapshotName) {
        for (File dir : this.sstableDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return true;
        }
        return false;
    }

    public static void clearSnapshot(String snapshotName, List<File> snapshotDirectories) {
        String tag = snapshotName == null ? "" : snapshotName;
        for (File dir : snapshotDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, tag));
            if (!snapshotDir.exists()) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Removing snapshot directory " + snapshotDir);
            }
            FileUtils.deleteRecursive(snapshotDir);
        }
    }

    public long snapshotCreationTime(String snapshotName) {
        for (File dir : this.sstableDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return snapshotDir.lastModified();
        }
        throw new RuntimeException("Snapshot " + snapshotName + " doesn't exist");
    }

    public static List<File> getKSChildDirectories(String ksName) {
        ArrayList<File> result = new ArrayList<File>();
        for (DataDirectory dataDirectory : dataFileLocations) {
            File ksDir = new File(dataDirectory.location, ksName);
            File[] cfDirs = ksDir.listFiles();
            if (cfDirs == null) continue;
            for (File cfDir : cfDirs) {
                if (!cfDir.isDirectory()) continue;
                result.add(cfDir);
            }
        }
        return result;
    }

    public List<File> getCFDirectories() {
        ArrayList<File> result = new ArrayList<File>();
        for (File dataDirectory : this.sstableDirectories) {
            if (!dataDirectory.isDirectory()) continue;
            result.add(dataDirectory);
        }
        return result;
    }

    private static File getOrCreate(File base, String ... subdirs) {
        File dir;
        File file = dir = subdirs == null || subdirs.length == 0 ? base : new File(base, Directories.join(subdirs));
        if (dir.exists()) {
            if (!dir.isDirectory()) {
                throw new AssertionError((Object)String.format("Invalid directory path %s: path exists but is not a directory", dir));
            }
        } else if (!(dir.mkdirs() || dir.exists() && dir.isDirectory())) {
            throw new FSWriteError((Throwable)new IOException("Unable to create directory " + dir), dir);
        }
        return dir;
    }

    private static String join(String ... s) {
        return StringUtils.join((Object[])s, (String)File.separator);
    }

    static void overrideDataDirectoriesForTest(String loc) {
        for (int i = 0; i < dataFileLocations.length; ++i) {
            Directories.dataFileLocations[i] = new DataDirectory(new File(loc));
        }
    }

    static void resetDataDirectoriesAfterTest() {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataFileLocations[i] = new DataDirectory(new File(locations[i]));
        }
    }

    static {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        dataFileLocations = new DataDirectory[locations.length];
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataFileLocations[i] = new DataDirectory(new File(locations[i]));
        }
    }

    public class SSTableLister {
        private boolean skipTemporary;
        private boolean includeBackups;
        private boolean onlyBackups;
        private int nbFiles;
        private final Map<Descriptor, Set<Component>> components = new HashMap<Descriptor, Set<Component>>();
        private boolean filtered;
        private String snapshotName;

        public SSTableLister skipTemporary(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.skipTemporary = b;
            return this;
        }

        public SSTableLister includeBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.includeBackups = b;
            return this;
        }

        public SSTableLister onlyBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.onlyBackups = b;
            this.includeBackups = b;
            return this;
        }

        public SSTableLister snapshots(String sn) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.snapshotName = sn;
            return this;
        }

        public Map<Descriptor, Set<Component>> list() {
            this.filter();
            return ImmutableMap.copyOf(this.components);
        }

        public List<File> listFiles() {
            this.filter();
            ArrayList<File> l = new ArrayList<File>(this.nbFiles);
            for (Map.Entry<Descriptor, Set<Component>> entry : this.components.entrySet()) {
                for (Component c : entry.getValue()) {
                    l.add(new File(entry.getKey().filenameFor(c)));
                }
            }
            return l;
        }

        private void filter() {
            if (this.filtered) {
                return;
            }
            for (File location : Directories.this.sstableDirectories) {
                if (BlacklistedDirectories.isUnreadable(location)) continue;
                if (this.snapshotName != null) {
                    new File(location, Directories.join(new String[]{Directories.SNAPSHOT_SUBDIR, this.snapshotName})).listFiles(this.getFilter());
                    continue;
                }
                if (!this.onlyBackups) {
                    location.listFiles(this.getFilter());
                }
                if (!this.includeBackups) continue;
                new File(location, Directories.BACKUPS_SUBDIR).listFiles(this.getFilter());
            }
            this.filtered = true;
        }

        private FileFilter getFilter() {
            final String sstablePrefix = Directories.this.keyspacename + '-' + Directories.this.cfname + '-';
            return new FileFilter(){

                @Override
                public boolean accept(File file) {
                    if (file.isDirectory() || !file.getName().startsWith(sstablePrefix)) {
                        return false;
                    }
                    Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParentFile(), file.getName());
                    if (pair == null) {
                        return false;
                    }
                    if (SSTableLister.this.skipTemporary && ((Descriptor)pair.left).temporary) {
                        return false;
                    }
                    HashSet previous = (HashSet)SSTableLister.this.components.get(pair.left);
                    if (previous == null) {
                        previous = new HashSet();
                        SSTableLister.this.components.put(pair.left, previous);
                    }
                    previous.add(pair.right);
                    SSTableLister.this.nbFiles++;
                    return false;
                }
            };
        }
    }

    static final class DataDirectoryCandidate
    implements Comparable<DataDirectoryCandidate> {
        final DataDirectory dataDirectory;
        final long availableSpace;
        double perc;

        public DataDirectoryCandidate(DataDirectory dataDirectory) {
            this.dataDirectory = dataDirectory;
            this.availableSpace = dataDirectory.getAvailableSpace();
        }

        void calcFreePerc(long totalAvailableSpace) {
            double w = this.availableSpace;
            this.perc = w /= (double)totalAvailableSpace;
        }

        @Override
        public int compareTo(DataDirectoryCandidate o) {
            if (this == o) {
                return 0;
            }
            int r = Double.compare(this.perc, o.perc);
            if (r != 0) {
                return -r;
            }
            return System.identityHashCode(this) - System.identityHashCode(o);
        }
    }

    public static class DataDirectory {
        public final File location;

        public DataDirectory(File location) {
            this.location = location;
        }

        public long getAvailableSpace() {
            return this.location.getUsableSpace();
        }
    }

    public static enum FileAction {
        X,
        W,
        XW,
        R,
        XR,
        RW,
        XRW;


        public static boolean hasPrivilege(File file, FileAction action) {
            boolean privilege = false;
            switch (action) {
                case X: {
                    privilege = file.canExecute();
                    break;
                }
                case W: {
                    privilege = file.canWrite();
                    break;
                }
                case XW: {
                    privilege = file.canExecute() && file.canWrite();
                    break;
                }
                case R: {
                    privilege = file.canRead();
                    break;
                }
                case XR: {
                    privilege = file.canExecute() && file.canRead();
                    break;
                }
                case RW: {
                    privilege = file.canRead() && file.canWrite();
                    break;
                }
                case XRW: {
                    privilege = file.canExecute() && file.canRead() && file.canWrite();
                }
            }
            return privilege;
        }
    }
}

