/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.BatchCommitLogExecutorService;
import org.apache.cassandra.db.commitlog.CommitLogAllocator;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogMBean;
import org.apache.cassandra.db.commitlog.CommitLogReplayer;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.ICommitLogExecutorService;
import org.apache.cassandra.db.commitlog.PeriodicCommitLogExecutorService;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.metrics.CommitLogMetrics;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLog
implements CommitLogMBean {
    private static final Logger logger = LoggerFactory.getLogger(CommitLog.class);
    public static final CommitLog instance = new CommitLog();
    private final ICommitLogExecutorService executor;
    public final CommitLogAllocator allocator;
    public final CommitLogArchiver archiver = new CommitLogArchiver();
    public static final int END_OF_SEGMENT_MARKER = 0;
    public static final int END_OF_SEGMENT_MARKER_SIZE = 4;
    public volatile CommitLogSegment activeSegment;
    private final CommitLogMetrics metrics;

    private CommitLog() {
        DatabaseDescriptor.createAllDirectories();
        this.allocator = new CommitLogAllocator();
        this.activateNextSegment();
        this.executor = DatabaseDescriptor.getCommitLogSync() == Config.CommitLogSync.batch ? new BatchCommitLogExecutorService() : new PeriodicCommitLogExecutorService(this);
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName("org.apache.cassandra.db:type=Commitlog"));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.metrics = new CommitLogMetrics(this.executor, this.allocator);
    }

    public void resetUnsafe() {
        this.allocator.resetUnsafe();
        this.activateNextSegment();
    }

    public int recover() throws IOException {
        FilenameFilter unmanagedFilesFilter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return CommitLogDescriptor.isValid(name) && !CommitLog.instance.allocator.manages(name);
            }
        };
        for (File file : new File(DatabaseDescriptor.getCommitLogLocation()).listFiles(unmanagedFilesFilter)) {
            this.archiver.maybeArchive(file.getPath(), file.getName());
            this.archiver.maybeWaitForArchiving(file.getName());
        }
        assert (this.archiver.archivePending.isEmpty()) : "Not all commit log archive tasks were completed before restore";
        this.archiver.maybeRestoreArchive();
        Object[] files = new File(DatabaseDescriptor.getCommitLogLocation()).listFiles(unmanagedFilesFilter);
        int replayed = 0;
        if (files.length == 0) {
            logger.info("No commitlog files found; skipping replay");
        } else {
            Arrays.sort(files, new CommitLogSegment.CommitLogSegmentFileComparator());
            logger.info("Replaying " + StringUtils.join((Object[])files, (String)", "));
            replayed = this.recover((File[])files);
            logger.info("Log replay complete, " + replayed + " replayed mutations");
            for (Object f : files) {
                CommitLog.instance.allocator.recycleSegment((File)f);
            }
        }
        this.allocator.enableReserveSegmentCreation();
        return replayed;
    }

    public int recover(File ... clogs) throws IOException {
        CommitLogReplayer recovery = new CommitLogReplayer();
        recovery.recover(clogs);
        return recovery.blockForWrites();
    }

    @Override
    public void recover(String path) throws IOException {
        this.recover(new File(path));
    }

    public Future<ReplayPosition> getContext() {
        Callable<ReplayPosition> task = new Callable<ReplayPosition>(){

            @Override
            public ReplayPosition call() {
                return CommitLog.this.activeSegment.getContext();
            }
        };
        return this.executor.submit(task);
    }

    public int activeSegments() {
        return this.allocator.getActiveSegments().size();
    }

    public void add(RowMutation rm) {
        this.executor.add(new LogRecordAdder(rm));
    }

    public void discardCompletedSegments(final UUID cfId, final ReplayPosition context) {
        Callable task = new Callable(){

            public Object call() {
                logger.debug("discard completed log segments for {}, column family {}", (Object)context, (Object)cfId);
                Iterator<CommitLogSegment> iter = CommitLog.this.allocator.getActiveSegments().iterator();
                while (iter.hasNext()) {
                    CommitLogSegment segment = iter.next();
                    segment.markClean(cfId, context);
                    if (iter.hasNext()) {
                        if (segment.isUnused()) {
                            logger.debug("Commit log segment {} is unused", (Object)segment);
                            CommitLog.this.allocator.recycleSegment(segment);
                        } else {
                            logger.debug("Not safe to delete commit log segment {}; dirty is {}", (Object)segment, (Object)segment.dirtyString());
                        }
                    } else {
                        logger.debug("Not deleting active commitlog segment {}", (Object)segment);
                    }
                    if (!segment.contains(context)) continue;
                    break;
                }
                return null;
            }
        };
        FBUtilities.waitOnFuture(this.executor.submit(task));
    }

    public void sync() {
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            segment.sync();
        }
    }

    @Override
    public long getCompletedTasks() {
        return (Long)this.metrics.completedTasks.value();
    }

    @Override
    public long getPendingTasks() {
        return (Long)this.metrics.pendingTasks.value();
    }

    @Override
    public long getTotalCommitlogSize() {
        return (Long)this.metrics.totalCommitLogSize.value();
    }

    @Override
    public String getArchiveCommand() {
        return this.archiver.archiveCommand;
    }

    @Override
    public String getRestoreCommand() {
        return this.archiver.restoreCommand;
    }

    @Override
    public String getRestoreDirectories() {
        return this.archiver.restoreDirectories;
    }

    @Override
    public long getRestorePointInTime() {
        return this.archiver.restorePointInTime;
    }

    @Override
    public String getRestorePrecision() {
        return this.archiver.precision.toString();
    }

    private void activateNextSegment() {
        this.activeSegment = this.allocator.fetchSegment();
        logger.debug("Active segment is now {}", (Object)this.activeSegment);
    }

    @Override
    public List<String> getActiveSegmentNames() {
        ArrayList<String> segmentNames = new ArrayList<String>();
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            segmentNames.add(segment.getName());
        }
        return segmentNames;
    }

    @Override
    public List<String> getArchivingSegmentNames() {
        return new ArrayList<String>(this.archiver.archivePending.keySet());
    }

    public void shutdownBlocking() throws InterruptedException {
        this.executor.shutdown();
        this.executor.awaitTermination();
        this.allocator.shutdown();
        this.allocator.awaitTermination();
    }

    @VisibleForTesting
    public static boolean handleCommitError(String message, Throwable t) {
        switch (DatabaseDescriptor.getCommitFailurePolicy()) {
            case stop: {
                StorageService.instance.stopTransports();
            }
            case stop_commit: {
                logger.error(String.format("%s. Commit disk failure policy is %s; terminating thread", new Object[]{message, DatabaseDescriptor.getCommitFailurePolicy()}), t);
                return false;
            }
            case ignore: {
                logger.error(message, t);
                Uninterruptibles.sleepUninterruptibly((long)1L, (TimeUnit)TimeUnit.SECONDS);
                return true;
            }
        }
        throw new AssertionError((Object)DatabaseDescriptor.getCommitFailurePolicy());
    }

    class LogRecordAdder
    implements Callable,
    Runnable {
        final RowMutation rowMutation;

        LogRecordAdder(RowMutation rm) {
            this.rowMutation = rm;
        }

        @Override
        public void run() {
            long totalSize = RowMutation.serializer.serializedSize(this.rowMutation, 7) + 20L;
            if (totalSize > (long)DatabaseDescriptor.getCommitLogSegmentSize()) {
                logger.warn("Skipping commitlog append of extremely large mutation ({} bytes)", (Object)totalSize);
                return;
            }
            if (!CommitLog.this.activeSegment.hasCapacityFor(totalSize)) {
                CommitLogSegment oldSegment = CommitLog.this.activeSegment;
                CommitLog.this.activateNextSegment();
                CommitLog.this.archiver.maybeArchive(oldSegment.getPath(), oldSegment.getName());
            }
            try {
                CommitLog.this.activeSegment.write(this.rowMutation);
            }
            catch (IOException e) {
                throw new FSWriteError((Throwable)e, CommitLog.this.activeSegment.getPath());
            }
        }

        public Object call() {
            this.run();
            return null;
        }
    }
}

