/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.base.Function;
import com.google.common.base.Throwables;
import java.io.File;
import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.concurrent.JMXEnabledThreadPoolExecutor;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.AtomicSortedColumns;
import org.apache.cassandra.db.Column;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.db.index.SecondaryIndexManager;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.io.sstable.SSTableMetadata;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.sstable.SSTableWriter;
import org.apache.cassandra.io.util.DiskAwareRunnable;
import org.apache.cassandra.utils.Allocator;
import org.cliffc.high_scale_lib.NonBlockingHashSet;
import org.github.jamm.MemoryMeter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Memtable {
    private static final Logger logger = LoggerFactory.getLogger(Memtable.class);
    private static final ExecutorService flushWriter = new JMXEnabledThreadPoolExecutor(DatabaseDescriptor.getFlushWriters(), 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(DatabaseDescriptor.getFlushQueueSize()), new NamedThreadFactory("FlushWriter"), "internal");
    private static final double MIN_SANE_LIVE_RATIO = 1.0;
    private static final double MAX_SANE_LIVE_RATIO = 64.0;
    private static final Set<ColumnFamilyStore> meteringInProgress = new NonBlockingHashSet();
    private static final ExecutorService meterExecutor = new JMXEnabledThreadPoolExecutor(1, Integer.MAX_VALUE, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(), new NamedThreadFactory("MemoryMeter"), "internal");
    private final MemoryMeter meter;
    static volatile ColumnFamilyStore activelyMeasuring;
    private final AtomicLong currentSize = new AtomicLong(0L);
    private final AtomicLong currentOperations = new AtomicLong(0L);
    private final ConcurrentNavigableMap<RowPosition, AtomicSortedColumns> rows = new ConcurrentSkipListMap<RowPosition, AtomicSortedColumns>();
    public final ColumnFamilyStore cfs;
    private final long creationTime = System.currentTimeMillis();
    private final long creationNano = System.nanoTime();
    private final Allocator allocator = DatabaseDescriptor.getMemtableAllocator();
    private final Function<Column, Column> localCopyFunction = new Function<Column, Column>(){

        public Column apply(Column c) {
            return c.localCopy(Memtable.this.cfs, Memtable.this.allocator);
        }
    };
    public final AbstractType initialComparator;

    public Memtable(ColumnFamilyStore cfs) {
        this.cfs = cfs;
        this.initialComparator = cfs.metadata.comparator;
        this.cfs.scheduleFlush();
        Callable<Set<Object>> provider = new Callable<Set<Object>>(){

            @Override
            public Set<Object> call() throws Exception {
                Set<Object> set = Collections.newSetFromMap(new IdentityHashMap());
                set.add(Memtable.this.cfs.metadata);
                return set;
            }
        };
        this.meter = new MemoryMeter().omitSharedBufferOverhead().withTrackerProvider((Callable)provider);
    }

    public long getLiveSize() {
        long estimatedSize = (long)((double)this.currentSize.get() * this.cfs.liveRatio);
        if (estimatedSize < this.allocator.getMinimumSize()) {
            return this.allocator.getMinimumSize();
        }
        if (estimatedSize > this.allocator.getMaximumSize()) {
            return this.allocator.getMaximumSize();
        }
        return estimatedSize;
    }

    public long getOperations() {
        return this.currentOperations.get();
    }

    void put(DecoratedKey key, ColumnFamily columnFamily, SecondaryIndexManager.Updater indexer) {
        this.resolve(key, columnFamily, indexer);
    }

    public void updateLiveRatio() throws RuntimeException {
        if (!MemoryMeter.isInitialized()) {
            logger.warn("MemoryMeter uninitialized (jamm not specified as java agent); assuming liveRatio of {}.   Usually this means cassandra-env.sh disabled jamm because you are using a buggy JRE;  upgrade to the Sun JRE instead", (Object)this.cfs.liveRatio);
            return;
        }
        if (!meteringInProgress.add(this.cfs)) {
            logger.debug("Metering already pending or active for {}; skipping liveRatio update", (Object)this.cfs);
            return;
        }
        meterExecutor.submit(new MeteringRunnable(this.cfs));
    }

    private void resolve(DecoratedKey key, ColumnFamily cf, SecondaryIndexManager.Updater indexer) {
        AtomicSortedColumns previous = (AtomicSortedColumns)this.rows.get(key);
        if (previous == null) {
            AtomicSortedColumns empty = cf.cloneMeShallow(AtomicSortedColumns.factory, false);
            previous = this.rows.putIfAbsent(new DecoratedKey(key.token, this.allocator.clone(key.key)), empty);
            if (previous == null) {
                previous = empty;
            }
        }
        long sizeDelta = previous.addAllWithSizeDelta(cf, this.allocator, this.localCopyFunction, indexer);
        this.currentSize.addAndGet(sizeDelta);
        this.currentOperations.addAndGet(cf.getColumnCount() == 0 ? (cf.isMarkedForDelete() ? 1L : 0L) : (long)cf.getColumnCount());
    }

    public String contents() {
        StringBuilder builder = new StringBuilder();
        builder.append("{");
        for (Map.Entry entry : this.rows.entrySet()) {
            builder.append(entry.getKey()).append(": ").append(entry.getValue()).append(", ");
        }
        builder.append("}");
        return builder.toString();
    }

    public void flushAndSignal(CountDownLatch latch, Future<ReplayPosition> context) {
        flushWriter.execute(new FlushRunnable(latch, context));
    }

    public String toString() {
        return String.format("Memtable-%s@%s(%s/%s serialized/live bytes, %s ops)", this.cfs.name, this.hashCode(), this.currentSize, this.getLiveSize(), this.currentOperations);
    }

    public Iterator<Map.Entry<DecoratedKey, AtomicSortedColumns>> getEntryIterator(final RowPosition startWith, final RowPosition stopAt) {
        return new Iterator<Map.Entry<DecoratedKey, AtomicSortedColumns>>(){
            private Iterator<Map.Entry<RowPosition, AtomicSortedColumns>> iter;
            {
                this.iter = stopAt.isMinimum(Memtable.this.cfs.partitioner) ? Memtable.this.rows.tailMap(startWith).entrySet().iterator() : Memtable.this.rows.subMap(startWith, true, stopAt, true).entrySet().iterator();
            }

            @Override
            public boolean hasNext() {
                return this.iter.hasNext();
            }

            @Override
            public Map.Entry<DecoratedKey, AtomicSortedColumns> next() {
                Map.Entry<RowPosition, AtomicSortedColumns> entry = this.iter.next();
                assert (entry.getKey() instanceof DecoratedKey);
                return entry;
            }

            @Override
            public void remove() {
                this.iter.remove();
            }
        };
    }

    public boolean isClean() {
        return this.rows.isEmpty();
    }

    public boolean isExpired() {
        int period = this.cfs.metadata.getMemtableFlushPeriod();
        return period > 0 && System.nanoTime() - this.creationNano >= TimeUnit.MILLISECONDS.toNanos(period);
    }

    public ColumnFamily getColumnFamily(DecoratedKey key) {
        return (ColumnFamily)this.rows.get(key);
    }

    public long creationTime() {
        return this.creationTime;
    }

    private static class MeteringRunnable
    implements Runnable {
        private final ColumnFamilyStore cfs;

        public MeteringRunnable(ColumnFamilyStore cfs) {
            this.cfs = cfs;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                activelyMeasuring = this.cfs;
                Memtable memtable = this.cfs.getMemtableThreadSafe();
                long start = System.nanoTime();
                long deepSize = memtable.meter.measure((Object)memtable.rows);
                int objects = 0;
                for (Map.Entry entry : memtable.rows.entrySet()) {
                    deepSize += memtable.meter.measureDeep(entry.getKey()) + memtable.meter.measureDeep(entry.getValue());
                    objects += ((AtomicSortedColumns)entry.getValue()).getColumnCount();
                }
                double newRatio = (double)deepSize / (double)memtable.currentSize.get();
                if (newRatio < 1.0) {
                    logger.warn("setting live ratio to minimum of {} instead of {}", (Object)1.0, (Object)newRatio);
                    newRatio = 1.0;
                }
                if (newRatio > 64.0) {
                    logger.warn("setting live ratio to maximum of {} instead of {}", (Object)64.0, (Object)newRatio);
                    newRatio = 64.0;
                }
                this.cfs.liveRatio = newRatio > this.cfs.liveRatio ? newRatio : (this.cfs.liveRatio + newRatio) / 2.0;
                logger.info("{} liveRatio is {} (just-counted was {}).  calculation took {}ms for {} columns", new Object[]{this.cfs, this.cfs.liveRatio, newRatio, TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - start), objects});
            }
            finally {
                activelyMeasuring = null;
                meteringInProgress.remove(this.cfs);
            }
        }
    }

    class FlushRunnable
    extends DiskAwareRunnable {
        private final CountDownLatch latch;
        private final Future<ReplayPosition> context;
        private final long estimatedSize;

        FlushRunnable(CountDownLatch latch, Future<ReplayPosition> context) {
            this.latch = latch;
            this.context = context;
            long keySize = 0L;
            for (RowPosition key : Memtable.this.rows.keySet()) {
                assert (key instanceof DecoratedKey);
                keySize += (long)((DecoratedKey)key).key.remaining();
            }
            this.estimatedSize = (long)((double)(keySize + keySize + Memtable.this.currentSize.get()) * 1.2);
        }

        @Override
        public long getExpectedWriteSize() {
            return this.estimatedSize;
        }

        @Override
        protected void runWith(File sstableDirectory) throws Exception {
            assert (sstableDirectory != null) : "Flush task is not bound to any disk";
            SSTableReader sstable = this.writeSortedContents(this.context, sstableDirectory);
            Memtable.this.cfs.replaceFlushed(Memtable.this, sstable);
            this.latch.countDown();
        }

        @Override
        protected Directories getDirectories() {
            return Memtable.this.cfs.directories;
        }

        private SSTableReader writeSortedContents(Future<ReplayPosition> context, File sstableDirectory) throws ExecutionException, InterruptedException {
            logger.info("Writing " + Memtable.this.toString());
            SSTableWriter writer = this.createFlushWriter(Memtable.this.cfs.getTempSSTablePath(sstableDirectory));
            try {
                SSTableReader ssTable;
                for (Map.Entry entry : Memtable.this.rows.entrySet()) {
                    ColumnFamily cf = (ColumnFamily)entry.getValue();
                    if (cf.isMarkedForDelete()) {
                        if (Memtable.this.cfs.name.equals("batchlog") && Memtable.this.cfs.keyspace.getName().equals("system") && cf.getColumnCount() != 0) continue;
                        ColumnFamilyStore.removeDeletedColumnsOnly(cf, Integer.MIN_VALUE);
                    }
                    writer.append((DecoratedKey)entry.getKey(), cf);
                }
                if (writer.getFilePointer() > 0L) {
                    ssTable = writer.closeAndOpenReader();
                    logger.info(String.format("Completed flushing %s (%d bytes) for commitlog position %s", ssTable.getFilename(), new File(ssTable.getFilename()).length(), context.get()));
                } else {
                    writer.abort();
                    ssTable = null;
                    logger.info("Completed flushing; nothing needed to be retained.  Commitlog position was {}", (Object)context.get());
                }
                return ssTable;
            }
            catch (Throwable e) {
                writer.abort();
                throw Throwables.propagate((Throwable)e);
            }
        }

        public SSTableWriter createFlushWriter(String filename) throws ExecutionException, InterruptedException {
            SSTableMetadata.Collector sstableMetadataCollector = SSTableMetadata.createCollector(Memtable.this.cfs.metadata.comparator).replayPosition(this.context.get());
            return new SSTableWriter(filename, Memtable.this.rows.size(), Memtable.this.cfs.metadata, Memtable.this.cfs.partitioner, sstableMetadataCollector);
        }
    }
}

