/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.primitives.Doubles;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.columniterator.OnDiskAtomIterator;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.ICompactionScanner;
import org.apache.cassandra.db.compaction.LeveledCompactionTask;
import org.apache.cassandra.db.compaction.LeveledManifest;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.notifications.INotification;
import org.apache.cassandra.notifications.INotificationConsumer;
import org.apache.cassandra.notifications.SSTableAddedNotification;
import org.apache.cassandra.notifications.SSTableListChangedNotification;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledCompactionStrategy
extends AbstractCompactionStrategy
implements INotificationConsumer {
    private static final Logger logger = LoggerFactory.getLogger(LeveledCompactionStrategy.class);
    private static final String SSTABLE_SIZE_OPTION = "sstable_size_in_mb";
    @VisibleForTesting
    final LeveledManifest manifest;
    private final int maxSSTableSizeInMB;

    public LeveledCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        int configuredMaxSSTableSize = 5;
        if (options != null) {
            String value = options.containsKey(SSTABLE_SIZE_OPTION) ? options.get(SSTABLE_SIZE_OPTION) : "5";
            configuredMaxSSTableSize = Integer.parseInt(value);
        }
        this.maxSSTableSizeInMB = configuredMaxSSTableSize;
        cfs.getDataTracker().subscribe(this);
        logger.debug("{} subscribed to the data tracker.", (Object)this);
        this.manifest = LeveledManifest.create(cfs, this.maxSSTableSizeInMB);
        logger.debug("Created {}", (Object)this.manifest);
    }

    @Override
    public void shutdown() {
        super.shutdown();
        this.cfs.getDataTracker().unsubscribe(this);
    }

    public int getLevelSize(int i) {
        return this.manifest.getLevelSize(i);
    }

    public int[] getAllLevelSize() {
        return this.manifest.getAllLevelSize();
    }

    @Override
    public AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        if (this.cfs.isCompactionDisabled()) {
            return null;
        }
        return this.getMaximalTask(gcBefore);
    }

    @Override
    public AbstractCompactionTask getMaximalTask(int gcBefore) {
        OperationType op;
        Collection<SSTableReader> sstables;
        do {
            sstables = this.manifest.getCompactionCandidates();
            op = OperationType.COMPACTION;
            if (!sstables.isEmpty()) continue;
            SSTableReader sstable = this.findDroppableSSTable(gcBefore);
            if (sstable == null) {
                logger.debug("No compaction necessary for {}", (Object)this);
                return null;
            }
            sstables = Collections.singleton(sstable);
            op = OperationType.TOMBSTONE_COMPACTION;
        } while (!this.cfs.getDataTracker().markCompacting(sstables));
        LeveledCompactionTask newTask = new LeveledCompactionTask(this.cfs, sstables, gcBefore, this.maxSSTableSizeInMB);
        newTask.setCompactionType(op);
        return newTask;
    }

    @Override
    public AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        throw new UnsupportedOperationException("LevelDB compaction strategy does not allow user-specified compactions");
    }

    @Override
    public int getEstimatedRemainingTasks() {
        return this.manifest.getEstimatedTasks();
    }

    @Override
    public void handleNotification(INotification notification, Object sender) {
        if (notification instanceof SSTableAddedNotification) {
            SSTableAddedNotification flushedNotification = (SSTableAddedNotification)notification;
            this.manifest.add(flushedNotification.added);
        } else if (notification instanceof SSTableListChangedNotification) {
            SSTableListChangedNotification listChangedNotification = (SSTableListChangedNotification)notification;
            switch (listChangedNotification.compactionType) {
                case CLEANUP: 
                case SCRUB: 
                case UPGRADE_SSTABLES: 
                case TOMBSTONE_COMPACTION: {
                    this.manifest.replace(listChangedNotification.removed, listChangedNotification.added);
                    break;
                }
                default: {
                    this.manifest.promote(listChangedNotification.removed, listChangedNotification.added);
                }
            }
        }
    }

    @Override
    public long getMaxSSTableSize() {
        return (long)this.maxSSTableSizeInMB * 1024L * 1024L;
    }

    @Override
    public List<ICompactionScanner> getScanners(Collection<SSTableReader> sstables, Range<Token> range) {
        ArrayListMultimap byLevel = ArrayListMultimap.create();
        for (SSTableReader sstable : sstables) {
            byLevel.get((Object)this.manifest.levelOf(sstable)).add(sstable);
        }
        ArrayList<ICompactionScanner> scanners = new ArrayList<ICompactionScanner>(sstables.size());
        for (Integer level : byLevel.keySet()) {
            if (level <= 0) {
                for (SSTableReader sstable : byLevel.get((Object)level)) {
                    scanners.add(sstable.getDirectScanner(range, CompactionManager.instance.getRateLimiter()));
                }
                continue;
            }
            scanners.add(new LeveledScanner(byLevel.get((Object)level), range));
        }
        return scanners;
    }

    public String toString() {
        return String.format("LCS@%d(%s)", this.hashCode(), this.cfs.columnFamily);
    }

    private SSTableReader findDroppableSSTable(final int gcBefore) {
        for (int i = this.manifest.getLevelCount(); i >= 0; --i) {
            SSTableReader sstable;
            SortedSet<SSTableReader> sstables = this.manifest.getLevelSorted(i, new Comparator<SSTableReader>(){

                @Override
                public int compare(SSTableReader o1, SSTableReader o2) {
                    double r1 = o1.getEstimatedDroppableTombstoneRatio(gcBefore);
                    double r2 = o2.getEstimatedDroppableTombstoneRatio(gcBefore);
                    return -1 * Doubles.compare((double)r1, (double)r2);
                }
            });
            if (sstables.isEmpty()) continue;
            Set<SSTableReader> compacting = this.cfs.getDataTracker().getCompacting();
            Iterator i$ = sstables.iterator();
            while (i$.hasNext() && !((sstable = (SSTableReader)i$.next()).getEstimatedDroppableTombstoneRatio(gcBefore) <= (double)this.tombstoneThreshold)) {
                if (compacting.contains(sstable) || sstable.isMarkedSuspect() || !this.worthDroppingTombstones(sstable, gcBefore)) continue;
                return sstable;
            }
        }
        return null;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        String size = options.containsKey(SSTABLE_SIZE_OPTION) ? options.get(SSTABLE_SIZE_OPTION) : "1";
        try {
            int ssSize = Integer.parseInt(size);
            if (ssSize < 1) {
                throw new ConfigurationException(String.format("%s must be larger than 0, but was %s", SSTABLE_SIZE_OPTION, ssSize));
            }
        }
        catch (NumberFormatException ex) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", size, SSTABLE_SIZE_OPTION), ex);
        }
        uncheckedOptions.remove(SSTABLE_SIZE_OPTION);
        return uncheckedOptions;
    }

    private static class LeveledScanner
    extends AbstractIterator<OnDiskAtomIterator>
    implements ICompactionScanner {
        private final Range<Token> range;
        private final List<SSTableReader> sstables;
        private final Iterator<SSTableReader> sstableIterator;
        private final long totalLength;
        private ICompactionScanner currentScanner;
        private long positionOffset;

        public LeveledScanner(Collection<SSTableReader> sstables, Range<Token> range) {
            this.range = range;
            this.sstables = new ArrayList<SSTableReader>(sstables);
            Collections.sort(this.sstables, SSTable.sstableComparator);
            this.sstableIterator = this.sstables.iterator();
            this.currentScanner = this.sstableIterator.next().getDirectScanner(range, CompactionManager.instance.getRateLimiter());
            long length = 0L;
            for (SSTableReader sstable : sstables) {
                length += sstable.uncompressedLength();
            }
            this.totalLength = length;
        }

        protected OnDiskAtomIterator computeNext() {
            try {
                while (true) {
                    if (this.currentScanner.hasNext()) {
                        return (OnDiskAtomIterator)this.currentScanner.next();
                    }
                    this.positionOffset += this.currentScanner.getLengthInBytes();
                    this.currentScanner.close();
                    if (!this.sstableIterator.hasNext()) {
                        this.currentScanner = null;
                        return (OnDiskAtomIterator)this.endOfData();
                    }
                    this.currentScanner = this.sstableIterator.next().getDirectScanner(this.range, CompactionManager.instance.getRateLimiter());
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public void close() throws IOException {
            if (this.currentScanner != null) {
                this.currentScanner.close();
            }
        }

        @Override
        public long getLengthInBytes() {
            return this.totalLength;
        }

        @Override
        public long getCurrentPosition() {
            return this.positionOffset + (this.currentScanner == null ? 0L : this.currentScanner.getCurrentPosition());
        }

        @Override
        public String getBackingFiles() {
            return Joiner.on((String)", ").join(this.sstables);
        }
    }
}

