/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.collect.ImmutableSortedSet;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.List;
import java.util.SortedSet;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.BatchlogManagerMBean;
import org.apache.cassandra.db.Column;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.IColumn;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.filter.IFilter;
import org.apache.cassandra.db.filter.NamesQueryFilter;
import org.apache.cassandra.db.filter.QueryFilter;
import org.apache.cassandra.db.filter.QueryPath;
import org.apache.cassandra.db.marshal.LongType;
import org.apache.cassandra.db.marshal.UTF8Type;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.util.FastByteArrayOutputStream;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BatchlogManager
implements BatchlogManagerMBean {
    private static final String MBEAN_NAME = "org.apache.cassandra.db:type=BatchlogManager";
    private static final int VERSION = 5;
    private static final long TIMEOUT = 2L * DatabaseDescriptor.getWriteRpcTimeout();
    private static final ByteBuffer WRITTEN_AT = BatchlogManager.columnName("written_at");
    private static final ByteBuffer DATA = BatchlogManager.columnName("data");
    private static final Logger logger = LoggerFactory.getLogger(BatchlogManager.class);
    public static final BatchlogManager instance = new BatchlogManager();
    private final AtomicLong totalBatchesReplayed = new AtomicLong();
    private final AtomicBoolean isReplaying = new AtomicBoolean();

    public void start() {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        StorageService.optionalTasks.scheduleWithFixedDelay(runnable, StorageService.RING_DELAY, 600000L, TimeUnit.MILLISECONDS);
    }

    @Override
    public int countAllBatches() {
        int count = 0;
        for (Row row : BatchlogManager.getRangeSlice(new NamesQueryFilter((SortedSet<ByteBuffer>)ImmutableSortedSet.of()))) {
            if (row.cf == null || row.cf.isMarkedForDelete()) continue;
            ++count;
        }
        return count;
    }

    @Override
    public long getTotalBatchesReplayed() {
        return this.totalBatchesReplayed.longValue();
    }

    @Override
    public void forceBatchlogReplay() {
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        StorageService.optionalTasks.execute(runnable);
    }

    public static RowMutation getBatchlogMutationFor(Collection<RowMutation> mutations, UUID uuid) {
        long timestamp = FBUtilities.timestampMicros();
        ByteBuffer writtenAt = LongType.instance.decompose(timestamp / 1000L);
        ByteBuffer data = BatchlogManager.serializeRowMutations(mutations);
        ColumnFamily cf = ColumnFamily.create(CFMetaData.BatchlogCF);
        cf.addColumn(new Column(WRITTEN_AT, writtenAt, timestamp));
        cf.addColumn(new Column(DATA, data, timestamp));
        RowMutation rm = new RowMutation("system", UUIDType.instance.decompose(uuid));
        rm.add(cf);
        return rm;
    }

    private static ByteBuffer serializeRowMutations(Collection<RowMutation> mutations) {
        FastByteArrayOutputStream bos = new FastByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);
        try {
            dos.writeInt(mutations.size());
            for (RowMutation rm : mutations) {
                RowMutation.serializer.serialize(rm, (DataOutput)dos, 5);
            }
        }
        catch (IOException e) {
            throw new AssertionError();
        }
        return ByteBuffer.wrap(bos.toByteArray());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void replayAllFailedBatches() {
        if (!this.isReplaying.compareAndSet(false, true)) {
            return;
        }
        try {
            logger.debug("Started replayAllFailedBatches");
            for (Row row : BatchlogManager.getRangeSlice(new NamesQueryFilter(WRITTEN_AT))) {
                IColumn writtenAt;
                if (row.cf == null || row.cf.isMarkedForDelete() || (writtenAt = row.cf.getColumn(WRITTEN_AT)) != null && System.currentTimeMillis() <= LongType.instance.compose(writtenAt.value()) + TIMEOUT) continue;
                this.replayBatch(row.key);
            }
        }
        finally {
            this.isReplaying.set(false);
        }
        logger.debug("Finished replayAllFailedBatches");
    }

    private void replayBatch(DecoratedKey key) {
        UUID uuid = UUIDType.instance.compose(key.key);
        logger.debug("Replaying batch {}", (Object)uuid);
        ColumnFamilyStore store = Table.open("system").getColumnFamilyStore("batchlog");
        QueryFilter filter = QueryFilter.getNamesFilter(key, new QueryPath("batchlog"), DATA);
        ColumnFamily batch = store.getColumnFamily(filter);
        if (batch == null || batch.isMarkedForDelete()) {
            return;
        }
        IColumn dataColumn = batch.getColumn(DATA);
        try {
            if (dataColumn != null) {
                BatchlogManager.writeHintsForSerializedMutations(dataColumn.value());
            }
        }
        catch (IOException e) {
            logger.warn("Skipped batch replay of {} due to {}", (Object)uuid, (Object)e);
        }
        BatchlogManager.deleteBatch(key);
        this.totalBatchesReplayed.incrementAndGet();
    }

    private static void writeHintsForSerializedMutations(ByteBuffer data) throws IOException {
        DataInputStream in = new DataInputStream(ByteBufferUtil.inputStream(data));
        int size = in.readInt();
        for (int i = 0; i < size; ++i) {
            BatchlogManager.writeHintsForMutation(RowMutation.serializer.deserialize(in, 5));
        }
    }

    private static void writeHintsForMutation(RowMutation mutation) throws IOException {
        for (InetAddress target : StorageProxy.getWriteEndpoints(mutation.getTable(), mutation.key())) {
            if (target.equals(FBUtilities.getBroadcastAddress())) {
                mutation.apply();
                continue;
            }
            StorageProxy.writeHintForMutation(mutation, target);
        }
    }

    private static void deleteBatch(DecoratedKey key) {
        RowMutation rm = new RowMutation("system", key.key);
        rm.delete(new QueryPath("batchlog"), FBUtilities.timestampMicros());
        rm.apply();
    }

    private static ByteBuffer columnName(String name) {
        ByteBuffer raw = UTF8Type.instance.decompose(name);
        return CFMetaData.BatchlogCF.getCfDef().getColumnNameBuilder().add(raw).build();
    }

    private static List<Row> getRangeSlice(IFilter columnFilter) {
        ColumnFamilyStore store = Table.open("system").getColumnFamilyStore("batchlog");
        IPartitioner partitioner = StorageService.getPartitioner();
        Token.KeyBound minPosition = ((Token)partitioner.getMinimumToken()).minKeyBound();
        Range<RowPosition> range = new Range<RowPosition>(minPosition, minPosition, partitioner);
        return store.getRangeSlice(null, range, Integer.MAX_VALUE, columnFilter, null);
    }
}

