/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ArrayBackedSortedColumns;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnIndexer;
import org.apache.cassandra.db.CounterColumn;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.IColumn;
import org.apache.cassandra.db.SuperColumn;
import org.apache.cassandra.db.compaction.AbstractCompactedRow;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.io.IColumnSerializer;
import org.apache.cassandra.io.compress.CompressedSequentialWriter;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.IndexSummary;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableMetadata;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.FileMark;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.SegmentedFile;
import org.apache.cassandra.io.util.SequentialWriter;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.BloomFilter;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSTableWriter
extends SSTable {
    private static Logger logger = LoggerFactory.getLogger(SSTableWriter.class);
    private IndexWriter iwriter;
    private SegmentedFile.Builder dbuilder;
    private final SequentialWriter dataFile;
    private DecoratedKey<?> lastWrittenKey;
    private FileMark dataMark;
    private SSTableMetadata.Collector sstableMetadataCollector;

    public SSTableWriter(String filename, long keyCount) throws IOException {
        this(filename, keyCount, Schema.instance.getCFMetaData(Descriptor.fromFilename(filename)), StorageService.getPartitioner(), SSTableMetadata.createCollector());
    }

    private static Set<Component> components(CFMetaData metadata) {
        HashSet<Component> components = new HashSet<Component>(Arrays.asList(Component.DATA, Component.FILTER, Component.PRIMARY_INDEX, Component.STATS));
        if (metadata.compressionParameters().sstableCompressor != null) {
            components.add(Component.COMPRESSION_INFO);
        } else {
            components.add(Component.DIGEST);
        }
        return components;
    }

    public SSTableWriter(String filename, long keyCount, CFMetaData metadata, IPartitioner<?> partitioner, SSTableMetadata.Collector sstableMetadataCollector) throws IOException {
        super(Descriptor.fromFilename(filename), SSTableWriter.components(metadata), metadata, partitioner);
        this.iwriter = new IndexWriter(keyCount);
        if (this.compression) {
            this.dbuilder = SegmentedFile.getCompressedBuilder();
            this.dataFile = CompressedSequentialWriter.open(this.getFilename(), this.descriptor.filenameFor(Component.COMPRESSION_INFO), !DatabaseDescriptor.populateIOCacheOnFlush(), metadata.compressionParameters(), sstableMetadataCollector);
        } else {
            this.dbuilder = SegmentedFile.getBuilder(DatabaseDescriptor.getDiskAccessMode());
            this.dataFile = SequentialWriter.open(new File(this.getFilename()), !DatabaseDescriptor.populateIOCacheOnFlush());
            this.dataFile.setComputeDigest();
        }
        this.sstableMetadataCollector = sstableMetadataCollector;
    }

    public void mark() {
        this.dataMark = this.dataFile.mark();
        this.iwriter.mark();
    }

    public void resetAndTruncate() {
        try {
            this.dataFile.resetAndTruncate(this.dataMark);
            this.iwriter.resetAndTruncate();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    private long beforeAppend(DecoratedKey<?> decoratedKey) throws IOException {
        assert (decoratedKey != null) : "Keys must not be null";
        if (this.lastWrittenKey != null && this.lastWrittenKey.compareTo(decoratedKey) >= 0) {
            throw new RuntimeException("Last written key " + this.lastWrittenKey + " >= current key " + decoratedKey + " writing into " + this.getFilename());
        }
        return this.lastWrittenKey == null ? 0L : this.dataFile.getFilePointer();
    }

    private void afterAppend(DecoratedKey<?> decoratedKey, long dataPosition) throws IOException {
        this.last = this.lastWrittenKey = decoratedKey;
        if (null == this.first) {
            this.first = this.lastWrittenKey;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("wrote " + decoratedKey + " at " + dataPosition);
        }
        this.iwriter.afterAppend(decoratedKey, dataPosition);
        this.dbuilder.addPotentialBoundary(dataPosition);
    }

    public long append(AbstractCompactedRow row) throws IOException {
        long currentPosition = this.beforeAppend(row.key);
        ByteBufferUtil.writeWithShortLength(row.key.key, this.dataFile.stream);
        long dataStart = this.dataFile.getFilePointer();
        long dataSize = row.write(this.dataFile.stream);
        assert (dataSize == this.dataFile.getFilePointer() - (dataStart + 8L)) : "incorrect row data size " + dataSize + " written to " + this.dataFile.getPath() + "; correct is " + (this.dataFile.getFilePointer() - (dataStart + 8L));
        this.sstableMetadataCollector.updateMaxTimestamp(row.maxTimestamp());
        this.sstableMetadataCollector.addRowSize(this.dataFile.getFilePointer() - currentPosition);
        this.sstableMetadataCollector.addColumnCount(row.columnCount());
        this.afterAppend(row.key, currentPosition);
        return currentPosition;
    }

    public void append(DecoratedKey<?> decoratedKey, ColumnFamily cf) throws IOException {
        long startPosition = this.beforeAppend(decoratedKey);
        ByteBufferUtil.writeWithShortLength(decoratedKey.key, this.dataFile.stream);
        ColumnIndexer.RowHeader header = ColumnIndexer.serialize(cf);
        this.dataFile.stream.writeLong(header.serializedSize() + cf.serializedSizeForSSTable());
        int columnCount = ColumnFamily.serializer().serializeWithIndexes(cf, header, this.dataFile.stream);
        this.afterAppend(decoratedKey, startPosition);
        this.sstableMetadataCollector.updateMaxTimestamp(cf.maxTimestamp());
        this.sstableMetadataCollector.addRowSize(this.dataFile.getFilePointer() - startPosition);
        this.sstableMetadataCollector.addColumnCount(columnCount);
    }

    public void append(DecoratedKey<?> decoratedKey, ByteBuffer value) throws IOException {
        long currentPosition = this.beforeAppend(decoratedKey);
        ByteBufferUtil.writeWithShortLength(decoratedKey.key, this.dataFile.stream);
        assert (value.remaining() > 0);
        this.dataFile.stream.writeLong(value.remaining());
        ByteBufferUtil.write(value, this.dataFile.stream);
        this.afterAppend(decoratedKey, currentPosition);
    }

    public long appendFromStream(DecoratedKey<?> key, CFMetaData metadata, long dataSize, DataInput in) throws IOException {
        long currentPosition = this.beforeAppend(key);
        ByteBufferUtil.writeWithShortLength(key.key, this.dataFile.stream);
        long dataStart = this.dataFile.getFilePointer();
        this.dataFile.stream.writeLong(dataSize);
        int bfSize = in.readInt();
        this.dataFile.stream.writeInt(bfSize);
        for (int i = 0; i < bfSize; ++i) {
            this.dataFile.stream.writeByte(in.readByte());
        }
        int indexSize = in.readInt();
        this.dataFile.stream.writeInt(indexSize);
        for (int i = 0; i < indexSize; ++i) {
            this.dataFile.stream.writeByte(in.readByte());
        }
        this.dataFile.stream.writeInt(in.readInt());
        this.dataFile.stream.writeLong(in.readLong());
        int columnCount = in.readInt();
        this.dataFile.stream.writeInt(columnCount);
        long maxTimestamp = Long.MIN_VALUE;
        ColumnFamily cf = ColumnFamily.create(metadata, ArrayBackedSortedColumns.factory());
        for (int i = 0; i < columnCount; ++i) {
            IColumn column = cf.getColumnSerializer().deserialize(in, IColumnSerializer.Flag.PRESERVE_SIZE, Integer.MIN_VALUE);
            if (column instanceof CounterColumn) {
                column = ((CounterColumn)column).markDeltaToBeCleared();
            } else if (column instanceof SuperColumn) {
                SuperColumn sc = (SuperColumn)column;
                for (IColumn subColumn : sc.getSubColumns()) {
                    if (!(subColumn instanceof CounterColumn)) continue;
                    IColumn marked = ((CounterColumn)subColumn).markDeltaToBeCleared();
                    sc.replace(subColumn, marked);
                }
            }
            maxTimestamp = Math.max(maxTimestamp, column.maxTimestamp());
            cf.getColumnSerializer().serialize(column, this.dataFile.stream);
        }
        assert (dataSize == this.dataFile.getFilePointer() - (dataStart + 8L)) : "incorrect row data size " + dataSize + " written to " + this.dataFile.getPath() + "; correct is " + (this.dataFile.getFilePointer() - (dataStart + 8L));
        this.sstableMetadataCollector.updateMaxTimestamp(maxTimestamp);
        this.sstableMetadataCollector.addRowSize(this.dataFile.getFilePointer() - currentPosition);
        this.sstableMetadataCollector.addColumnCount(columnCount);
        this.afterAppend(key, currentPosition);
        return currentPosition;
    }

    public void updateMaxTimestamp(long timestamp) {
        this.sstableMetadataCollector.updateMaxTimestamp(timestamp);
    }

    public void abort() {
        assert (this.descriptor.temporary);
        FileUtils.closeQuietly(this.iwriter);
        FileUtils.closeQuietly(this.dataFile);
        try {
            Set<Component> components = SSTable.componentsFor(this.descriptor);
            if (!components.isEmpty()) {
                SSTable.delete(this.descriptor, components);
            }
        }
        catch (Exception e) {
            logger.error(String.format("Failed deleting temp components for %s", this.descriptor), (Throwable)e);
        }
    }

    public SSTableReader closeAndOpenReader() throws IOException {
        return this.closeAndOpenReader(System.currentTimeMillis());
    }

    public SSTableReader closeAndOpenReader(long maxDataAge) throws IOException {
        this.iwriter.close();
        this.dataFile.close();
        SSTableMetadata sstableMetadata = this.sstableMetadataCollector.finalizeMetadata(this.partitioner.getClass().getCanonicalName());
        SSTableWriter.writeMetadata(this.descriptor, sstableMetadata);
        this.maybeWriteDigest();
        Descriptor newdesc = SSTableWriter.rename(this.descriptor, this.components);
        SegmentedFile ifile = this.iwriter.builder.complete(newdesc.filenameFor(SSTable.COMPONENT_INDEX));
        SegmentedFile dfile = this.dbuilder.complete(newdesc.filenameFor(SSTable.COMPONENT_DATA));
        SSTableReader sstable = SSTableReader.internalOpen(newdesc, this.components, this.metadata, this.partitioner, ifile, dfile, this.iwriter.summary, this.iwriter.bf, maxDataAge, sstableMetadata);
        sstable.first = SSTableWriter.getMinimalKey(this.first);
        sstable.last = SSTableWriter.getMinimalKey(this.last);
        this.iwriter = null;
        this.dbuilder = null;
        return sstable;
    }

    private void maybeWriteDigest() throws IOException {
        byte[] digest = this.dataFile.digest();
        if (digest == null) {
            return;
        }
        SequentialWriter out = SequentialWriter.open(new File(this.descriptor.filenameFor(SSTable.COMPONENT_DIGEST)), true);
        Descriptor newdesc = this.descriptor.asTemporary(false);
        String[] tmp = newdesc.filenameFor(SSTable.COMPONENT_DATA).split(Pattern.quote(File.separator));
        String dataFileName = tmp[tmp.length - 1];
        out.write(String.format("%s  %s", Hex.bytesToHex(digest), dataFileName).getBytes());
        out.close();
    }

    private static void writeMetadata(Descriptor desc, SSTableMetadata sstableMetadata) throws IOException {
        SequentialWriter out = SequentialWriter.open(new File(desc.filenameFor(SSTable.COMPONENT_STATS)), true);
        SSTableMetadata.serializer.serialize(sstableMetadata, out.stream);
        out.close();
    }

    static Descriptor rename(Descriptor tmpdesc, Set<Component> components) {
        Descriptor newdesc = tmpdesc.asTemporary(false);
        SSTableWriter.rename(tmpdesc, newdesc, components);
        return newdesc;
    }

    public static void rename(Descriptor tmpdesc, Descriptor newdesc, Set<Component> components) {
        try {
            for (Component component : Sets.difference(components, Collections.singleton(Component.DATA))) {
                FBUtilities.renameWithConfirm(tmpdesc.filenameFor(component), newdesc.filenameFor(component));
            }
            FBUtilities.renameWithConfirm(tmpdesc.filenameFor(Component.DATA), newdesc.filenameFor(Component.DATA));
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public long getFilePointer() {
        return this.dataFile.getFilePointer();
    }

    public long getOnDiskFilePointer() throws IOException {
        return this.dataFile.getOnDiskFilePointer();
    }

    class IndexWriter
    implements Closeable {
        private final SequentialWriter indexFile;
        public final SegmentedFile.Builder builder;
        public final IndexSummary summary;
        public final BloomFilter bf;
        private FileMark mark;

        IndexWriter(long keyCount) throws IOException {
            this.indexFile = SequentialWriter.open(new File(SSTableWriter.this.descriptor.filenameFor(SSTable.COMPONENT_INDEX)), !DatabaseDescriptor.populateIOCacheOnFlush());
            this.builder = SegmentedFile.getBuilder(DatabaseDescriptor.getIndexAccessMode());
            this.summary = new IndexSummary(keyCount);
            Double fpChance = SSTableWriter.this.metadata.getBloomFilterFpChance();
            if (fpChance != null && fpChance == 0.0) {
                logger.error("Bloom filter FP chance of zero isn't supposed to happen");
                fpChance = null;
            }
            this.bf = fpChance == null ? BloomFilter.getFilter(keyCount, 15) : BloomFilter.getFilter(keyCount, fpChance);
        }

        public void afterAppend(DecoratedKey<?> key, long dataPosition) throws IOException {
            this.bf.add(key.key);
            long indexPosition = this.indexFile.getFilePointer();
            ByteBufferUtil.writeWithShortLength(key.key, this.indexFile.stream);
            this.indexFile.stream.writeLong(dataPosition);
            if (logger.isTraceEnabled()) {
                logger.trace("wrote index of " + key + " at " + indexPosition);
            }
            this.summary.maybeAddEntry(key, indexPosition);
            this.builder.addPotentialBoundary(indexPosition);
        }

        @Override
        public void close() throws IOException {
            FileOutputStream fos = new FileOutputStream(SSTableWriter.this.descriptor.filenameFor(SSTable.COMPONENT_FILTER));
            DataOutputStream stream = new DataOutputStream(fos);
            BloomFilter.serializer().serialize(this.bf, (DataOutput)stream);
            stream.flush();
            fos.getFD().sync();
            stream.close();
            long position = this.indexFile.getFilePointer();
            this.indexFile.close();
            FileUtils.truncate(this.indexFile.getPath(), position);
            this.summary.complete();
        }

        public void mark() {
            this.mark = this.indexFile.mark();
        }

        public void resetAndTruncate() throws IOException {
            this.indexFile.resetAndTruncate(this.mark);
        }

        public String toString() {
            return "IndexWriter(" + SSTableWriter.this.descriptor + ")";
        }
    }
}

