/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.index;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.config.ConfigurationException;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.IColumn;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.SystemTable;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.filter.IFilter;
import org.apache.cassandra.db.index.PerColumnSecondaryIndex;
import org.apache.cassandra.db.index.PerRowSecondaryIndex;
import org.apache.cassandra.db.index.SecondaryIndex;
import org.apache.cassandra.db.index.SecondaryIndexBuilder;
import org.apache.cassandra.db.index.SecondaryIndexSearcher;
import org.apache.cassandra.dht.AbstractBounds;
import org.apache.cassandra.dht.LocalToken;
import org.apache.cassandra.io.sstable.ReducingKeyIterator;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.IndexClause;
import org.apache.cassandra.thrift.IndexExpression;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecondaryIndexManager {
    private static final Logger logger = LoggerFactory.getLogger(SecondaryIndexManager.class);
    private final ConcurrentNavigableMap<ByteBuffer, SecondaryIndex> indexesByColumn = new ConcurrentSkipListMap<ByteBuffer, SecondaryIndex>();
    private final Map<Class<? extends SecondaryIndex>, SecondaryIndex> rowLevelIndexMap = new HashMap<Class<? extends SecondaryIndex>, SecondaryIndex>();
    public final ColumnFamilyStore baseCfs;

    public SecondaryIndexManager(ColumnFamilyStore baseCfs) {
        this.baseCfs = baseCfs;
    }

    public void reload() throws IOException {
        SortedSet<ByteBuffer> indexedColumnNames = this.getIndexedColumns();
        for (ByteBuffer indexedColumn : indexedColumnNames) {
            ColumnDefinition def = this.baseCfs.metadata.getColumn_metadata().get(indexedColumn);
            if (def != null && def.getIndexType() != null) continue;
            this.removeIndexedColumn(indexedColumn);
        }
        for (ColumnDefinition cdef : this.baseCfs.metadata.getColumn_metadata().values()) {
            if (cdef.getIndexType() == null || indexedColumnNames.contains(cdef.name)) continue;
            this.addIndexedColumn(cdef);
        }
    }

    public void maybeBuildSecondaryIndexes(Collection<SSTableReader> sstables, SortedSet<ByteBuffer> columns) throws IOException {
        if (columns.isEmpty()) {
            return;
        }
        logger.info(String.format("Submitting index build of %s for data in %s", this.baseCfs.metadata.comparator.getString(columns), StringUtils.join(sstables, (String)", ")));
        SecondaryIndexBuilder builder = new SecondaryIndexBuilder(this.baseCfs, columns, new ReducingKeyIterator(sstables));
        Future<?> future = CompactionManager.instance.submitIndexBuild(builder);
        try {
            future.get();
            this.flushIndexesBlocking();
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        logger.info("Index build of " + this.baseCfs.metadata.comparator.getString(columns) + " complete");
    }

    public SortedSet<ByteBuffer> getIndexedColumns() {
        return this.indexesByColumn.keySet();
    }

    public void removeIndexedColumn(ByteBuffer column) throws IOException {
        SecondaryIndex index = (SecondaryIndex)this.indexesByColumn.remove(column);
        if (index == null) {
            return;
        }
        if (index instanceof PerRowSecondaryIndex) {
            index.removeColumnDef(column);
            if (index.getColumnDefs().isEmpty()) {
                this.rowLevelIndexMap.remove(index.getClass());
            }
        }
        index.removeIndex(column);
        SystemTable.setIndexRemoved(this.baseCfs.metadata.ksName, index.getNameForSystemTable(column));
    }

    public synchronized Future<?> addIndexedColumn(ColumnDefinition cdef) {
        SecondaryIndex index;
        if (this.indexesByColumn.containsKey(cdef.name)) {
            return null;
        }
        assert (cdef.getIndexType() != null);
        logger.info("Creating new index : {}", (Object)cdef);
        try {
            index = SecondaryIndex.createInstance(this.baseCfs, cdef);
        }
        catch (ConfigurationException e) {
            throw new RuntimeException(e);
        }
        if (index instanceof PerRowSecondaryIndex) {
            SecondaryIndex currentIndex = this.rowLevelIndexMap.get(index.getClass());
            if (currentIndex == null) {
                this.rowLevelIndexMap.put(index.getClass(), index);
                index.init();
            } else {
                index = currentIndex;
                index.addColumnDef(cdef);
            }
        } else {
            index.init();
        }
        this.indexesByColumn.put(cdef.name, index);
        if (index.isIndexBuilt(cdef.name)) {
            return null;
        }
        return index.buildIndexAsync();
    }

    public SecondaryIndex getIndexForColumn(ByteBuffer column) {
        return (SecondaryIndex)this.indexesByColumn.get(column);
    }

    public void invalidate() {
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            ((SecondaryIndex)entry.getValue()).invalidate();
        }
    }

    public void flushIndexesBlocking() throws IOException {
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            ((SecondaryIndex)entry.getValue()).forceBlockingFlush();
        }
    }

    public DecoratedKey<LocalToken> getIndexKeyFor(ByteBuffer name, ByteBuffer value) {
        return new DecoratedKey<LocalToken>(new LocalToken(this.baseCfs.metadata.getColumnDefinition(name).getValidator(), value), value);
    }

    public List<String> getBuiltIndexes() {
        ArrayList<String> indexList = new ArrayList<String>();
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            SecondaryIndex index = (SecondaryIndex)entry.getValue();
            if (!index.isIndexBuilt((ByteBuffer)entry.getKey())) continue;
            indexList.add(((SecondaryIndex)entry.getValue()).getIndexName());
        }
        return indexList;
    }

    public Collection<ColumnFamilyStore> getIndexesBackedByCfs() {
        ArrayList<ColumnFamilyStore> cfsList = new ArrayList<ColumnFamilyStore>();
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            ColumnFamilyStore cfs = ((SecondaryIndex)entry.getValue()).getIndexCfs();
            if (cfs == null) continue;
            cfsList.add(cfs);
        }
        return cfsList;
    }

    public Collection<SecondaryIndex> getIndexesNotBackedByCfs() {
        IdentityHashMap indexList = new IdentityHashMap();
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            ColumnFamilyStore cfs = ((SecondaryIndex)entry.getValue()).getIndexCfs();
            if (cfs != null) continue;
            indexList.put(entry.getValue(), null);
        }
        return indexList.keySet();
    }

    public Collection<SecondaryIndex> getIndexes() {
        IdentityHashMap indexList = new IdentityHashMap();
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            indexList.put(entry.getValue(), null);
        }
        return indexList.keySet();
    }

    public long getTotalLiveSize() {
        long total = 0L;
        IdentityHashMap<SecondaryIndex, SecondaryIndex> indexList = new IdentityHashMap<SecondaryIndex, SecondaryIndex>();
        for (Map.Entry entry : this.indexesByColumn.entrySet()) {
            SecondaryIndex index = (SecondaryIndex)entry.getValue();
            if (indexList.put(index, index) != null) continue;
            total += index.getLiveSize();
        }
        return total;
    }

    public void applyIndexUpdates(ByteBuffer rowKey, ColumnFamily cf, SortedSet<ByteBuffer> mutatedIndexedColumns, ColumnFamily oldIndexedColumns) throws IOException {
        DecoratedKey<LocalToken> valueKey;
        SecondaryIndex index;
        IColumn column;
        HashSet appliedRowLevelIndexes = null;
        if (oldIndexedColumns != null) {
            for (ByteBuffer columnName : oldIndexedColumns.getColumnNames()) {
                column = oldIndexedColumns.getColumn(columnName);
                if (column == null || column.isMarkedForDelete()) continue;
                index = this.getIndexForColumn(columnName);
                if (index == null) {
                    logger.debug("Looks like index got dropped mid-update.  Skipping");
                    continue;
                }
                if (index instanceof PerRowSecondaryIndex) {
                    if (appliedRowLevelIndexes == null) {
                        appliedRowLevelIndexes = new HashSet();
                    }
                    if (!appliedRowLevelIndexes.add(index.getClass())) continue;
                    ((PerRowSecondaryIndex)index).applyIndexUpdates(rowKey, cf, mutatedIndexedColumns, oldIndexedColumns);
                    continue;
                }
                valueKey = this.getIndexKeyFor(columnName, column.value());
                ((PerColumnSecondaryIndex)index).deleteColumn(valueKey, rowKey, column);
            }
        }
        for (ByteBuffer columnName : mutatedIndexedColumns) {
            column = cf.getColumn(columnName);
            if (column == null || column.isMarkedForDelete()) continue;
            index = this.getIndexForColumn(columnName);
            if (index == null) {
                logger.debug("index on {} removed; skipping remove-old for {}", (Object)columnName, (Object)ByteBufferUtil.bytesToHex(rowKey));
                continue;
            }
            if (index instanceof PerRowSecondaryIndex) {
                if (appliedRowLevelIndexes == null) {
                    appliedRowLevelIndexes = new HashSet();
                }
                if (!appliedRowLevelIndexes.add(index.getClass())) continue;
                ((PerRowSecondaryIndex)index).applyIndexUpdates(rowKey, cf, mutatedIndexedColumns, oldIndexedColumns);
                continue;
            }
            valueKey = this.getIndexKeyFor(columnName, column.value());
            ((PerColumnSecondaryIndex)index).insertColumn(valueKey, rowKey, column);
        }
    }

    public void deleteFromIndexes(DecoratedKey<?> key, List<IColumn> indexedColumnsInRow) throws IOException {
        HashSet cleanedRowLevelIndexes = null;
        for (IColumn column : indexedColumnsInRow) {
            SecondaryIndex index = (SecondaryIndex)this.indexesByColumn.get(column.name());
            if (index == null) continue;
            if (index instanceof PerRowSecondaryIndex) {
                if (cleanedRowLevelIndexes == null) {
                    cleanedRowLevelIndexes = new HashSet();
                }
                if (!cleanedRowLevelIndexes.add(index.getClass())) continue;
                ((PerRowSecondaryIndex)index).deleteFromIndex(key, indexedColumnsInRow);
                continue;
            }
            DecoratedKey<LocalToken> valueKey = this.getIndexKeyFor(column.name(), column.value());
            ((PerColumnSecondaryIndex)index).deleteColumn(valueKey, key.key, column);
        }
    }

    private List<SecondaryIndexSearcher> getIndexSearchersForQuery(IndexClause clause) {
        ArrayList<SecondaryIndexSearcher> indexSearchers = new ArrayList<SecondaryIndexSearcher>();
        HashMap<String, HashSet<ByteBuffer>> groupByIndexType = new HashMap<String, HashSet<ByteBuffer>>();
        for (IndexExpression indexExpression : clause.expressions) {
            SecondaryIndex index = this.getIndexForColumn(indexExpression.column_name);
            if (index == null) continue;
            HashSet<ByteBuffer> columns = (HashSet<ByteBuffer>)groupByIndexType.get(index.getClass().getCanonicalName());
            if (columns == null) {
                columns = new HashSet<ByteBuffer>();
                groupByIndexType.put(index.getClass().getCanonicalName(), columns);
            }
            columns.add(indexExpression.column_name);
        }
        for (Map.Entry entry : groupByIndexType.entrySet()) {
            indexSearchers.add(this.getIndexForColumn((ByteBuffer)((Set)entry.getValue()).iterator().next()).createSecondaryIndexSearcher((Set)entry.getValue()));
        }
        return indexSearchers;
    }

    public List<Row> search(IndexClause clause, AbstractBounds range, IFilter dataFilter) {
        List<SecondaryIndexSearcher> indexSearchers = this.getIndexSearchersForQuery(clause);
        if (indexSearchers.isEmpty()) {
            return Collections.emptyList();
        }
        if (indexSearchers.size() > 1) {
            throw new RuntimeException("Unable to search across multiple secondary index types");
        }
        return indexSearchers.get(0).search(clause, range, dataFilter);
    }

    public boolean validate(Column column) {
        SecondaryIndex index = this.getIndexForColumn(column.name);
        return index != null ? index.validate(column) : true;
    }
}

