/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.context;

import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.cassandra.db.context.IContext;
import org.apache.cassandra.db.marshal.MarshalException;
import org.apache.cassandra.utils.Allocator;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.HeapAllocator;
import org.apache.cassandra.utils.NodeId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CounterContext
implements IContext {
    private static final int HEADER_SIZE_LENGTH = 2;
    private static final int HEADER_ELT_LENGTH = 2;
    private static final int CLOCK_LENGTH = 8;
    private static final int COUNT_LENGTH = 8;
    private static final int STEP_LENGTH = 32;
    private static final Logger logger = LoggerFactory.getLogger(CounterContext.class);

    public static CounterContext instance() {
        return LazyHolder.counterContext;
    }

    public ByteBuffer create(long value, Allocator allocator) {
        ByteBuffer context = allocator.allocate(36);
        context.putShort(context.position(), (short)1);
        context.putShort(context.position() + 2, (short)0);
        CounterContext.writeElementAtOffset(context, context.position() + 2 + 2, NodeId.getLocalId(), 1L, value);
        return context;
    }

    public ByteBuffer create(NodeId id, long clock, long value, boolean isDelta) {
        ByteBuffer context = ByteBuffer.allocate(2 + (isDelta ? 2 : 0) + 32);
        context.putShort(context.position(), (short)(isDelta ? 1 : 0));
        if (isDelta) {
            context.putShort(context.position() + 2, (short)0);
        }
        CounterContext.writeElementAtOffset(context, context.position() + 2 + (isDelta ? 2 : 0), id, clock, value);
        return context;
    }

    private static void writeElementAtOffset(ByteBuffer context, int offset, NodeId id, long clock, long count) {
        context = context.duplicate();
        context.position(offset);
        context.put(id.bytes().duplicate());
        context.putLong(clock);
        context.putLong(count);
    }

    private static int headerLength(ByteBuffer context) {
        return 2 + Math.abs(context.getShort(context.position())) * 2;
    }

    private static int compareId(ByteBuffer bb1, int pos1, ByteBuffer bb2, int pos2) {
        return ByteBufferUtil.compareSubArrays(bb1, pos1, bb2, pos2, 16);
    }

    @Override
    public IContext.ContextRelationship diff(ByteBuffer left, ByteBuffer right) {
        IContext.ContextRelationship relationship = IContext.ContextRelationship.EQUAL;
        ContextState leftState = new ContextState(left, CounterContext.headerLength(left));
        ContextState rightState = new ContextState(right, CounterContext.headerLength(right));
        while (leftState.hasRemaining() && rightState.hasRemaining()) {
            int compareId = leftState.compareIdTo(rightState);
            if (compareId == 0) {
                long leftClock = leftState.getClock();
                long rightClock = rightState.getClock();
                leftState.moveToNext();
                rightState.moveToNext();
                if (leftClock == rightClock) continue;
                if (leftClock >= 0L && rightClock > 0L && leftClock > rightClock || leftClock < 0L && (rightClock > 0L || leftClock < rightClock)) {
                    if (relationship == IContext.ContextRelationship.EQUAL) {
                        relationship = IContext.ContextRelationship.GREATER_THAN;
                        continue;
                    }
                    if (relationship == IContext.ContextRelationship.GREATER_THAN) continue;
                    return IContext.ContextRelationship.DISJOINT;
                }
                if (relationship == IContext.ContextRelationship.EQUAL) {
                    relationship = IContext.ContextRelationship.LESS_THAN;
                    continue;
                }
                if (relationship != IContext.ContextRelationship.GREATER_THAN) continue;
                return IContext.ContextRelationship.DISJOINT;
            }
            if (compareId > 0) {
                rightState.moveToNext();
                if (relationship == IContext.ContextRelationship.EQUAL) {
                    relationship = IContext.ContextRelationship.LESS_THAN;
                    continue;
                }
                if (relationship != IContext.ContextRelationship.GREATER_THAN) continue;
                return IContext.ContextRelationship.DISJOINT;
            }
            leftState.moveToNext();
            if (relationship == IContext.ContextRelationship.EQUAL) {
                relationship = IContext.ContextRelationship.GREATER_THAN;
                continue;
            }
            if (relationship == IContext.ContextRelationship.GREATER_THAN) continue;
            return IContext.ContextRelationship.DISJOINT;
        }
        if (leftState.hasRemaining()) {
            if (relationship == IContext.ContextRelationship.EQUAL) {
                return IContext.ContextRelationship.GREATER_THAN;
            }
            if (relationship == IContext.ContextRelationship.LESS_THAN) {
                return IContext.ContextRelationship.DISJOINT;
            }
        } else if (rightState.hasRemaining()) {
            if (relationship == IContext.ContextRelationship.EQUAL) {
                return IContext.ContextRelationship.LESS_THAN;
            }
            if (relationship == IContext.ContextRelationship.GREATER_THAN) {
                return IContext.ContextRelationship.DISJOINT;
            }
        }
        return relationship;
    }

    @Override
    public ByteBuffer merge(ByteBuffer left, ByteBuffer right, Allocator allocator) {
        ContextState leftState = new ContextState(left, CounterContext.headerLength(left));
        ContextState rightState = new ContextState(right, CounterContext.headerLength(right));
        int mergedHeaderLength = 2;
        int mergedBodyLength = 0;
        while (leftState.hasRemaining() && rightState.hasRemaining()) {
            int cmp = leftState.compareIdTo(rightState);
            if (cmp == 0) {
                mergedBodyLength += 32;
                if (leftState.isDelta() || rightState.isDelta()) {
                    mergedHeaderLength += 2;
                }
                leftState.moveToNext();
                rightState.moveToNext();
                continue;
            }
            if (cmp > 0) {
                mergedBodyLength += 32;
                if (rightState.isDelta()) {
                    mergedHeaderLength += 2;
                }
                rightState.moveToNext();
                continue;
            }
            mergedBodyLength += 32;
            if (leftState.isDelta()) {
                mergedHeaderLength += 2;
            }
            leftState.moveToNext();
        }
        ByteBuffer merged = allocator.allocate((mergedHeaderLength += leftState.remainingHeaderLength() + rightState.remainingHeaderLength()) + (mergedBodyLength += leftState.remainingBodyLength() + rightState.remainingBodyLength()));
        merged.putShort(merged.position(), (short)((mergedHeaderLength - 2) / 2));
        ContextState mergedState = new ContextState(merged, mergedHeaderLength);
        leftState.reset();
        rightState.reset();
        while (leftState.hasRemaining() && rightState.hasRemaining()) {
            int cmp = leftState.compareIdTo(rightState);
            if (cmp == 0) {
                if (leftState.isDelta() || rightState.isDelta()) {
                    if (leftState.isDelta() && rightState.isDelta()) {
                        long clock = leftState.getClock() + rightState.getClock();
                        long count = leftState.getCount() + rightState.getCount();
                        mergedState.writeElement(leftState.getNodeId(), clock, count, true);
                    } else {
                        (leftState.isDelta() ? leftState : rightState).copyTo(mergedState);
                    }
                } else {
                    long leftClock = leftState.getClock();
                    long rightClock = rightState.getClock();
                    if (leftClock >= 0L && rightClock > 0L && leftClock >= rightClock || leftClock < 0L && (rightClock > 0L || leftClock < rightClock)) {
                        leftState.copyTo(mergedState);
                    } else {
                        rightState.copyTo(mergedState);
                    }
                }
                rightState.moveToNext();
                leftState.moveToNext();
                continue;
            }
            if (cmp > 0) {
                rightState.copyTo(mergedState);
                rightState.moveToNext();
                continue;
            }
            leftState.copyTo(mergedState);
            leftState.moveToNext();
        }
        while (leftState.hasRemaining()) {
            leftState.copyTo(mergedState);
            leftState.moveToNext();
        }
        while (rightState.hasRemaining()) {
            rightState.copyTo(mergedState);
            rightState.moveToNext();
        }
        return merged;
    }

    @Override
    public String toString(ByteBuffer context) {
        ContextState state = new ContextState(context, CounterContext.headerLength(context));
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        while (state.hasRemaining()) {
            if (state.elementIdx() > 0) {
                sb.append(",");
            }
            sb.append("{");
            sb.append(state.getNodeId().toString()).append(", ");
            sb.append(state.getClock()).append(", ");
            sb.append(state.getCount());
            sb.append("}");
            if (state.isDelta()) {
                sb.append("*");
            }
            state.moveToNext();
        }
        sb.append("]");
        return sb.toString();
    }

    public long total(ByteBuffer context) {
        long total = 0L;
        for (int offset = context.position() + CounterContext.headerLength(context); offset < context.limit(); offset += 32) {
            long count = context.getLong(offset + 16 + 8);
            total += count;
        }
        return total;
    }

    public ByteBuffer markDeltaToBeCleared(ByteBuffer context) {
        int headerLength = CounterContext.headerLength(context);
        if (headerLength == 0) {
            return context;
        }
        ByteBuffer marked = context.duplicate();
        short count = context.getShort(context.position());
        if (count > 0) {
            marked.putShort(marked.position(), (short)(count * -1));
        }
        return marked;
    }

    public ByteBuffer clearAllDelta(ByteBuffer context) {
        int headerLength = CounterContext.headerLength(context);
        if (headerLength == 0) {
            return context;
        }
        ByteBuffer cleaned = ByteBuffer.allocate(context.remaining() - headerLength + 2);
        cleaned.putShort(cleaned.position(), (short)0);
        ByteBufferUtil.arrayCopy(context, context.position() + headerLength, cleaned, cleaned.position() + 2, context.remaining() - headerLength);
        return cleaned;
    }

    public void validateContext(ByteBuffer context) throws MarshalException {
        int headerLength = CounterContext.headerLength(context);
        if (headerLength < 0 || (context.remaining() - headerLength) % 32 != 0) {
            throw new MarshalException("Invalid size for a counter context");
        }
    }

    public void updateDigest(MessageDigest message, ByteBuffer context) {
        int hlength = CounterContext.headerLength(context);
        ByteBuffer dup = context.duplicate();
        dup.position(context.position() + hlength);
        message.update(dup);
    }

    public boolean hasNodeId(ByteBuffer context, NodeId id) {
        for (int offset = context.position() + CounterContext.headerLength(context); offset < context.limit(); offset += 32) {
            if (!id.equals(NodeId.wrap(context, offset))) continue;
            return true;
        }
        return false;
    }

    public ByteBuffer computeOldShardMerger(ByteBuffer context, List<NodeId.NodeIdRecord> oldIds, long mergeBefore) {
        NodeId nodeId;
        long now = System.currentTimeMillis();
        int hlength = CounterContext.headerLength(context);
        NodeId localId = NodeId.getLocalId();
        Iterator<NodeId.NodeIdRecord> recordIterator = oldIds.iterator();
        NodeId.NodeIdRecord currRecord = recordIterator.hasNext() ? recordIterator.next() : null;
        ContextState state = new ContextState(context, hlength);
        Object foundState = null;
        ArrayList<NodeId> toMerge = new ArrayList<NodeId>();
        long mergeTotal = 0L;
        while (state.hasRemaining() && currRecord != null) {
            assert (!currRecord.id.equals(localId));
            nodeId = state.getNodeId();
            int c = nodeId.compareTo(currRecord.id);
            if (c > 0) {
                currRecord = recordIterator.hasNext() ? recordIterator.next() : null;
                continue;
            }
            if (state.isDelta()) {
                if (state.getClock() < 0L) {
                    if (nodeId.equals(localId)) {
                        throw new RuntimeException("Current nodeId with a negative clock (likely due to #2968). You need to restart this node with -Dcassandra.renew_counter_id=true to fix.");
                    }
                    if (state.getCount() != 0L) {
                        logger.error(String.format("Invalid counter context (clock is %d and count is %d for NodeId %s), will fix", state.getCount(), state.getCount(), nodeId.toString()));
                        toMerge.add(nodeId);
                        mergeTotal += state.getCount();
                    }
                } else if (c == 0 && currRecord.timestamp < mergeBefore) {
                    toMerge.add(nodeId);
                    mergeTotal += state.getCount();
                }
            }
            if (c == 0) {
                currRecord = recordIterator.hasNext() ? recordIterator.next() : null;
            }
            state.moveToNext();
        }
        while (state.hasRemaining()) {
            nodeId = state.getNodeId();
            if (state.isDelta() && state.getClock() < 0L) {
                if (nodeId.equals(localId)) {
                    throw new RuntimeException("Current nodeId with a negative clock (likely due to #2968). You need to restart this node with -Dcassandra.renew_counter_id=true to fix.");
                }
                if (state.getCount() != 0L) {
                    logger.error(String.format("Invalid counter context (clock is %d and count is %d for NodeId %s), will fix", state.getClock(), state.getCount(), nodeId.toString()));
                    toMerge.add(nodeId);
                    mergeTotal += state.getCount();
                }
            }
            state.moveToNext();
        }
        if (toMerge.isEmpty()) {
            return null;
        }
        ContextState merger = ContextState.allocate(toMerge.size() + 1, toMerge.size() + 1);
        state.reset();
        int i = 0;
        int removedTotal = 0;
        boolean localWritten = false;
        while (state.hasRemaining()) {
            NodeId nodeId2 = state.getNodeId();
            if (nodeId2.compareTo(localId) > 0) {
                merger.writeElement(localId, 1L, mergeTotal, true);
                localWritten = true;
            } else if (i < toMerge.size() && nodeId2.compareTo((NodeId)toMerge.get(i)) == 0) {
                long count = state.getCount();
                removedTotal = (int)((long)removedTotal + count);
                merger.writeElement(nodeId2, -now - state.getClock(), -count, true);
                ++i;
            }
            state.moveToNext();
        }
        if (!localWritten) {
            merger.writeElement(localId, 1L, mergeTotal, true);
        }
        assert (mergeTotal == (long)removedTotal);
        return merger.context;
    }

    public ByteBuffer removeOldShards(ByteBuffer context, int gcBefore) {
        int hlength = CounterContext.headerLength(context);
        ContextState state = new ContextState(context, hlength);
        int removedShards = 0;
        int removedDelta = 0;
        while (state.hasRemaining()) {
            long clock = state.getClock();
            if (clock < 0L) {
                if (state.getCount() != 0L) {
                    if (state.isDelta()) {
                        throw new IllegalStateException("Counter shard with negative clock but count != 0; context = " + this.toString(context));
                    }
                    logger.debug("Ignoring non-removable non-delta corrupted shard in context " + this.toString(context));
                    state.moveToNext();
                    continue;
                }
                if (-((int)(clock / 1000L)) < gcBefore) {
                    ++removedShards;
                    if (state.isDelta()) {
                        ++removedDelta;
                    }
                }
            }
            state.moveToNext();
        }
        if (removedShards == 0) {
            return context;
        }
        int removedHeaderSize = removedDelta * 2;
        int removedBodySize = removedShards * 32;
        int newSize = context.remaining() - removedHeaderSize - removedBodySize;
        int newHlength = hlength - removedHeaderSize;
        ByteBuffer cleanedContext = HeapAllocator.instance.allocate(newSize);
        cleanedContext.putShort(cleanedContext.position(), (short)((newHlength - 2) / 2));
        ContextState cleaned = new ContextState(cleanedContext, newHlength);
        state.reset();
        while (state.hasRemaining()) {
            long clock = state.getClock();
            if (clock >= 0L || state.getCount() != 0L || -((int)(clock / 1000L)) >= gcBefore) {
                state.copyTo(cleaned);
            }
            state.moveToNext();
        }
        return cleanedContext;
    }

    public static class ContextState {
        public final ByteBuffer context;
        public final int headerLength;
        private int headerOffset;
        private int bodyOffset;
        private boolean currentIsDelta;

        public ContextState(ByteBuffer context, int headerLength) {
            this(context, headerLength, 2, headerLength, false);
            this.updateIsDelta();
        }

        public ContextState(ByteBuffer context) {
            this(context, CounterContext.headerLength(context));
        }

        private ContextState(ByteBuffer context, int headerLength, int headerOffset, int bodyOffset, boolean currentIsDelta) {
            this.context = context;
            this.headerLength = headerLength;
            this.headerOffset = headerOffset;
            this.bodyOffset = bodyOffset;
            this.currentIsDelta = currentIsDelta;
        }

        public boolean isDelta() {
            return this.currentIsDelta;
        }

        private void updateIsDelta() {
            this.currentIsDelta = this.headerOffset < this.headerLength && this.context.getShort(this.context.position() + this.headerOffset) == (short)this.elementIdx();
        }

        public boolean hasRemaining() {
            return this.bodyOffset < this.context.remaining();
        }

        public int remainingHeaderLength() {
            return this.headerLength - this.headerOffset;
        }

        public int remainingBodyLength() {
            return this.context.remaining() - this.bodyOffset;
        }

        public void moveToNext() {
            this.bodyOffset += 32;
            if (this.currentIsDelta) {
                this.headerOffset += 2;
            }
            this.updateIsDelta();
        }

        public void copyTo(ContextState other) {
            ByteBufferUtil.arrayCopy(this.context, this.context.position() + this.bodyOffset, other.context, other.context.position() + other.bodyOffset, 32);
            if (this.currentIsDelta) {
                other.context.putShort(other.context.position() + other.headerOffset, (short)other.elementIdx());
            }
            other.currentIsDelta = this.currentIsDelta;
            other.moveToNext();
        }

        public int compareIdTo(ContextState other) {
            return CounterContext.compareId(this.context, this.context.position() + this.bodyOffset, other.context, other.context.position() + other.bodyOffset);
        }

        public void reset() {
            this.headerOffset = 2;
            this.bodyOffset = this.headerLength;
            this.updateIsDelta();
        }

        public NodeId getNodeId() {
            return NodeId.wrap(this.context, this.context.position() + this.bodyOffset);
        }

        public long getClock() {
            return this.context.getLong(this.context.position() + this.bodyOffset + 16);
        }

        public long getCount() {
            return this.context.getLong(this.context.position() + this.bodyOffset + 16 + 8);
        }

        public void writeElement(NodeId id, long clock, long count, boolean isDelta) {
            CounterContext.writeElementAtOffset(this.context, this.context.position() + this.bodyOffset, id, clock, count);
            if (isDelta) {
                this.context.putShort(this.context.position() + this.headerOffset, (short)this.elementIdx());
            }
            this.currentIsDelta = isDelta;
            this.moveToNext();
        }

        public void writeElement(NodeId id, long clock, long count) {
            this.writeElement(id, clock, count, false);
        }

        public int elementIdx() {
            return (this.bodyOffset - this.headerLength) / 32;
        }

        public ContextState duplicate() {
            return new ContextState(this.context, this.headerLength, this.headerOffset, this.bodyOffset, this.currentIsDelta);
        }

        public static ContextState allocate(int elementCount, int deltaCount) {
            return ContextState.allocate(elementCount, deltaCount, HeapAllocator.instance);
        }

        public static ContextState allocate(int elementCount, int deltaCount, Allocator allocator) {
            assert (deltaCount <= elementCount);
            int hlength = 2 + deltaCount * 2;
            ByteBuffer context = allocator.allocate(hlength + elementCount * 32);
            context.putShort(context.position(), (short)deltaCount);
            return new ContextState(context, hlength);
        }
    }

    private static class LazyHolder {
        private static final CounterContext counterContext = new CounterContext();

        private LazyHolder() {
        }
    }
}

