/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.compaction.LeveledCompactionStrategy;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.commons.lang.StringUtils;
import org.codehaus.jackson.JsonEncoding;
import org.codehaus.jackson.JsonFactory;
import org.codehaus.jackson.JsonGenerator;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledManifest {
    private static final Logger logger = LoggerFactory.getLogger(LeveledCompactionStrategy.class);
    private final ColumnFamilyStore cfs;
    private final List<SSTableReader>[] generations;
    private final DecoratedKey[] lastCompactedKeys;
    private final int maxSSTableSizeInMB;

    private LeveledManifest(ColumnFamilyStore cfs, int maxSSTableSizeInMB) {
        this.cfs = cfs;
        this.maxSSTableSizeInMB = maxSSTableSizeInMB;
        int n = (int)Math.log10(1000000000 / maxSSTableSizeInMB);
        this.generations = new List[n];
        this.lastCompactedKeys = new DecoratedKey[n];
        for (int i = 0; i < this.generations.length; ++i) {
            this.generations[i] = new ArrayList<SSTableReader>();
            this.lastCompactedKeys[i] = new DecoratedKey(cfs.partitioner.getMinimumToken(), null);
        }
    }

    static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize) {
        LeveledManifest manifest = new LeveledManifest(cfs, maxSSTableSize);
        LeveledManifest.load(cfs, manifest);
        for (SSTableReader ssTableReader : cfs.getSSTables()) {
            if (manifest.levelOf(ssTableReader) >= 0) continue;
            manifest.add(ssTableReader);
        }
        return manifest;
    }

    private static void load(ColumnFamilyStore cfs, LeveledManifest manifest) {
        ObjectMapper m = new ObjectMapper();
        try {
            File manifestFile = LeveledManifest.tryGetManifest(cfs);
            if (manifestFile != null && manifestFile.exists()) {
                JsonNode rootNode = (JsonNode)m.readValue(manifestFile, JsonNode.class);
                JsonNode generations = rootNode.get("generations");
                assert (generations.isArray());
                for (JsonNode generation : generations) {
                    int level = generation.get("generation").getIntValue();
                    JsonNode generationValues = generation.get("members");
                    for (JsonNode generationValue : generationValues) {
                        for (SSTableReader ssTableReader : cfs.getSSTables()) {
                            if (ssTableReader.descriptor.generation != generationValue.getIntValue()) continue;
                            logger.debug("Loading {} at L{}", (Object)ssTableReader, (Object)level);
                            manifest.add(ssTableReader, level);
                        }
                    }
                }
            }
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public synchronized void add(SSTableReader reader) {
        logger.debug("Adding {} to L0", (Object)reader);
        this.add(reader, 0);
        this.serialize();
    }

    private int skipLevels(int newLevel, Iterable<SSTableReader> added) {
        if (this.maxBytesForLevel(newLevel) < (double)SSTableReader.getTotalBytes(added) && SSTableReader.getTotalBytes(this.generations[newLevel + 1]) == 0L) {
            newLevel = this.skipLevels(newLevel + 1, added);
        }
        return newLevel;
    }

    public synchronized void promote(Iterable<SSTableReader> removed, Iterable<SSTableReader> added) {
        logger.debug("Replacing [{}] with [{}]", (Object)StringUtils.join(removed.iterator(), (String)", "), (Object)StringUtils.join(added.iterator(), (String)", "));
        int minimumLevel = Integer.MAX_VALUE;
        int maximumLevel = 0;
        for (SSTableReader sstable : removed) {
            int thisLevel = this.levelOf(sstable);
            maximumLevel = Math.max(maximumLevel, thisLevel);
            minimumLevel = Math.min(minimumLevel, thisLevel);
            this.remove(sstable);
        }
        if (!added.iterator().hasNext()) {
            return;
        }
        int newLevel = minimumLevel == maximumLevel ? maximumLevel + 1 : maximumLevel;
        newLevel = this.skipLevels(newLevel, added);
        this.lastCompactedKeys[minimumLevel] = ((SSTableReader)SSTable.sstableOrdering.max(added)).last;
        logger.debug("Adding [{}] to L{}", (Object)StringUtils.join(added.iterator(), (String)", "), (Object)newLevel);
        for (SSTableReader ssTableReader : added) {
            this.add(ssTableReader, newLevel);
        }
        this.serialize();
    }

    private double maxBytesForLevel(int level) {
        return level == 0 ? (double)(4 * this.maxSSTableSizeInMB * 1024 * 1024) : Math.pow(10.0, level) * (double)this.maxSSTableSizeInMB * 1024.0 * 1024.0;
    }

    public synchronized Collection<SSTableReader> getCompactionCandidates() {
        this.logDistribution();
        double bestScore = -1.0;
        int bestLevel = -1;
        for (int level = 0; level < this.generations.length; ++level) {
            List<SSTableReader> sstables = this.generations[level];
            if (sstables.isEmpty()) continue;
            double score = (double)SSTableReader.getTotalBytes(sstables) / this.maxBytesForLevel(level);
            score = level == 0 && score < 1.0 ? 1.001 : 0.0;
            logger.debug("Compaction score for level {} is {}", (Object)level, (Object)score);
            if (!(score > bestScore)) continue;
            bestScore = score;
            bestLevel = level;
        }
        return bestScore > 1.0 ? this.getCandidatesFor(bestLevel) : Collections.emptyList();
    }

    public int getLevelSize(int i) {
        return this.generations.length > i ? this.generations[i].size() : 0;
    }

    public void logDistribution() {
        for (int i = 0; i < this.generations.length; ++i) {
            logger.debug("Level {} contains {} SSTables", (Object)i, (Object)this.generations[i].size());
        }
    }

    private int levelOf(SSTableReader sstable) {
        for (int level = 0; level < this.generations.length; ++level) {
            if (!this.generations[level].contains(sstable)) continue;
            return level;
        }
        return -1;
    }

    private void remove(SSTableReader reader) {
        int level = this.levelOf(reader);
        assert (level >= 0) : reader + " not present in manifest";
        this.generations[level].remove(reader);
    }

    private void add(SSTableReader sstable, int level) {
        this.generations[level].add(sstable);
    }

    private static List<SSTableReader> overlapping(SSTableReader sstable, Iterable<SSTableReader> candidates) {
        ArrayList<SSTableReader> overlapped = new ArrayList<SSTableReader>();
        overlapped.add(sstable);
        Range promotedRange = new Range((Token)sstable.first.token, (Token)sstable.last.token);
        for (SSTableReader candidate : candidates) {
            Range candidateRange = new Range((Token)candidate.first.token, (Token)candidate.last.token);
            if (!candidateRange.intersects(promotedRange)) continue;
            overlapped.add(candidate);
        }
        return overlapped;
    }

    private Collection<SSTableReader> getCandidatesFor(int level) {
        assert (!this.generations[level].isEmpty());
        if (level == 0) {
            HashSet<SSTableReader> candidates = new HashSet<SSTableReader>();
            HashSet<SSTableReader> remaining = new HashSet<SSTableReader>(this.generations[0]);
            while (!remaining.isEmpty()) {
                List<SSTableReader> L0 = LeveledManifest.overlapping((SSTableReader)remaining.iterator().next(), remaining);
                for (SSTableReader sstable : L0) {
                    candidates.addAll(LeveledManifest.overlapping(sstable, this.generations[1]));
                    remaining.remove(sstable);
                }
            }
            return candidates;
        }
        Collections.sort(this.generations[level], SSTable.sstableComparator);
        for (SSTableReader sstable : this.generations[level]) {
            if (sstable.first.compareTo(this.lastCompactedKeys[level]) <= 0) continue;
            return LeveledManifest.overlapping(sstable, this.generations[level + 1]);
        }
        return LeveledManifest.overlapping(this.generations[level].get(0), this.generations[level + 1]);
    }

    public synchronized void serialize() {
        File currentManifest;
        String dataFileLocation = LeveledManifest.getDataFilePrefix(this.cfs);
        String tempManifestFileName = dataFileLocation + this.cfs.getColumnFamilyName() + "-" + "tmp.json";
        String manifestFileName = dataFileLocation + this.cfs.getColumnFamilyName() + ".json";
        String oldManifestFileName = dataFileLocation + this.cfs.getColumnFamilyName() + "-" + "old.json";
        File tmpManifest = new File(tempManifestFileName);
        JsonFactory f = new JsonFactory();
        try {
            JsonGenerator g = f.createJsonGenerator(tmpManifest, JsonEncoding.UTF8);
            g.useDefaultPrettyPrinter();
            g.writeStartObject();
            g.writeArrayFieldStart("generations");
            for (int level = 0; level < this.generations.length; ++level) {
                g.writeStartObject();
                g.writeNumberField("generation", level);
                g.writeArrayFieldStart("members");
                for (SSTableReader ssTableReader : this.generations[level]) {
                    g.writeNumber(ssTableReader.descriptor.generation);
                }
                g.writeEndArray();
                g.writeEndObject();
            }
            g.writeEndArray();
            g.writeEndObject();
            g.close();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        File oldFile = new File(oldManifestFileName);
        if (oldFile.exists()) {
            oldFile.delete();
        }
        if ((currentManifest = new File(manifestFileName)).exists()) {
            currentManifest.renameTo(new File(oldManifestFileName));
        }
        if (tmpManifest.exists()) {
            tmpManifest.renameTo(new File(manifestFileName));
        }
    }

    public static File tryGetManifest(ColumnFamilyStore cfs) {
        for (String dataFileLocation : DatabaseDescriptor.getAllDataFileLocations()) {
            dataFileLocation = LeveledManifest.getDataFilePrefix(cfs);
            String manifestFileName = dataFileLocation + System.getProperty("file.separator") + cfs.table.name + ".json";
            File manifestFile = new File(manifestFileName);
            if (!manifestFile.exists()) continue;
            return manifestFile;
        }
        return null;
    }

    public static String getDataFilePrefix(ColumnFamilyStore cfs) {
        return DatabaseDescriptor.getAllDataFileLocations()[0] + System.getProperty("file.separator") + cfs.table.name + System.getProperty("file.separator");
    }
}

