/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.util.BufferedRandomAccessFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegment {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegment.class);
    private static Pattern COMMIT_LOG_FILE_PATTERN = Pattern.compile("CommitLog-(\\d+).log");
    public final long id;
    private final BufferedRandomAccessFile logWriter;
    private Map<Integer, Integer> cfLastWrite = new HashMap<Integer, Integer>();

    public CommitLogSegment() {
        this.id = System.currentTimeMillis();
        String logFile = DatabaseDescriptor.getCommitLogLocation() + File.separator + "CommitLog-" + this.id + ".log";
        logger.info("Creating new commitlog segment " + logFile);
        try {
            this.logWriter = CommitLogSegment.createWriter(logFile);
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public static long idFromFilename(String filename) {
        Matcher matcher = COMMIT_LOG_FILE_PATTERN.matcher(filename);
        try {
            if (matcher.matches()) {
                return Long.valueOf(matcher.group(1));
            }
            return -1L;
        }
        catch (NumberFormatException e) {
            return -1L;
        }
    }

    public static boolean possibleCommitLogFile(String filename) {
        return COMMIT_LOG_FILE_PATTERN.matcher(filename).matches();
    }

    private static BufferedRandomAccessFile createWriter(String file) throws IOException {
        return new BufferedRandomAccessFile(new File(file), "rw", 131072, true);
    }

    public ReplayPosition write(RowMutation rowMutation) throws IOException {
        long currentPosition = -1L;
        try {
            currentPosition = this.logWriter.getFilePointer();
            assert (currentPosition <= Integer.MAX_VALUE);
            ReplayPosition cLogCtx = new ReplayPosition(this.id, (int)currentPosition);
            for (ColumnFamily columnFamily : rowMutation.getColumnFamilies()) {
                CFMetaData cfm = DatabaseDescriptor.getCFMetaData(columnFamily.id());
                if (cfm == null) {
                    logger.error("Attempted to write commit log entry for unrecognized column family: " + columnFamily.id());
                    continue;
                }
                this.turnOn(cfm.cfId, (int)currentPosition);
            }
            CRC32 checksum = new CRC32();
            byte[] serializedRow = rowMutation.getSerializedBuffer(2);
            checksum.update(serializedRow.length);
            this.logWriter.writeInt(serializedRow.length);
            this.logWriter.writeLong(checksum.getValue());
            this.logWriter.write(serializedRow);
            checksum.update(serializedRow, 0, serializedRow.length);
            this.logWriter.writeLong(checksum.getValue());
            return cLogCtx;
        }
        catch (IOException e) {
            if (currentPosition != -1L) {
                this.logWriter.seek(currentPosition);
            }
            throw e;
        }
    }

    public void sync() throws IOException {
        this.logWriter.sync();
    }

    public ReplayPosition getContext() {
        long position = this.logWriter.getFilePointer();
        assert (position <= Integer.MAX_VALUE);
        return new ReplayPosition(this.id, (int)position);
    }

    public String getPath() {
        return this.logWriter.getPath();
    }

    public String getName() {
        return this.logWriter.getPath().substring(this.logWriter.getPath().lastIndexOf(File.separator) + 1);
    }

    public long length() {
        try {
            return this.logWriter.length();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public void close() {
        try {
            this.logWriter.close();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    void turnOn(Integer cfId, Integer position) {
        this.cfLastWrite.put(cfId, position);
    }

    void turnOffIfNotWritten(Integer cfId, Integer flushPosition) {
        Integer lastWritten = this.cfLastWrite.get(cfId);
        if (lastWritten == null || lastWritten < flushPosition) {
            this.cfLastWrite.remove(cfId);
        }
    }

    void turnOff(Integer cfId) {
        this.cfLastWrite.remove(cfId);
    }

    String dirtyString() {
        StringBuilder sb = new StringBuilder();
        for (Integer cfId : this.cfLastWrite.keySet()) {
            CFMetaData m = DatabaseDescriptor.getCFMetaData(cfId);
            sb.append(m == null ? "<deleted>" : m.cfName).append(" (").append(cfId).append("), ");
        }
        return sb.toString();
    }

    boolean isSafeToDelete() {
        return this.cfLastWrite.isEmpty();
    }

    public String toString() {
        return "CommitLogSegment(" + this.logWriter.getPath() + ')';
    }
}

