/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.cache.AutoSavingCache;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Memtable;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataTracker {
    private static final Logger logger = LoggerFactory.getLogger(DataTracker.class);
    public final ColumnFamilyStore cfstore;
    private final AtomicReference<View> view;
    private final AtomicLong liveSize = new AtomicLong();
    private final AtomicLong totalSize = new AtomicLong();

    public DataTracker(ColumnFamilyStore cfstore) {
        this.cfstore = cfstore;
        this.view = new AtomicReference();
        this.init();
    }

    public Memtable getMemtable() {
        return this.view.get().memtable;
    }

    public Set<Memtable> getMemtablesPendingFlush() {
        return this.view.get().memtablesPendingFlush;
    }

    public Set<SSTableReader> getSSTables() {
        return this.view.get().sstables;
    }

    public View getView() {
        return this.view.get();
    }

    public Memtable switchMemtable() {
        Memtable toFlushMemtable;
        View newView;
        View currentView;
        Memtable newMemtable = new Memtable(this.cfstore);
        do {
            currentView = this.view.get();
            toFlushMemtable = currentView.memtable;
        } while (!this.view.compareAndSet(currentView, newView = currentView.switchMemtable(newMemtable)));
        return toFlushMemtable;
    }

    public void renewMemtable() {
        View newView;
        View currentView;
        Memtable newMemtable = new Memtable(this.cfstore);
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.renewMemtable(newMemtable))) {
        }
    }

    public void replaceFlushed(Memtable memtable, SSTableReader sstable) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.replaceFlushed(memtable, sstable))) {
        }
        this.addNewSSTablesSize(Arrays.asList(sstable));
        this.cfstore.updateCacheSizes();
        this.incrementallyBackup(sstable);
    }

    public void incrementallyBackup(SSTableReader sstable) {
        if (DatabaseDescriptor.incrementalBackupsEnabled()) {
            File keyspaceDir = new File(sstable.getFilename()).getParentFile();
            File backupsDir = new File(keyspaceDir, "backups");
            try {
                if (!backupsDir.exists() && !backupsDir.mkdirs()) {
                    throw new IOException("Unable to create " + backupsDir);
                }
                sstable.createLinks(backupsDir.getCanonicalPath());
            }
            catch (IOException e) {
                throw new IOError(e);
            }
        }
    }

    public Set<SSTableReader> markCompacting(Collection<SSTableReader> tomark, int min, int max) {
        View newView;
        View currentView;
        if (max < min || max < 1) {
            return null;
        }
        HashSet<SSTableReader> subset = null;
        LinkedHashSet<SSTableReader> remaining = new LinkedHashSet<SSTableReader>(tomark);
        do {
            currentView = this.view.get();
            remaining.removeAll(currentView.compacting);
            remaining.retainAll(currentView.sstables);
            if (remaining.size() < min) {
                return null;
            }
            subset = new HashSet<SSTableReader>();
            Iterator iter = remaining.iterator();
            for (int added = 0; added < max && iter.hasNext(); ++added) {
                subset.add((SSTableReader)iter.next());
            }
        } while (!this.view.compareAndSet(currentView, newView = currentView.markCompacting(subset)));
        return subset;
    }

    public void unmarkCompacting(Collection<SSTableReader> unmark) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.unmarkCompacting(unmark))) {
        }
    }

    public void markCompacted(Collection<SSTableReader> sstables) {
        this.replace(sstables, Collections.<SSTableReader>emptyList());
    }

    public void replaceCompactedSSTables(Collection<SSTableReader> sstables, Iterable<SSTableReader> replacements) {
        this.replace(sstables, replacements);
    }

    public void addSSTables(Collection<SSTableReader> sstables) {
        this.replace(Collections.<SSTableReader>emptyList(), sstables);
    }

    public void addStreamedSSTable(SSTableReader sstable) {
        this.addSSTables(Arrays.asList(sstable));
        this.incrementallyBackup(sstable);
    }

    public void removeAllSSTables() {
        this.replace(this.getSSTables(), Collections.<SSTableReader>emptyList());
    }

    void init() {
        this.view.set(new View(new Memtable(this.cfstore), Collections.<Memtable>emptySet(), Collections.<SSTableReader>emptySet(), Collections.<SSTableReader>emptySet()));
    }

    private void replace(Collection<SSTableReader> oldSSTables, Iterable<SSTableReader> replacements) {
        View newView;
        View currentView;
        while (!this.view.compareAndSet(currentView = this.view.get(), newView = currentView.replace(oldSSTables, replacements))) {
        }
        this.addNewSSTablesSize(replacements);
        this.removeOldSSTablesSize(oldSSTables);
        this.cfstore.updateCacheSizes();
    }

    private void addNewSSTablesSize(Iterable<SSTableReader> newSSTables) {
        for (SSTableReader sstable : newSSTables) {
            assert (sstable.getKeySamples() != null);
            if (logger.isDebugEnabled()) {
                logger.debug(String.format("adding %s to list of files tracked for %s.%s", sstable.descriptor, this.cfstore.table.name, this.cfstore.getColumnFamilyName()));
            }
            long size = sstable.bytesOnDisk();
            this.liveSize.addAndGet(size);
            this.totalSize.addAndGet(size);
            sstable.setTrackedBy(this);
        }
    }

    private void removeOldSSTablesSize(Iterable<SSTableReader> oldSSTables) {
        for (SSTableReader sstable : oldSSTables) {
            if (logger.isDebugEnabled()) {
                logger.debug(String.format("removing %s from list of files tracked for %s.%s", sstable.descriptor, this.cfstore.table.name, this.cfstore.getColumnFamilyName()));
            }
            sstable.markCompacted();
            this.liveSize.addAndGet(-sstable.bytesOnDisk());
        }
    }

    public AutoSavingCache<Pair<Descriptor, DecoratedKey>, Long> getKeyCache() {
        return this.cfstore.getKeyCache();
    }

    public long getLiveSize() {
        return this.liveSize.get();
    }

    public long getTotalSize() {
        return this.totalSize.get();
    }

    public void spaceReclaimed(long size) {
        this.totalSize.addAndGet(-size);
    }

    public long estimatedKeys() {
        long n = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            n += sstable.estimatedKeys();
        }
        return n;
    }

    public long[] getEstimatedRowSizeHistogram() {
        long[] histogram = new long[90];
        for (SSTableReader sstable : this.getSSTables()) {
            long[] rowSize = sstable.getEstimatedRowSize().getBuckets(false);
            for (int i = 0; i < histogram.length; ++i) {
                int n = i;
                histogram[n] = histogram[n] + rowSize[i];
            }
        }
        return histogram;
    }

    public long[] getEstimatedColumnCountHistogram() {
        long[] histogram = new long[90];
        for (SSTableReader sstable : this.getSSTables()) {
            long[] columnSize = sstable.getEstimatedColumnCount().getBuckets(false);
            for (int i = 0; i < histogram.length; ++i) {
                int n = i;
                histogram[n] = histogram[n] + columnSize[i];
            }
        }
        return histogram;
    }

    public long getMinRowSize() {
        long min = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            if (min != 0L && sstable.getEstimatedRowSize().min() >= min) continue;
            min = sstable.getEstimatedRowSize().min();
        }
        return min;
    }

    public long getMaxRowSize() {
        long max = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            if (sstable.getEstimatedRowSize().max() <= max) continue;
            max = sstable.getEstimatedRowSize().max();
        }
        return max;
    }

    public long getMeanRowSize() {
        long sum = 0L;
        long count = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            sum += sstable.getEstimatedRowSize().mean();
            ++count;
        }
        return count > 0L ? sum / count : 0L;
    }

    public int getMeanColumns() {
        long sum = 0L;
        int count = 0;
        for (SSTableReader sstable : this.getSSTables()) {
            sum += sstable.getEstimatedColumnCount().mean();
            ++count;
        }
        return count > 0 ? (int)(sum / (long)count) : 0;
    }

    public long getBloomFilterFalsePositives() {
        long count = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            count += sstable.getBloomFilterFalsePositiveCount();
        }
        return count;
    }

    public long getRecentBloomFilterFalsePositives() {
        long count = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            count += sstable.getRecentBloomFilterFalsePositiveCount();
        }
        return count;
    }

    public double getBloomFilterFalseRatio() {
        long falseCount = 0L;
        long trueCount = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            falseCount += sstable.getBloomFilterFalsePositiveCount();
            trueCount += sstable.getBloomFilterTruePositiveCount();
        }
        if (falseCount == 0L && trueCount == 0L) {
            return 0.0;
        }
        return (double)falseCount / (double)(trueCount + falseCount);
    }

    public double getRecentBloomFilterFalseRatio() {
        long falseCount = 0L;
        long trueCount = 0L;
        for (SSTableReader sstable : this.getSSTables()) {
            falseCount += sstable.getRecentBloomFilterFalsePositiveCount();
            trueCount += sstable.getRecentBloomFilterTruePositiveCount();
        }
        if (falseCount == 0L && trueCount == 0L) {
            return 0.0;
        }
        return (double)falseCount / (double)(trueCount + falseCount);
    }

    static class View {
        public final Memtable memtable;
        public final Set<Memtable> memtablesPendingFlush;
        public final Set<SSTableReader> sstables;
        public final Set<SSTableReader> compacting;

        View(Memtable memtable, Set<Memtable> pendingFlush, Set<SSTableReader> sstables, Set<SSTableReader> compacting) {
            this.memtable = memtable;
            this.memtablesPendingFlush = pendingFlush;
            this.sstables = sstables;
            this.compacting = compacting;
        }

        public View switchMemtable(Memtable newMemtable) {
            ImmutableSet newPending = ImmutableSet.builder().addAll(this.memtablesPendingFlush).add((Object)this.memtable).build();
            return new View(newMemtable, (Set<Memtable>)newPending, this.sstables, this.compacting);
        }

        public View renewMemtable(Memtable newMemtable) {
            return new View(newMemtable, this.memtablesPendingFlush, this.sstables, this.compacting);
        }

        public View replaceFlushed(Memtable flushedMemtable, SSTableReader newSSTable) {
            ImmutableSet newPending = ImmutableSet.copyOf((Collection)Sets.difference(this.memtablesPendingFlush, Collections.singleton(flushedMemtable)));
            ImmutableSet newSSTables = ImmutableSet.builder().addAll(this.sstables).add((Object)newSSTable).build();
            return new View(this.memtable, (Set<Memtable>)newPending, (Set<SSTableReader>)newSSTables, this.compacting);
        }

        public View replace(Collection<SSTableReader> oldSSTables, Iterable<SSTableReader> replacements) {
            Sets.SetView remaining = Sets.difference(this.sstables, (Set)ImmutableSet.copyOf(oldSSTables));
            ImmutableSet newSSTables = ImmutableSet.builder().addAll((Iterable)remaining).addAll(replacements).build();
            return new View(this.memtable, this.memtablesPendingFlush, (Set<SSTableReader>)newSSTables, this.compacting);
        }

        public View markCompacting(Collection<SSTableReader> tomark) {
            ImmutableSet compactingNew = ImmutableSet.builder().addAll(this.sstables).addAll(tomark).build();
            return new View(this.memtable, this.memtablesPendingFlush, this.sstables, (Set<SSTableReader>)compactingNew);
        }

        public View unmarkCompacting(Collection<SSTableReader> tounmark) {
            ImmutableSet compactingNew = ImmutableSet.copyOf((Collection)Sets.difference(this.compacting, (Set)ImmutableSet.copyOf(tounmark)));
            return new View(this.memtable, this.memtablesPendingFlush, this.sstables, (Set<SSTableReader>)compactingNew);
        }
    }
}

