/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileMark;
import org.apache.cassandra.utils.CLibrary;

public class BufferedRandomAccessFile
extends RandomAccessFile
implements FileDataInput {
    private static final long MAX_BYTES_IN_PAGE_CACHE = (long)Math.pow(2.0, 27.0);
    private final String filePath;
    public static final int DEFAULT_BUFFER_SIZE = 65535;
    private boolean isDirty;
    private boolean syncNeeded;
    private byte[] buffer;
    private long bufferOffset;
    private long current = 0L;
    private int validBufferBytes = 0;
    private final long fileLength;
    private final FileChannel channel;
    private long markedPointer;
    private int fd;
    private final boolean skipCache;
    private long bytesSinceCacheFlush = 0L;
    private long minBufferOffset = Long.MAX_VALUE;
    private final byte[] singleByteBuffer = new byte[1];

    public BufferedRandomAccessFile(String name, String mode) throws IOException {
        this(new File(name), mode, 65535);
    }

    public BufferedRandomAccessFile(String name, String mode, int bufferSize) throws IOException {
        this(new File(name), mode, bufferSize);
    }

    public BufferedRandomAccessFile(File file, String mode) throws IOException {
        this(file, mode, 65535);
    }

    public BufferedRandomAccessFile(File file, String mode, int bufferSize) throws IOException {
        this(file, mode, bufferSize, false);
    }

    public BufferedRandomAccessFile(File file, String mode, int bufferSize, boolean skipCache) throws IOException {
        super(file, mode);
        this.skipCache = skipCache;
        this.channel = super.getChannel();
        this.filePath = file.getAbsolutePath();
        if (bufferSize <= 0) {
            throw new IllegalArgumentException("bufferSize must be positive");
        }
        this.buffer = new byte[bufferSize];
        this.reBuffer();
        this.fileLength = mode.equals("r") ? this.channel.size() : -1L;
        this.fd = CLibrary.getfd(this.getFD());
    }

    public void sync() throws IOException {
        if (this.syncNeeded) {
            this.flush();
            this.channel.force(true);
            if (this.skipCache) {
                CLibrary.trySkipCache(this.fd, 0, 0);
                this.minBufferOffset = Long.MAX_VALUE;
                this.bytesSinceCacheFlush = 0L;
            }
            this.syncNeeded = false;
        }
    }

    public void flush() throws IOException {
        if (this.isDirty) {
            if (this.channel.position() != this.bufferOffset) {
                this.channel.position(this.bufferOffset);
            }
            super.write(this.buffer, 0, this.validBufferBytes);
            if (this.skipCache) {
                this.bytesSinceCacheFlush += (long)this.validBufferBytes;
                if (this.bufferOffset < this.minBufferOffset) {
                    this.minBufferOffset = this.bufferOffset;
                }
                if (this.bytesSinceCacheFlush >= MAX_BYTES_IN_PAGE_CACHE) {
                    CLibrary.trySkipCache(this.fd, (int)this.minBufferOffset, 0);
                    this.minBufferOffset = this.bufferOffset;
                    this.bytesSinceCacheFlush = 0L;
                }
            }
            this.isDirty = false;
        }
    }

    @Override
    public void setLength(long newLength) throws IOException {
        if (newLength < 0L) {
            throw new IllegalArgumentException();
        }
        if (this.isDirty) {
            if (newLength < this.bufferOffset) {
                this.validBufferBytes = 0;
            } else if (newLength > this.bufferOffset + (long)this.validBufferBytes) {
                this.flush();
            } else {
                this.validBufferBytes = (int)(newLength - this.bufferOffset);
                this.flush();
            }
        }
        super.setLength(newLength);
        this.validBufferBytes = 0;
        this.current = newLength;
        this.reBuffer();
    }

    private void reBuffer() throws IOException {
        int read;
        int n;
        this.flush();
        this.bufferOffset = this.current;
        if (this.bufferOffset >= this.channel.size()) {
            this.validBufferBytes = 0;
            return;
        }
        if (this.bufferOffset < this.minBufferOffset) {
            this.minBufferOffset = this.bufferOffset;
        }
        this.channel.position(this.bufferOffset);
        for (read = 0; read < this.buffer.length && (n = super.read(this.buffer, read, this.buffer.length - read)) >= 0; read += n) {
        }
        this.validBufferBytes = read;
        this.bytesSinceCacheFlush += (long)read;
        if (this.skipCache && this.bytesSinceCacheFlush >= MAX_BYTES_IN_PAGE_CACHE) {
            CLibrary.trySkipCache(this.fd, (int)this.minBufferOffset, 0);
            this.bytesSinceCacheFlush = 0L;
            this.minBufferOffset = Long.MAX_VALUE;
        }
    }

    @Override
    public int read() throws IOException {
        if (this.isEOF()) {
            return -1;
        }
        if (this.current >= this.bufferOffset + (long)this.buffer.length) {
            this.reBuffer();
        }
        assert (this.current >= this.bufferOffset && this.current < this.bufferOffset + (long)this.validBufferBytes);
        return this.buffer[(int)(this.current++ - this.bufferOffset)] & 0xFF;
    }

    @Override
    public int read(byte[] buffer) throws IOException {
        return this.read(buffer, 0, buffer.length);
    }

    @Override
    public int read(byte[] buff, int offset, int length) throws IOException {
        if (length == 0) {
            return 0;
        }
        if (this.isEOF()) {
            return -1;
        }
        if (this.current >= this.bufferOffset + (long)this.buffer.length) {
            this.reBuffer();
        }
        assert (this.current >= this.bufferOffset && this.current < this.bufferOffset + (long)this.validBufferBytes);
        int toCopy = Math.min(length, this.validBufferBytes - (int)(this.current - this.bufferOffset));
        System.arraycopy(this.buffer, (int)(this.current - this.bufferOffset), buff, offset, toCopy);
        this.current += (long)toCopy;
        return toCopy;
    }

    @Override
    public ByteBuffer readBytes(int length) throws IOException {
        assert (length >= 0) : "buffer length should not be negative: " + length;
        byte[] buff = new byte[length];
        this.readFully(buff);
        return ByteBuffer.wrap(buff);
    }

    @Override
    public void write(int val) throws IOException {
        this.singleByteBuffer[0] = (byte)val;
        this.write(this.singleByteBuffer, 0, 1);
    }

    @Override
    public void write(byte[] b) throws IOException {
        this.write(b, 0, b.length);
    }

    @Override
    public void write(byte[] buff, int offset, int length) throws IOException {
        if (this.buffer == null) {
            throw new ClosedChannelException();
        }
        if (this.isReadOnly()) {
            throw new IOException("Unable to write: file is in the read-only mode.");
        }
        while (length > 0) {
            int n = this.writeAtMost(buff, offset, length);
            offset += n;
            length -= n;
            this.isDirty = true;
            this.syncNeeded = true;
        }
    }

    private boolean isReadOnly() {
        return this.fileLength != -1L;
    }

    private int writeAtMost(byte[] buff, int offset, int length) throws IOException {
        if (this.current >= this.bufferOffset + (long)this.buffer.length) {
            this.reBuffer();
        }
        assert (this.current < this.bufferOffset + (long)this.buffer.length);
        int positionWithinBuffer = (int)(this.current - this.bufferOffset);
        int toCopy = Math.min(length, this.buffer.length - positionWithinBuffer);
        System.arraycopy(buff, offset, this.buffer, positionWithinBuffer, toCopy);
        this.current += (long)toCopy;
        this.validBufferBytes = Math.max(this.validBufferBytes, positionWithinBuffer + toCopy);
        assert (this.current <= this.bufferOffset + (long)this.buffer.length);
        return toCopy;
    }

    @Override
    public void seek(long newPosition) throws IOException {
        if (newPosition < 0L) {
            throw new IllegalArgumentException("new position should not be negative");
        }
        if (this.isReadOnly() && newPosition > this.fileLength) {
            throw new EOFException(String.format("unable to seek to position %d in %s (%d bytes) in read-only mode", newPosition, this.filePath, this.fileLength));
        }
        this.current = newPosition;
        if (newPosition > this.bufferOffset + (long)this.validBufferBytes || newPosition < this.bufferOffset) {
            this.reBuffer();
        }
    }

    @Override
    public int skipBytes(int count) throws IOException {
        if (count > 0) {
            long eof;
            long currentPos = this.getFilePointer();
            int newCount = (int)(currentPos + (long)count > (eof = this.length()) ? eof - currentPos : (long)count);
            this.seek(currentPos + (long)newCount);
            return newCount;
        }
        return 0;
    }

    @Override
    public long length() throws IOException {
        return this.fileLength == -1L ? Math.max(Math.max(this.current, this.channel.size()), this.bufferOffset + (long)this.validBufferBytes) : this.fileLength;
    }

    @Override
    public long getFilePointer() {
        return this.current;
    }

    @Override
    public String getPath() {
        return this.filePath;
    }

    @Override
    public boolean isEOF() throws IOException {
        return this.getFilePointer() == this.length();
    }

    @Override
    public long bytesRemaining() throws IOException {
        return this.length() - this.getFilePointer();
    }

    @Override
    public void close() throws IOException {
        this.sync();
        this.buffer = null;
        if (this.skipCache && this.bytesSinceCacheFlush > 0L) {
            CLibrary.trySkipCache(this.fd, 0, 0);
        }
        super.close();
    }

    public void reset() throws IOException {
        this.seek(this.markedPointer);
    }

    public long bytesPastMark() {
        long bytes = this.getFilePointer() - this.markedPointer;
        assert (bytes >= 0L);
        return bytes;
    }

    @Override
    public FileMark mark() {
        this.markedPointer = this.getFilePointer();
        return new BufferedRandomAccessFileMark(this.markedPointer);
    }

    @Override
    public void reset(FileMark mark) throws IOException {
        assert (mark instanceof BufferedRandomAccessFileMark);
        this.seek(((BufferedRandomAccessFileMark)mark).pointer);
    }

    @Override
    public long bytesPastMark(FileMark mark) {
        assert (mark instanceof BufferedRandomAccessFileMark);
        long bytes = this.getFilePointer() - ((BufferedRandomAccessFileMark)mark).pointer;
        assert (bytes >= 0L);
        return bytes;
    }

    public static BufferedRandomAccessFile getUncachingReader(String filename) throws IOException {
        return new BufferedRandomAccessFile(new File(filename), "r", 0x800000, true);
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + "filePath='" + this.filePath + "'" + ", length=" + this.fileLength + ", skipCache=" + this.skipCache + ")";
    }

    protected static class BufferedRandomAccessFileMark
    implements FileMark {
        long pointer;

        public BufferedRandomAccessFileMark(long pointer) {
            this.pointer = pointer;
        }
    }
}

