/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.avro;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Comparator;
import java.util.SortedSet;
import org.apache.avro.util.Utf8;
import org.apache.cassandra.avro.AvroRecordFactory;
import org.apache.cassandra.avro.Column;
import org.apache.cassandra.avro.ColumnOrSuperColumn;
import org.apache.cassandra.avro.ColumnParent;
import org.apache.cassandra.avro.ColumnPath;
import org.apache.cassandra.avro.Deletion;
import org.apache.cassandra.avro.ErrorFactory;
import org.apache.cassandra.avro.IndexClause;
import org.apache.cassandra.avro.IndexExpression;
import org.apache.cassandra.avro.IndexOperator;
import org.apache.cassandra.avro.InvalidRequestException;
import org.apache.cassandra.avro.KeyRange;
import org.apache.cassandra.avro.KeyspaceNotDefinedException;
import org.apache.cassandra.avro.Mutation;
import org.apache.cassandra.avro.SlicePredicate;
import org.apache.cassandra.avro.SliceRange;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyType;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.MarshalException;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.RandomPartitioner;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.service.StorageService;

public class AvroValidation {
    static void validateKey(ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw ErrorFactory.newInvalidRequestException("Key may not be empty");
        }
        if (key.remaining() > 65535) {
            throw ErrorFactory.newInvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
    }

    static void validateKeyspace(String keyspace) throws KeyspaceNotDefinedException {
        if (!DatabaseDescriptor.getTables().contains(keyspace)) {
            throw new KeyspaceNotDefinedException(new Utf8("Keyspace " + keyspace + " does not exist in this schema."));
        }
    }

    static ColumnFamilyType validateColumnFamily(String keyspace, String columnFamily) throws InvalidRequestException {
        if (columnFamily.isEmpty()) {
            throw ErrorFactory.newInvalidRequestException("non-empty columnfamily is required");
        }
        ColumnFamilyType cfType = DatabaseDescriptor.getColumnFamilyType(keyspace, columnFamily);
        if (cfType == null) {
            throw ErrorFactory.newInvalidRequestException("unconfigured columnfamily " + columnFamily);
        }
        return cfType;
    }

    static void validateColumnPath(String keyspace, ColumnPath cp) throws InvalidRequestException {
        AvroValidation.validateKeyspace(keyspace);
        String column_family = ((Object)cp.column_family).toString();
        ColumnFamilyType cfType = AvroValidation.validateColumnFamily(keyspace, column_family);
        if (cfType == ColumnFamilyType.Standard) {
            if (cp.super_column != null) {
                throw ErrorFactory.newInvalidRequestException("supercolumn parameter is invalid for standard CF " + column_family);
            }
            if (cp.column == null) {
                throw ErrorFactory.newInvalidRequestException("column parameter is not optional for standard CF " + column_family);
            }
        } else if (cp.super_column == null) {
            throw ErrorFactory.newInvalidRequestException("supercolumn parameter is not optional for super CF " + column_family);
        }
        if (cp.column != null) {
            AvroValidation.validateColumns(keyspace, column_family, cp.super_column, Arrays.asList(cp.column));
        }
        if (cp.super_column != null) {
            AvroValidation.validateColumns(keyspace, column_family, null, Arrays.asList(cp.super_column));
        }
    }

    static void validateColumnParent(String keyspace, ColumnParent parent) throws InvalidRequestException {
        AvroValidation.validateKeyspace(keyspace);
        String cfName = ((Object)parent.column_family).toString();
        ColumnFamilyType cfType = AvroValidation.validateColumnFamily(keyspace, cfName);
        if (cfType == ColumnFamilyType.Standard && parent.super_column != null) {
            throw ErrorFactory.newInvalidRequestException("super column specified for standard column family");
        }
        if (parent.super_column != null) {
            AvroValidation.validateColumns(keyspace, cfName, null, Arrays.asList(parent.super_column));
        }
    }

    static void validateColumns(String keyspace, String cfName, ByteBuffer superColumnName, Iterable<ByteBuffer> columnNames) throws InvalidRequestException {
        if (superColumnName != null) {
            if (superColumnName.remaining() > 65535) {
                throw ErrorFactory.newInvalidRequestException("supercolumn name length must not be greater than 65535");
            }
            if (superColumnName.remaining() == 0) {
                throw ErrorFactory.newInvalidRequestException("supercolumn name must not be empty");
            }
            if (DatabaseDescriptor.getColumnFamilyType(keyspace, cfName) == ColumnFamilyType.Standard) {
                throw ErrorFactory.newInvalidRequestException("supercolumn specified to ColumnFamily " + cfName + " containing normal columns");
            }
        }
        AbstractType comparator = ColumnFamily.getComparatorFor(keyspace, cfName, superColumnName);
        for (ByteBuffer buff : columnNames) {
            if (buff.remaining() > 65535) {
                throw ErrorFactory.newInvalidRequestException("column name length must not be greater than 65535");
            }
            if (buff.remaining() == 0) {
                throw ErrorFactory.newInvalidRequestException("column name must not be empty");
            }
            try {
                comparator.validate(buff);
            }
            catch (MarshalException e) {
                throw ErrorFactory.newInvalidRequestException(e.getMessage());
            }
        }
    }

    static void validateColumns(String keyspace, ColumnParent parent, Iterable<ByteBuffer> columnNames) throws InvalidRequestException {
        AvroValidation.validateColumns(keyspace, ((Object)parent.column_family).toString(), parent.super_column, columnNames);
    }

    static void validateColumn(String keyspace, ColumnParent parent, Column column) throws InvalidRequestException {
        AvroValidation.validateTtl(column);
        AvroValidation.validateColumns(keyspace, parent, Arrays.asList(column.name));
    }

    static void validateColumnOrSuperColumn(String keyspace, String cfName, ColumnOrSuperColumn cosc) throws InvalidRequestException {
        if (cosc.column != null) {
            AvroValidation.validateColumnPath(keyspace, AvroRecordFactory.newColumnPath(cfName, null, cosc.column.name));
        }
        if (cosc.super_column != null) {
            for (Column c : cosc.super_column.columns) {
                AvroValidation.validateColumnPath(keyspace, AvroRecordFactory.newColumnPath(cfName, cosc.super_column.name, c.name));
            }
        }
        if (cosc.column == null && cosc.super_column == null) {
            throw ErrorFactory.newInvalidRequestException("ColumnOrSuperColumn must have one or both of Column or SuperColumn");
        }
    }

    static void validateRange(String keyspace, String cfName, ByteBuffer superName, SliceRange range) throws InvalidRequestException {
        Comparator<ByteBuffer> orderedComparator;
        AbstractType comparator = ColumnFamily.getComparatorFor(keyspace, cfName, superName);
        try {
            comparator.validate(range.start);
            comparator.validate(range.finish);
        }
        catch (MarshalException me) {
            throw ErrorFactory.newInvalidRequestException(me.getMessage());
        }
        if (range.count < 0) {
            throw ErrorFactory.newInvalidRequestException("Ranges require a non-negative count.");
        }
        Comparator<ByteBuffer> comparator2 = orderedComparator = range.reversed ? comparator.reverseComparator : comparator;
        if (range.start.remaining() > 0 && range.finish.remaining() > 0 && orderedComparator.compare(range.start, range.finish) > 0) {
            throw ErrorFactory.newInvalidRequestException("range finish must come after start in the order of traversal");
        }
    }

    static void validateRange(String keyspace, ColumnParent cp, SliceRange range) throws InvalidRequestException {
        AvroValidation.validateRange(keyspace, ((Object)cp.column_family).toString(), cp.super_column, range);
    }

    static void validateSlicePredicate(String keyspace, String cfName, ByteBuffer superName, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw ErrorFactory.newInvalidRequestException("A SlicePredicate must be given a list of Columns, a SliceRange, or both");
        }
        if (predicate.slice_range != null) {
            AvroValidation.validateRange(keyspace, cfName, superName, predicate.slice_range);
        }
        if (predicate.column_names != null) {
            AvroValidation.validateColumns(keyspace, cfName, superName, predicate.column_names);
        }
    }

    static void validateDeletion(String keyspace, String cfName, Deletion del) throws InvalidRequestException {
        AvroValidation.validateColumnFamily(keyspace, cfName);
        if (del.super_column == null && del.predicate == null) {
            throw ErrorFactory.newInvalidRequestException("A Deletion must have a SuperColumn, a SlicePredicate, or both.");
        }
        if (del.predicate != null) {
            AvroValidation.validateSlicePredicate(keyspace, cfName, del.super_column, del.predicate);
            if (del.predicate.slice_range != null) {
                throw ErrorFactory.newInvalidRequestException("Deletion does not yet support SliceRange predicates.");
            }
        }
    }

    static void validateMutation(String keyspace, String cfName, Mutation mutation) throws InvalidRequestException {
        ColumnOrSuperColumn cosc = mutation.column_or_supercolumn;
        Deletion del = mutation.deletion;
        if (cosc != null && del != null) {
            throw ErrorFactory.newInvalidRequestException("Mutation may have either a ColumnOrSuperColumn or a Deletion, but not both");
        }
        if (cosc != null) {
            AvroValidation.validateColumnOrSuperColumn(keyspace, cfName, cosc);
        } else if (del != null) {
            AvroValidation.validateDeletion(keyspace, cfName, del);
        } else {
            throw ErrorFactory.newInvalidRequestException("Mutation must have one ColumnOrSuperColumn, or one Deletion");
        }
    }

    static void validateTtl(Column column) throws InvalidRequestException {
        if (column.ttl != null && column.ttl < 0) {
            throw ErrorFactory.newInvalidRequestException("ttl must be a positive value");
        }
    }

    static void validatePredicate(String keyspace, ColumnParent cp, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw ErrorFactory.newInvalidRequestException("predicate column_names and slice_range may not both be null");
        }
        if (predicate.column_names != null && predicate.slice_range != null) {
            throw ErrorFactory.newInvalidRequestException("predicate column_names and slice_range may not both be set");
        }
        if (predicate.slice_range != null) {
            AvroValidation.validateRange(keyspace, cp, predicate.slice_range);
        } else {
            AvroValidation.validateColumns(keyspace, cp, predicate.column_names);
        }
    }

    public static void validateKeyRange(KeyRange range) throws InvalidRequestException {
        Object endToken;
        IPartitioner p;
        Object startToken;
        if (range.start_key == null != (range.end_key == null)) {
            throw ErrorFactory.newInvalidRequestException("start key and end key must either both be non-null, or both be null");
        }
        if (range.start_token == null != (range.end_token == null)) {
            throw ErrorFactory.newInvalidRequestException("start token and end token must either both be non-null, or both be null");
        }
        if (range.start_key == null == (range.start_token == null)) {
            throw ErrorFactory.newInvalidRequestException("exactly one of {start key, end key} or {start token, end token} must be specified");
        }
        if (range.start_key != null && ((Token)(startToken = (p = StorageService.getPartitioner()).getToken(range.start_key))).compareTo(endToken = p.getToken(range.end_key)) > 0 && !((Token)endToken).equals(p.getMinimumToken())) {
            if (p instanceof RandomPartitioner) {
                throw ErrorFactory.newInvalidRequestException("start key's md5 sorts after end key's md5.  this is not allowed; you probably should not specify end key at all, under RandomPartitioner");
            }
            throw ErrorFactory.newInvalidRequestException("start key must sort before (or equal to) finish key in your partitioner!");
        }
        if (range.count <= 0) {
            throw ErrorFactory.newInvalidRequestException("maxRows must be positive");
        }
    }

    static void validateIndexClauses(String keyspace, String columnFamily, IndexClause index_clause) throws InvalidRequestException {
        if (index_clause.expressions.isEmpty()) {
            throw ErrorFactory.newInvalidRequestException("index clause list may not be empty");
        }
        SortedSet<ByteBuffer> indexedColumns = Table.open(keyspace).getColumnFamilyStore(columnFamily).getIndexedColumns();
        for (IndexExpression expression : index_clause.expressions) {
            if (!expression.op.equals((Object)IndexOperator.EQ) || !indexedColumns.contains(expression.column_name)) continue;
            return;
        }
        throw ErrorFactory.newInvalidRequestException("No indexed columns present in index clause with operator EQ");
    }
}

