/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.tools;

import java.io.IOException;
import java.io.PrintStream;
import java.lang.management.MemoryUsage;
import java.net.InetAddress;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import org.apache.cassandra.cache.JMXInstrumentedCacheMBean;
import org.apache.cassandra.concurrent.IExecutorMBean;
import org.apache.cassandra.db.ColumnFamilyStoreMBean;
import org.apache.cassandra.db.CompactionManagerMBean;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.net.MessagingServiceMBean;
import org.apache.cassandra.tools.NodeProbe;
import org.apache.cassandra.utils.EstimatedHistogram;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;

public class NodeCmd {
    private static final String HOST_OPT_LONG = "host";
    private static final String HOST_OPT_SHORT = "h";
    private static final String PORT_OPT_LONG = "port";
    private static final String PORT_OPT_SHORT = "p";
    private static final int defaultPort = 8080;
    private static Options options = null;
    private NodeProbe probe;

    public NodeCmd(NodeProbe probe) {
        this.probe = probe;
    }

    private static void printUsage() {
        HelpFormatter hf = new HelpFormatter();
        String header = "\nAvailable commands:\nring\ninfo\ncfstats\nclearsnapshot\nversion\ntpstats\ndrain\ndecommission\nloadbalance\ncompactionstats\nsnapshot [snapshotname]\nnetstats [host]\nmove <new token>\nremovetoken status|force|<token>\nflush [keyspace] [cfnames]\nrepair [keyspace] [cfnames]\ncleanup [keyspace] [cfnames]\ncompact [keyspace] [cfnames]\ngetcompactionthreshold <keyspace> <cfname>\ncfhistograms <keyspace> <cfname>\nsetcachecapacity <keyspace> <cfname> <keycachecapacity> <rowcachecapacity>\nsetcompactionthreshold <keyspace> <cfname> <minthreshold> <maxthreshold>\n";
        String usage = String.format("java %s --host <arg> <command>%n", NodeCmd.class.getName());
        hf.printHelp(usage, "", options, header);
    }

    public void printRing(PrintStream outs) {
        Map<Token, String> tokenToEndpoint = this.probe.getTokenToEndpointMap();
        ArrayList<Token> sortedTokens = new ArrayList<Token>(tokenToEndpoint.keySet());
        Collections.sort(sortedTokens);
        List<String> liveNodes = this.probe.getLiveNodes();
        List<String> deadNodes = this.probe.getUnreachableNodes();
        List<String> joiningNodes = this.probe.getJoiningNodes();
        List<String> leavingNodes = this.probe.getLeavingNodes();
        Map<String, String> loadMap = this.probe.getLoadMap();
        outs.printf("%-16s%-7s%-8s%-16s%-8s%-44s%n", "Address", "Status", "State", "Load", "Owns", "Token");
        if (sortedTokens.size() > 1) {
            outs.printf("%-16s%-7s%-8s%-16s%-8s%-44s%n", "", "", "", "", "", sortedTokens.get(sortedTokens.size() - 1));
        }
        Map<Token, Float> ownerships = this.probe.getOwnership();
        for (Token token : sortedTokens) {
            String status;
            String primaryEndpoint = tokenToEndpoint.get(token);
            String string = liveNodes.contains(primaryEndpoint) ? "Up" : (status = deadNodes.contains(primaryEndpoint) ? "Down" : "?");
            String state = joiningNodes.contains(primaryEndpoint) ? "Joining" : (leavingNodes.contains(primaryEndpoint) ? "Leaving" : "Normal");
            String load = loadMap.containsKey(primaryEndpoint) ? loadMap.get(primaryEndpoint) : "?";
            String owns = new DecimalFormat("##0.00%").format(ownerships.get(token));
            outs.printf("%-16s%-7s%-8s%-16s%-8s%-44s%n", primaryEndpoint, status, state, load, owns, token);
        }
    }

    public void printThreadPoolStats(PrintStream outs) {
        outs.printf("%-25s%10s%10s%15s%n", "Pool Name", "Active", "Pending", "Completed");
        Iterator<Map.Entry<String, IExecutorMBean>> threads = this.probe.getThreadPoolMBeanProxies();
        while (threads.hasNext()) {
            Map.Entry<String, IExecutorMBean> thread = threads.next();
            String poolName = thread.getKey();
            IExecutorMBean threadPoolProxy = thread.getValue();
            outs.printf("%-25s%10s%10s%15s%n", poolName, threadPoolProxy.getActiveCount(), threadPoolProxy.getPendingTasks(), threadPoolProxy.getCompletedTasks());
        }
    }

    public void printInfo(PrintStream outs) {
        outs.println(this.probe.getToken());
        outs.printf("%-17s: %s%n", "Load", this.probe.getLoadString());
        outs.printf("%-17s: %s%n", "Generation No", this.probe.getCurrentGenerationNumber());
        long secondsUp = this.probe.getUptime() / 1000L;
        outs.printf("%-17s: %d%n", "Uptime (seconds)", secondsUp);
        MemoryUsage heapUsage = this.probe.getHeapMemoryUsage();
        double memUsed = (double)heapUsage.getUsed() / 1048576.0;
        double memMax = (double)heapUsage.getMax() / 1048576.0;
        outs.printf("%-17s: %.2f / %.2f%n", "Heap Memory (MB)", memUsed, memMax);
    }

    public void printReleaseVersion(PrintStream outs) {
        outs.println("ReleaseVersion: " + this.probe.getReleaseVersion());
    }

    public void printNetworkStats(final InetAddress addr, PrintStream outs) {
        List<String> files;
        Set<InetAddress> hosts;
        outs.printf("Mode: %s%n", this.probe.getOperationMode());
        Set<InetAddress> set = hosts = addr == null ? this.probe.getStreamDestinations() : new HashSet<InetAddress>(){
            {
                this.add(addr);
            }
        };
        if (hosts.size() == 0) {
            outs.println("Not sending any streams.");
        }
        for (InetAddress host : hosts) {
            try {
                files = this.probe.getFilesDestinedFor(host);
                if (files.size() > 0) {
                    outs.printf("Streaming to: %s%n", host);
                    for (String file : files) {
                        outs.printf("   %s%n", file);
                    }
                    continue;
                }
                outs.printf(" Nothing streaming to %s%n", host);
            }
            catch (IOException ex) {
                outs.printf("   Error retrieving file data for %s%n", host);
            }
        }
        Set<InetAddress> set2 = hosts = addr == null ? this.probe.getStreamSources() : new HashSet<InetAddress>(){
            {
                this.add(addr);
            }
        };
        if (hosts.size() == 0) {
            outs.println("Not receiving any streams.");
        }
        for (InetAddress host : hosts) {
            try {
                files = this.probe.getIncomingFiles(host);
                if (files.size() > 0) {
                    outs.printf("Streaming from: %s%n", host);
                    for (String file : files) {
                        outs.printf("   %s%n", file);
                    }
                    continue;
                }
                outs.printf(" Nothing streaming from %s%n", host);
            }
            catch (IOException ex) {
                outs.printf("   Error retrieving file data for %s%n", host);
            }
        }
        MessagingServiceMBean ms = this.probe.getMsProxy();
        outs.printf("%-25s", "Pool Name");
        outs.printf("%10s", "Active");
        outs.printf("%10s", "Pending");
        outs.printf("%15s%n", "Completed");
        int pending = 0;
        Iterator<Number> i$ = ms.getCommandPendingTasks().values().iterator();
        while (i$.hasNext()) {
            int n = i$.next();
            pending += n;
        }
        long completed = 0L;
        i$ = ms.getCommandCompletedTasks().values().iterator();
        while (i$.hasNext()) {
            long n = (Long)i$.next();
            completed += n;
        }
        outs.printf("%-25s%10s%10s%15s%n", "Commands", "n/a", pending, completed);
        pending = 0;
        i$ = ms.getResponsePendingTasks().values().iterator();
        while (i$.hasNext()) {
            int n = (Integer)i$.next();
            pending += n;
        }
        completed = 0L;
        i$ = ms.getResponseCompletedTasks().values().iterator();
        while (i$.hasNext()) {
            long n = (Long)i$.next();
            completed += n;
        }
        outs.printf("%-25s%10s%10s%15s%n", "Responses", "n/a", pending, completed);
    }

    public void printCompactionStats(PrintStream outs) {
        CompactionManagerMBean cm = this.probe.getCompactionManagerProxy();
        outs.println("compaction type: " + (cm.getCompactionType() == null ? "n/a" : cm.getCompactionType()));
        outs.println("column family: " + (cm.getColumnFamilyInProgress() == null ? "n/a" : cm.getColumnFamilyInProgress()));
        outs.println("bytes compacted: " + (cm.getBytesCompacted() == null ? "n/a" : cm.getBytesCompacted()));
        outs.println("bytes total in progress: " + (cm.getBytesTotalInProgress() == null ? "n/a" : cm.getBytesTotalInProgress()));
        outs.println("pending tasks: " + cm.getPendingTasks());
    }

    public void printColumnFamilyStats(PrintStream outs) {
        ArrayList<ColumnFamilyStoreMBean> columnFamilies;
        HashMap cfstoreMap = new HashMap();
        Iterator<Map.Entry<String, ColumnFamilyStoreMBean>> cfamilies = this.probe.getColumnFamilyStoreMBeanProxies();
        while (cfamilies.hasNext()) {
            Map.Entry<String, ColumnFamilyStoreMBean> entry = cfamilies.next();
            String tableName = entry.getKey();
            ColumnFamilyStoreMBean cfsProxy = entry.getValue();
            if (!cfstoreMap.containsKey(tableName)) {
                columnFamilies = new ArrayList<ColumnFamilyStoreMBean>();
                columnFamilies.add(cfsProxy);
                cfstoreMap.put(tableName, columnFamilies);
                continue;
            }
            ((List)cfstoreMap.get(tableName)).add(cfsProxy);
        }
        for (Map.Entry entry : cfstoreMap.entrySet()) {
            String tableName = (String)entry.getKey();
            columnFamilies = (ArrayList<ColumnFamilyStoreMBean>)entry.getValue();
            int tableReadCount = 0;
            int tableWriteCount = 0;
            int tablePendingTasks = 0;
            double tableTotalReadTime = 0.0;
            double tableTotalWriteTime = 0.0;
            outs.println("Keyspace: " + tableName);
            for (ColumnFamilyStoreMBean cfstore : columnFamilies) {
                long writeCount = cfstore.getWriteCount();
                long readCount = cfstore.getReadCount();
                if (readCount > 0L) {
                    tableReadCount = (int)((long)tableReadCount + readCount);
                    tableTotalReadTime += (double)cfstore.getTotalReadLatencyMicros();
                }
                if (writeCount > 0L) {
                    tableWriteCount = (int)((long)tableWriteCount + writeCount);
                    tableTotalWriteTime += (double)cfstore.getTotalWriteLatencyMicros();
                }
                tablePendingTasks += cfstore.getPendingTasks();
            }
            double tableReadLatency = tableReadCount > 0 ? tableTotalReadTime / (double)tableReadCount / 1000.0 : Double.NaN;
            double tableWriteLatency = tableWriteCount > 0 ? tableTotalWriteTime / (double)tableWriteCount / 1000.0 : Double.NaN;
            outs.println("\tRead Count: " + tableReadCount);
            outs.println("\tRead Latency: " + String.format("%s", tableReadLatency) + " ms.");
            outs.println("\tWrite Count: " + tableWriteCount);
            outs.println("\tWrite Latency: " + String.format("%s", tableWriteLatency) + " ms.");
            outs.println("\tPending Tasks: " + tablePendingTasks);
            for (ColumnFamilyStoreMBean cfstore : columnFamilies) {
                outs.println("\t\tColumn Family: " + cfstore.getColumnFamilyName());
                outs.println("\t\tSSTable count: " + cfstore.getLiveSSTableCount());
                outs.println("\t\tSpace used (live): " + cfstore.getLiveDiskSpaceUsed());
                outs.println("\t\tSpace used (total): " + cfstore.getTotalDiskSpaceUsed());
                outs.println("\t\tMemtable Columns Count: " + cfstore.getMemtableColumnsCount());
                outs.println("\t\tMemtable Data Size: " + cfstore.getMemtableDataSize());
                outs.println("\t\tMemtable Switch Count: " + cfstore.getMemtableSwitchCount());
                outs.println("\t\tRead Count: " + cfstore.getReadCount());
                outs.println("\t\tRead Latency: " + String.format("%01.3f", cfstore.getRecentReadLatencyMicros() / 1000.0) + " ms.");
                outs.println("\t\tWrite Count: " + cfstore.getWriteCount());
                outs.println("\t\tWrite Latency: " + String.format("%01.3f", cfstore.getRecentWriteLatencyMicros() / 1000.0) + " ms.");
                outs.println("\t\tPending Tasks: " + cfstore.getPendingTasks());
                JMXInstrumentedCacheMBean keyCacheMBean = this.probe.getKeyCacheMBean(tableName, cfstore.getColumnFamilyName());
                if (keyCacheMBean.getCapacity() > 0) {
                    outs.println("\t\tKey cache capacity: " + keyCacheMBean.getCapacity());
                    outs.println("\t\tKey cache size: " + keyCacheMBean.getSize());
                    outs.println("\t\tKey cache hit rate: " + keyCacheMBean.getRecentHitRate());
                } else {
                    outs.println("\t\tKey cache: disabled");
                }
                JMXInstrumentedCacheMBean rowCacheMBean = this.probe.getRowCacheMBean(tableName, cfstore.getColumnFamilyName());
                if (rowCacheMBean.getCapacity() > 0) {
                    outs.println("\t\tRow cache capacity: " + rowCacheMBean.getCapacity());
                    outs.println("\t\tRow cache size: " + rowCacheMBean.getSize());
                    outs.println("\t\tRow cache hit rate: " + rowCacheMBean.getRecentHitRate());
                } else {
                    outs.println("\t\tRow cache: disabled");
                }
                outs.println("\t\tCompacted row minimum size: " + cfstore.getMinRowSize());
                outs.println("\t\tCompacted row maximum size: " + cfstore.getMaxRowSize());
                outs.println("\t\tCompacted row mean size: " + cfstore.getMeanRowSize());
                outs.println("");
            }
            outs.println("----------------");
        }
    }

    public void printRemovalStatus(PrintStream outs) {
        outs.println("RemovalStatus: " + this.probe.getRemovalStatus());
    }

    private void printCfHistograms(String keySpace, String columnFamily, PrintStream output) {
        ColumnFamilyStoreMBean store = this.probe.getCfsProxy(keySpace, columnFamily);
        long[] offsets = new EstimatedHistogram(90).getBucketOffsets();
        long[] rrlh = store.getRecentReadLatencyHistogramMicros();
        long[] rwlh = store.getRecentWriteLatencyHistogramMicros();
        long[] sprh = store.getRecentSSTablesPerReadHistogram();
        long[] ersh = store.getEstimatedRowSizeHistogram();
        long[] ecch = store.getEstimatedColumnCountHistogram();
        output.println(String.format("%s/%s histograms", keySpace, columnFamily));
        output.println(String.format("%-10s%10s%18s%18s%18s%18s", "Offset", "SSTables", "Write Latency", "Read Latency", "Row Size", "Column Count"));
        for (int i = 0; i < offsets.length; ++i) {
            output.println(String.format("%-10d%10s%18s%18s%18s%18s", offsets[i], i < sprh.length ? Long.valueOf(sprh[i]) : "", i < rrlh.length ? Long.valueOf(rrlh[i]) : "", i < rwlh.length ? Long.valueOf(rwlh[i]) : "", i < ersh.length ? Long.valueOf(ersh[i]) : "", i < ecch.length ? Long.valueOf(ecch[i]) : ""));
        }
    }

    public static void main(String[] args) throws IOException, InterruptedException, ParseException {
        PosixParser parser = new PosixParser();
        CommandLine cmd = null;
        try {
            cmd = parser.parse(options, args);
        }
        catch (ParseException parseExcep) {
            NodeCmd.badUse(parseExcep.toString());
        }
        String host = cmd.getOptionValue(HOST_OPT_LONG);
        int port = 8080;
        String portNum = cmd.getOptionValue(PORT_OPT_LONG);
        if (portNum != null) {
            try {
                port = Integer.parseInt(portNum);
            }
            catch (NumberFormatException e) {
                throw new ParseException("Port must be a number");
            }
        }
        NodeProbe probe = null;
        try {
            probe = new NodeProbe(host, port);
        }
        catch (IOException ioe) {
            NodeCmd.err(ioe, "Error connection to remote JMX agent!");
        }
        if (cmd.getArgs().length < 1) {
            NodeCmd.badUse("Missing argument for command.");
        }
        NodeCmd nodeCmd = new NodeCmd(probe);
        String[] arguments = cmd.getArgs();
        String cmdName = arguments[0];
        boolean validCommand = false;
        for (NodeCommand n : NodeCommand.values()) {
            if (!cmdName.toUpperCase().equals(n.name())) continue;
            validCommand = true;
        }
        if (!validCommand) {
            NodeCmd.badUse("Unrecognized command: " + cmdName);
        }
        NodeCommand nc = NodeCommand.valueOf(cmdName.toUpperCase());
        switch (nc) {
            case RING: {
                nodeCmd.printRing(System.out);
                break;
            }
            case INFO: {
                nodeCmd.printInfo(System.out);
                break;
            }
            case CFSTATS: {
                nodeCmd.printColumnFamilyStats(System.out);
                break;
            }
            case DECOMMISSION: {
                probe.decommission();
                break;
            }
            case LOADBALANCE: {
                probe.loadBalance();
                break;
            }
            case CLEARSNAPSHOT: {
                probe.clearSnapshot();
                break;
            }
            case TPSTATS: {
                nodeCmd.printThreadPoolStats(System.out);
                break;
            }
            case VERSION: {
                nodeCmd.printReleaseVersion(System.out);
                break;
            }
            case COMPACTIONSTATS: {
                nodeCmd.printCompactionStats(System.out);
                break;
            }
            case DRAIN: {
                try {
                    probe.drain();
                }
                catch (ExecutionException ee) {
                    NodeCmd.err(ee, "Error occured during flushing");
                }
                break;
            }
            case NETSTATS: {
                if (arguments.length > 1) {
                    nodeCmd.printNetworkStats(InetAddress.getByName(arguments[1]), System.out);
                    break;
                }
                nodeCmd.printNetworkStats(null, System.out);
                break;
            }
            case SNAPSHOT: {
                if (arguments.length > 1) {
                    probe.takeSnapshot(arguments[1]);
                    break;
                }
                probe.takeSnapshot("");
                break;
            }
            case MOVE: {
                if (arguments.length != 2) {
                    NodeCmd.badUse("Missing token argument for move.");
                }
                probe.move(arguments[1]);
                break;
            }
            case REMOVETOKEN: {
                if (arguments.length != 2) {
                    NodeCmd.badUse("Missing an argument for removetoken (either status, force, or a token)");
                    break;
                }
                if (arguments[1].equals("status")) {
                    nodeCmd.printRemovalStatus(System.out);
                    break;
                }
                if (arguments[1].equals("force")) {
                    nodeCmd.printRemovalStatus(System.out);
                    probe.forceRemoveCompletion();
                    break;
                }
                probe.removeToken(arguments[1]);
                break;
            }
            case CLEANUP: 
            case COMPACT: 
            case REPAIR: 
            case FLUSH: {
                NodeCmd.optionalKSandCFs(nc, arguments, probe);
                break;
            }
            case GETCOMPACTIONTHRESHOLD: {
                if (arguments.length != 3) {
                    NodeCmd.badUse("getcompactionthreshold requires ks and cf args.");
                }
                probe.getCompactionThreshold(System.out, arguments[1], arguments[2]);
                break;
            }
            case CFHISTOGRAMS: {
                if (arguments.length != 3) {
                    NodeCmd.badUse("cfhistograms requires ks and cf args");
                }
                nodeCmd.printCfHistograms(arguments[1], arguments[2], System.out);
                break;
            }
            case SETCACHECAPACITY: {
                if (arguments.length != 5) {
                    NodeCmd.badUse("setcachecapacity requires ks, cf, keycachecap, and rowcachecap args.");
                }
                probe.setCacheCapacities(arguments[1], arguments[2], Integer.parseInt(arguments[3]), Integer.parseInt(arguments[4]));
                break;
            }
            case SETCOMPACTIONTHRESHOLD: {
                if (arguments.length != 5) {
                    NodeCmd.badUse("setcompactionthreshold requires ks, cf, min, and max threshold args.");
                }
                int minthreshold = Integer.parseInt(arguments[3]);
                int maxthreshold = Integer.parseInt(arguments[4]);
                if (minthreshold < 0 || maxthreshold < 0) {
                    NodeCmd.badUse("Thresholds must be positive integers");
                }
                if (minthreshold > maxthreshold) {
                    NodeCmd.badUse("Min threshold cannot be greater than max.");
                }
                if (minthreshold < 2 && maxthreshold != 0) {
                    NodeCmd.badUse("Min threshold must be at least 2");
                }
                probe.setCompactionThreshold(arguments[1], arguments[2], minthreshold, maxthreshold);
                break;
            }
            default: {
                throw new RuntimeException("Unreachable code.");
            }
        }
        System.exit(0);
    }

    private static void badUse(String useStr) {
        System.err.println(useStr);
        NodeCmd.printUsage();
        System.exit(1);
    }

    private static void err(Exception e, String errStr) {
        System.err.println(errStr);
        e.printStackTrace();
        System.exit(3);
    }

    private static void optionalKSandCFs(NodeCommand nc, String[] cmdArgs, NodeProbe probe) throws InterruptedException, IOException {
        if (cmdArgs.length == 1) {
            for (String keyspace : probe.getKeyspaces()) {
                switch (nc) {
                    case REPAIR: {
                        probe.forceTableRepair(keyspace, new String[0]);
                        break;
                    }
                    case FLUSH: {
                        try {
                            probe.forceTableFlush(keyspace, new String[0]);
                        }
                        catch (ExecutionException ee) {
                            NodeCmd.err(ee, "Error occured while flushing keyspace " + keyspace);
                        }
                        break;
                    }
                    case COMPACT: {
                        try {
                            probe.forceTableCompaction(keyspace, new String[0]);
                        }
                        catch (ExecutionException ee) {
                            NodeCmd.err(ee, "Error occured while compacting keyspace " + keyspace);
                        }
                        break;
                    }
                    case CLEANUP: {
                        if (keyspace.equals("system")) break;
                        try {
                            probe.forceTableCleanup(keyspace, new String[0]);
                        }
                        catch (ExecutionException ee) {
                            NodeCmd.err(ee, "Error occured while cleaning up keyspace " + keyspace);
                        }
                        break;
                    }
                    default: {
                        throw new RuntimeException("Unreachable code.");
                    }
                }
            }
        } else {
            String keyspace = cmdArgs[1];
            String[] columnFamilies = new String[cmdArgs.length - 2];
            for (int i = 0; i < columnFamilies.length; ++i) {
                columnFamilies[i] = cmdArgs[i + 2];
            }
            switch (nc) {
                case REPAIR: {
                    probe.forceTableRepair(keyspace, columnFamilies);
                    break;
                }
                case FLUSH: {
                    try {
                        probe.forceTableFlush(keyspace, columnFamilies);
                    }
                    catch (ExecutionException ee) {
                        NodeCmd.err(ee, "Error occured during flushing");
                    }
                    break;
                }
                case COMPACT: {
                    try {
                        probe.forceTableCompaction(keyspace, columnFamilies);
                    }
                    catch (ExecutionException ee) {
                        NodeCmd.err(ee, "Error occured during compaction");
                    }
                    break;
                }
                case CLEANUP: {
                    try {
                        probe.forceTableCleanup(keyspace, columnFamilies);
                    }
                    catch (ExecutionException ee) {
                        NodeCmd.err(ee, "Error occured during cleanup");
                    }
                    break;
                }
                default: {
                    throw new RuntimeException("Unreachable code.");
                }
            }
        }
    }

    static {
        options = new Options();
        Option optHost = new Option(HOST_OPT_SHORT, HOST_OPT_LONG, true, "node hostname or ip address");
        optHost.setRequired(true);
        options.addOption(optHost);
        options.addOption(PORT_OPT_SHORT, PORT_OPT_LONG, true, "remote jmx agent port number");
    }

    public static enum NodeCommand {
        RING,
        INFO,
        CFSTATS,
        SNAPSHOT,
        CLEARSNAPSHOT,
        VERSION,
        TPSTATS,
        FLUSH,
        DRAIN,
        DECOMMISSION,
        MOVE,
        LOADBALANCE,
        REMOVETOKEN,
        REPAIR,
        CLEANUP,
        COMPACT,
        SETCACHECAPACITY,
        GETCOMPACTIONTHRESHOLD,
        SETCOMPACTIONTHRESHOLD,
        NETSTATS,
        CFHISTOGRAMS,
        COMPACTIONSTATS;

    }
}

